/**
* @file globalFlags.h
* @brief コマンド解析<-->ホバーバイク制御タスク間及びホバーバイク制御タスク内で使用する共用フラグ定義と宣言
* @author Shirota, Nomura
* @date 2018/10/01 ?
*/


#ifndef __GLOBALFLAGS_H__
#define __GLOBALFLAGS_H__

#include "mbed.h"
#include "typedef.h"

#ifndef M_PI
#define M_PI 3.14159265358979
#endif

#define deg_to_rad(deg) (((deg)/360)*2*M_PI)
#define rad_to_deg(rad) (((rad)/2/M_PI)*360)

/**
* @def G_CMD_BUF_SIZ
* @brief コマンド受け渡しバッファサイズ
* @details コマンド解析タスクで使用するコマンド受け渡しバッファサイズの定義
*/
#define G_CMD_BUF_SIZ 32

/**
* @def MAX_VAL_12BIT
* @brief FPGAに渡す12BitDataで表現できる最大値
*/
#define MAX_VAL_12BIT   4095

/**
* @def BRK_RPM
* @brief 前部サブプロペラのブレーキ用途で使用する回転数
*/
//#define BRK_RPM 9000
#define DEF_BRK_RPM 5000

/**
* @def DEF_BRK_ANG
* @brief ブレーキボタンを押した場合の1秒間の目標角度変化量[deg]
*/
#define DEF_BRK_ANG 20

/**
* @def DEF_MOT_OFFSET
* @brief モーターのTAKE_OFFステート時に使用するデフォルトオフセット値
*/
#define DEF_MOT_OFFSET 1500

/**
* @def UPDATE_RATE
* @brief タスクのアップデートレート 50Hz
*/
#define UPDATE_RATE 50


/**
* @typedef typPrintFlg
* @brief debug printさせるデータのフラグを収めた構造体(内部共用体)
*/
typedef struct {
    union{
        UINT16 flg;    //!< フラグデータの型(大きさ)
        struct{
            bool    e1   : 1;//!< 前エンジン回転数
            bool    e2   : 1;//!< 後ろエンジン回転数
            bool    m1   : 1;//!< サブモーター回転数
            bool    m2   : 1;//!< サブモーター回転数
            bool    m3   : 1;//!< サブモーター回転数
            bool    m4   : 1;//!< サブモーター回転数
            bool    gy   : 1;//!< ジャイロ
            bool    yaw  : 1;//!< ヨー角
            bool    fb   : 1;//!< PID制御feedback結果
            bool    pp   : 1;//!< P制御ゲイン（アウターループ）
            bool    p    : 1;//!< P制御ゲイン
            bool    i    : 1;//!< I制御ゲイン
            bool    d    : 1;//!< D制御ゲイン
            bool    v    : 1;//!< PID2(torgue)角速度feedback制御用ゲイン
            bool    stat : 1;//!< ステート表示
            bool    err  : 1;//!< errorカウント表示
        }bf;
    }d1;
    union{
        UINT16 flg;    //!< フラグデータの型(大きさ)
        struct{
            bool    ain   : 1;//!< アナログ入力値
            bool    sw    : 1;//!< User Ope Switch
            bool    mo1   : 1;//!< Motor1 Offset val
            bool    mo2   : 1;//!< Motor2 Offset val
            bool    mo3   : 1;//!< Motor3 Offset val
            bool    mo4   : 1;//!< Motor4 Offset val
            bool    ep1   : 1;//!< 前エンジンにセットされたパラメータ値
            bool    ep2   : 1;//!< 後エンジンにセットされたパラメータ値
            bool    mrpm   :1;//!< Motor RPM
            bool    rimu_stat   : 1;//!< Readimg IMU ON/OFF state
            bool    rsv_01   : 1;//!< reserve 01
            bool    rsv_02   : 1;//!< reserve 02
            bool    rsv_03   : 1;//!< reserve 03
            bool    rsv_04   : 1;//!< reserve 04
            bool    rsv_05   : 1;//!< reserve 05
            bool    rsv_06   : 1;//!< reserve 06
        }bf;
    }d2;
}typPrintFlg;

/**
* @typedef typCalFlag
* @brief コマンドにより、キャリブレートするデータのフラグを収めた共用体
*/
typedef union{
    UINT16  flg;    //!< フラグデータの型(大きさ)
    struct{    
        bool    e1   : 1;//!< 前エンジン回転数
        bool    e2   : 1;//!< 後ろエンジン回転数
        bool    m1   : 1;//!< サブモーター回転数1
        bool    m2   : 1;//!< サブモーター回転数2
        bool    m3   : 1;//!< サブモーター回転数3
        bool    m4   : 1;//!< サブモーター回転数4
        bool    gy   : 1;//!< ジャイロ
        bool    yaw  : 1;//!< ヨー角
        bool    fb   : 1;//!< PID制御feedback結果
        bool    pp   : 1;//!< P制御ゲイン（アウターループ）
        bool    p    : 1;//!< P制御ゲイン
        bool    i    : 1;//!< I制御ゲイン
        bool    d    : 1;//!< D制御ゲイン
    }bf;
}typCalFlag;


/**
* @typedef typSWCmd
* @brief コマンドにより、スイッチが押されたことを模擬するフラグをまとめた共用体
*/
typedef union{
    UINT16  dt;     //!< フラグデータの型(大きさ)
    struct{    
        UINT16  val   : 15;//!< 立てたいbit番号(現状0~10)
        bool    req   : 1;//!< コマンド要求フラグ
    }bf;
}typSWCmd;

/**
* @typedef typAccel
* @brief アクセル更新要求と設定値をまとめた共用体
*/
typedef union{
    UINT16  dt;     //!< フラグデータの型(大きさ)
    struct{    
        UINT16  val   : 12;//!< アクセル設定値
        bool    req   : 1;//!< アクセル更新要求
    }bf;
}typAccel;

/**
* @typedef typAxlRpm
* @brief モーター回転数更新要求と設定値をまとめた構造体
*/
typedef struct{
    INT16  val;     //!< モーター回転数設定値
    bool   req;     //!< モーター回転数更新要求
}typAxlRpm;


/**
* @enum enmHbState
* @brief ホバーバイク制御タスクのステート定義
*/
enum enmHbState
    {
    NONE = 0        //!< 何もなし
    ,SLEEP          //!< スリープモード（最初ここ）
    ,WAKEUP         //!< 起動処理
    ,STANDBY        //!< スタンバイ(エンジンかかるの待ち)
    ,IDLE           //!< アイドル(エンジンがかかった状態+サブプロペラ使用可能)
    ,UPPER_IDLE     //!< 離陸用既定値の半分までエンジン用スロットルを開く
    ,TAKE_OFF       //!< 離陸(エンジン回転数上昇　規定値まで)
    ,HOVER          //!< ホバリング
    ,DRIVE          //!< 運転
    ,GROUND         //!< 着陸(エンジン同時下げ)
    ,EMGGND         //!< 緊急着陸（後ろエンジンを先に下げる予定）
    ,CHK_EG_ENT     //!< SLEEPからエンジン調整モードへ（フラグリセット、アナログ入力が下がっていること、ボタンが押されいないことの確認State）
    ,CHK_EG_F       //!< フロントエンジン調整
    ,CHK_EG_MID     //!< アナログ入力が下がっていること、ボタンが押されていないことを確認するState
    ,CHK_EG_R       //!< リアエンジン調整
    ,CHK_EG_EXIT    //!< アナログ入力が下がっていること、他のボタンが押されていないことを確認してSLEEPへ
    ,CHK_ENT        //!< チェックエンター
    ,CHK_MOT        //!< モーターチェック
    ,CHK_AXL        //!< アクセルサーボチェック
    ,CHK_ATT        //!< 姿勢制御チェック
    ,CHK_EXIT       //!< チェックステート脱出
    ,MOT_STOP       //!< モーター停止用
};


/**
* @enum eLedCol
* @brief 状態表示用LEDカラーを表すENUM
*/
enum eLedCol{
    OFF = 0,
    WHITE,
    RED,
    GREEN,
    BLUE,
    YELLOW,
    PURPLE,
    LIGHT_BLUE,
    BLK_WHITE,
    BLK_RED,
    BLK_GREEN,
    BLK_BLUE,
    BLK_YELLOW,
    BLK_PURPLE,
    BLK_LIGHT_BLUE,
};

/**
* @enum eMotPos
* @brief サブプロペラ(モーター)の位置を表すENUM
*/
enum eMotPos{
    F_L = 0,    //!< 前　左
    F_R,        //!< 前　右
    R_L,        //!< 後　左
    R_R,        //!< 後　右
};

/**
* @enum eMotType
* @brief 個別サブプロペラ(モーター)のINSIDE,OUTSIDE位置を表すENUM
*/
enum eMotType{
    IN = 0,
    OUT,
};

/**
* @enum eMotInType
* @brief サブプロペラ(モーター)へ設定する回転数のタイプを表すENUM
*/
enum eMotInType{
    OFS = 0,    //!< オフセット回転数(この値で制御関係なく回る)
    ATT,        //!< 姿勢制御のために算出された要求回転数
    USER,       //!< ユーザー操作により要求される回転数
};

/**
* @enum eEgPos
* @brief エンジンの位置を表すENUM
*/
enum eEgPos{
    FRONT = 0,  //!< 前部エンジン
    REAR,       //!< 後部エンジン
};


/////////////////////////////////////////////////

/**
 * 共用フラグを初期化する関数
 */
void initFlags();

/**
 * DigitalOutを利用してLEDの色を指定する関数
 */
void setDOCol(eLedCol col);

/**
 * ホバーバイク制御タスクのステートによって、DigitalOutを利用してLEDの色を変える関数
 */
void setDO4LED(enmHbState stat);

/**
 * ホバーバイク制御タスクのステートを変更する関数
 */
void setState(enmHbState stat);

/**
 * ホバーバイク制御タスクのステートを強制的に変更する関数
 */
void setStateF(enmHbState stat);

/**
 * ホバーバイク制御タスクのステートで飛行若しくは準備状態か否かを返す関数
 */
bool isActiveState();


extern DigitalOut led1;     //!< モニタ用LED1用変数
extern DigitalOut led2;     //!< モニタ用LED2用変数
extern DigitalOut led3;     //!< モニタ用LED3用変数
extern DigitalOut led4;     //!< モニタ用LED4用変数

extern DigitalOut DO_01;    //!< 状態表示用LED1(R)用変数
extern DigitalOut DO_02;    //!< 状態表示用LED2(G)用変数
extern DigitalOut DO_03;    //!< 状態表示用LED3(B)用変数

extern AnalogIn AinAxl;     //!< モーターアクセル用アナログ入力


extern char g_CmdBuf[G_CMD_BUF_SIZ];    //!< コマンド受け渡しバッファ
extern bool gf_CmdPrs;      //!< コマンドパーサー実行要求フラグ

extern bool gf_Armed;       //!< アーミングフラグ
extern bool gf_Dbg;         //!< デバッグフラグ
extern bool gf_StopMot;     //!< モーターの強制停止 サーボ全閉
extern bool gf_FromActiveStat;  //!< モーターの強制停止 サーボ全閉が動作時に発生したかどうか

extern bool gf_BlinkLED;    //!< ステート表示LEDをブリンクさせるフラグ

extern bool gf_EnbImuRead;  //!< IMUの値読み出し可否 TRUE:読み込み FALSE:無視(0埋め)

extern typPrintFlg  gf_Print;       //!< デバッグプリントフラグ（１回表示）
extern typPrintFlg  gf_Mon;         //!< デバッグモニタフラグ（繰り返し表示）
extern typCalFlag   gf_Cal;         //!< キャリブレーションフラグ

extern typAccel     gf_AxReq[2];        //!< エンジンアクセル値更新フラグ
extern typAccel     gf_AxReqH[2];       //!< 浮上時エンジンアクセル値更新フラグ
extern typAccel     gf_AxStepReq[2];    //!< エンジンアクセル用サーボ動作ステップ値更新フラグ
extern typAccel     gf_AxAdjStepReq[2]; //!< エンジンアクセル調整用サーボ動作ステップ値更新フラグ

extern typAxlRpm     gf_MtReq[4];       //!< モーター姿勢制御値更新
extern typAxlRpm     gf_MtAttOfs[4];    //!< モーター姿勢制御オフセット値更新
extern typAxlRpm     gf_MtReqOfs[4];    //!< モーターオフセット値更新
extern typAxlRpm     gf_MtReqU[4];      //!< モーターユーザー値更新
extern typAxlRpm     gf_MtBrk;          //!< モーターブレーキ時のRPM値更新
extern typAxlRpm     gf_MtDefaultOffset;//!< モーターのTAKE_OFFステート時に使用するデフォルトオフセット値更新
extern typAxlRpm     gf_MtReqDct[8];    //!< ダイレクト(FPGA関数直接呼び出し)モーター値更新フラグ

extern typAxlRpm     gf_AngBrk;         //!< ブレーキ時の1秒間の目標角度変化値[deg]更新

extern typSWCmd     gf_SwCmd;           //!< スイッチが押されたことを模擬するフラグ

extern enmHbState      gf_State;        //!< 現在のステートを格納する変数
extern bool            gf_StateEnt;     //!< 状態遷移後、最初であることを示すフラグ

extern bool         gf_PidParaUpdate;   //!< PID Pp,P,I,Dの係数アップデートフラグ
extern typPidPara   g_PidPara;          //!< PID Pp,P,I,Dの係数の外部設定用

#endif