/**
* @file fpga.cpp
* @brief FPGAにSPI通信でやり取りするための関数群
* @author Shirota
* @date 2018/10/01 ?
*/

#include "mbed.h"
#include "fpga.h"
#include "globalFlags.h"

#define CLK_CYC 20.83   //FPGAのクロック周期

//-----------------------------
//FPGAレジスタアドレスマップ
#define ADR_TEST_REG 0x0 //テストレジスタ
#define ADR_RPM_FLSB 0x1 //エンジン回転数 前 LSB
#define ADR_RPM_FMSB 0x2 //エンジン回転数 前 MSB
#define ADR_RPM_BLSB 0x3 //エンジン回転数 後ろ LSB
#define ADR_RPM_BMSB 0x4 //エンジン回転数 後ろ MSB

#define ADR_SUB_FL_I 0x5 //前  ◀ in
#define ADR_SUB_FL_O 0x6 //前  ◀ out
#define ADR_SUB_FR_I 0x7 //前  ▶ in
#define ADR_SUB_FR_O 0x8 //前  ▶ out

#define ADR_SUB_BL_I 0x9 //後ろ ◀ in
#define ADR_SUB_BL_O 0xA //後ろ ◀ out
#define ADR_SUB_BR_I 0xB //後ろ ▶ in
#define ADR_SUB_BR_O 0xC //後ろ ▶ out

#define ADR_USER_SW 0xD //ユーザースイッチ
#define ADR_ACCEL_F 0xE //アクセル前
#define ADR_ACCEL_B 0xF //アクセル後ろ
//-----------------------------

// ################################################################
// SPI通信フォーマットに合わせた供用体定義
//      bit |15|14|13|12|11|10| 9| 8| 7| 6| 5| 4| 3| 2| 1| 0|
// function |  ID       |               Data                |
// ################################################################
typedef union{
    UINT16  spiCmd;
    struct{    
        UINT16    val   : 12;//bit[11: 0] 書き込みデータ
        UINT16    id    : 4 ;//bit[15:12] レジスタアドレス
    }bf;
}SPI_CMD;

// ###############################################################
// グローバル変数
// ###############################################################
static  SPI             spi (p5,p6,p7)  ;//SPI通信ポート定義： mosi, miso, clk
static  DigitalOut      CS_n(p8)        ;//SPIのチップセレクトピン

//-----------------------------------
//SPIデータ送出
//-----------------------------------
static UINT16 xSpi16bitSend(UINT16  dat){
    unsigned int rtn;

    CS_n= 0;                //チップセレクトアサート
    rtn = spi.write(dat);   //データ出力
    CS_n = 1;               //チップセレクトネゲート
    return rtn;             //SPIリードデータを返す
}
//-----------------------------------
//SPI送出データの組み立てて通信
//-----------------------------------
static UINT16 xSpi(UCHAR  id , UINT16 val){
    UINT16 spiRtn=0;
    SPI_CMD cmd;

    //引数を送信するコマンドに成形
    cmd.bf.id  = id;
    cmd.bf.val = val; 
    //SPIデータ送出し読み出し値を取得
    spiRtn = xSpi16bitSend(cmd.spiCmd);
    //レジスタ読み出し値を返す
    return spiRtn;
}

//================================================================
//テストレジスタアクセス
//================================================================
UINT16 fpgaTest(UINT16 val){
    //SPI通信
    UINT16 spiRtn = xSpi(ADR_TEST_REG , val);
    //レジスタ読み出し値を返す
    return spiRtn;
}
//================================================================
//エンジン回転数を取得
//================================================================
UINT16 fpgaGetRpm(bool frontFlg){
    UINT16      RpmMsb=0;
    UINT16      RpmLsb=0;
    UINT32      Rpm=0;
    double      clc;
    
    RpmLsb = xSpi((frontFlg==true) ? ADR_RPM_FLSB : ADR_RPM_BLSB , 0);//下位12bit読み出し
    RpmMsb = xSpi((frontFlg==true) ? ADR_RPM_FMSB : ADR_RPM_BMSB , 0);//上位12bit読み出し
    
    //24bit連結:48MHzのクロックサイクル数換算
    Rpm = (RpmMsb << 12) | RpmLsb;
    
    //RPM換算
    clc = (double)Rpm * CLK_CYC;
    clc = clc / 1000;
    clc = 30000000 / clc;
    //回転数rpmを返す
    return (UINT16)clc;
}
//================================================================
//サブプロペラ設定 　　　位置  0:前左 1:前右 2:後左 3:後右
//================================================================
bool fpgaSubProp(UCHAR iPos,INT16 iVal){
    UCHAR   regAdr[1];
    UINT16  regVal;

    switch (iPos){
        case 0x0://
            regAdr[0] = ADR_SUB_FL_I;
            regAdr[1] = ADR_SUB_FL_O;
            break;
        case 0x1://
            regAdr[0] = ADR_SUB_FR_I;
            regAdr[1] = ADR_SUB_FR_O;
            break;
        case 0x2://
            regAdr[0] = ADR_SUB_BL_I;
            regAdr[1] = ADR_SUB_BL_O;
            break;
        case 0x3://
            regAdr[0] = ADR_SUB_BR_I;
            regAdr[1] = ADR_SUB_BR_O;
            break;
        default:
            return false;//何もしないで終わる
    }
    
    //レジスタ設定範囲内にリミット
    regVal = (iVal<0) ? 0 :(iVal>4095)? 4095 : iVal;
//    if(iVal<0){
//        regVal = 0;
//    }else if(iVal>4095){
//        regVal = 4095;
//    }else{ 
//        regVal = iVal;
//    }

    xSpi(regAdr[0] , regVal);//吸気側
    xSpi(regAdr[1] , regVal);//排気側
    
    //正常終了
    return true;
}

//================================================================
//モーター個別設定
//================================================================
void fpgaMotor(UCHAR iPos,INT16 iVal){
    UCHAR   regAdr = ADR_SUB_FL_I;//モーターの先頭番地を指しておく
    UINT16  regVal;

    if( iPos<8){ regAdr += iPos; } //指定位置だけオフセットさせる
    // if(gf_Print.d1.bf.fb){
    //     sp.printf("POS[%d] Val[%d]", iPos, iVal);
    // }
    regVal = (iVal<0) ? 0 :(iVal>4095)? 4095 : iVal; //レジスタ設定範囲内にリミット
    xSpi(regAdr , regVal);//
}

//================================================================
//ユーザースイッチ読み取り
//================================================================
UINT16 fpgaGetUserSw(){
    //SPIデータ送出/読み出し
    UINT16 spiRtn = xSpi(ADR_USER_SW , 0);
    //戻り値
    return spiRtn;
}

//================================================================
//アクセル設定 　　　位置  0:前 1後
//================================================================
void fpgaEngine(UCHAR iID , UINT16 iSlotl){
    xSpi((iID==0)? ADR_ACCEL_F : ADR_ACCEL_B , iSlotl);    
}

//================================================================
//FPGA初期化
//================================================================
void fpgaInit(){
    spi.format(16,3)        ;//SPIのフォーマット指定（bit長、極性）
    spi.frequency(500000)   ;//クロック周波数
    CS_n =1                 ;//Low有意なのでHighにしておく
}
