/*
Copyright (c) 2010 Peter Barrett

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

/* 
Tue Apr 26 2011 Bart Janssens: added a socket listener
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "Utils.h"
#include "Socket.h"

#define MAX_SOCKET_HANDLERS 3
#define MAX_SOCKETS 16
#define MAX_LISTEN 8

class SocketInternalPad
{
    public:
    SocketInternal si;
    u8  pad[8];
};


    

class SocketManager
{
    SocketHandler*  _handlers[MAX_SOCKET_HANDLERS];
    SocketInternalPad  _sockets[MAX_SOCKETS];
    SocketListener _listeners[MAX_LISTEN];

    public:
    SocketManager()
    {
        memset(_handlers,0,sizeof(_handlers));
        memset(_sockets,0,sizeof(_sockets));
        memset(_listeners,0,sizeof(_listeners));
    }

    SocketHandler* GetHandler(int type)
    {
        if (type < 1 || type > MAX_SOCKET_HANDLERS)
            return 0;
        return _handlers[type - 1];
    }

    SocketInternal* GetInternal(int s)
    {
        if (s < 1 || s > MAX_SOCKETS)
            return 0;
        return &_sockets[s - 1].si;
    }

    int RegisterSocketHandler(int type, SocketHandler* handler)
    {
       if (type < 1 || type > MAX_SOCKET_HANDLERS)
            return ERR_SOCKET_TYPE_NOT_FOUND;
        _handlers[type - 1] = handler;
        return 0;
    }

    int Open(int type, SocketAddrHdr* addr, SocketCallback callback, void* userData)
    {
        SocketHandler* h = GetHandler(type);
        if (!h)
            return ERR_SOCKET_TYPE_NOT_FOUND;
        
        for (int i = 0; i < MAX_SOCKETS; i++)
        {
            SocketInternal* si = (SocketInternal*)(_sockets+i);
            if (si->ID == 0)
            {
                //printf("Call to Socket Manager Open \r\n");
                si->ID = i+1;
                si->Type = type;
                si->Callback = callback;
                si->userData = userData;
                return h->Open(si,addr);
            }
        }
        return ERR_SOCKET_NONE_LEFT;
    }
    
    SocketInternal* Create(int type, SocketAddrHdr* addr, int port)
    {
        SocketInternal* si;
        SocketListener* li;
        SocketHandler* h = GetHandler(type);
        if (!h)
            return 0;
        
        for (int i = 0; i < MAX_SOCKETS; i++)
        {
            si = (SocketInternal*)(_sockets+i);
            if (si->ID == 0)
            {
                si->ID = i+1;
                si->State = SocketState_Listen;
                si->Type = type;
                si->port = port;
                for (int i = 0; i < MAX_LISTEN; i++){
                    li = (SocketListener*)(_listeners+i);
                    if (( li->Type == si->Type )&& (li->port == si->port)) {
                        si->Callback = li->Callback;
                        si->userData = li->userData;
                        h->Create(si,addr);
                        return si;
                    }

                }
             }
        }

    }
    
    
    int Listen(int type, int port, SocketCallback callback,void* userData)
    {
        SocketListener* li;
        SocketHandler* h = GetHandler(type);
        if (!h) return ERR_SOCKET_TYPE_NOT_FOUND;
        
        //printf("Call to Socket Manager Listen \r\n");
        for (int i = 0; i < MAX_LISTEN; i++)
        {
            li = (SocketListener*)(_listeners+i);
            if (( li->Type == type )&& (li->port == port)) {
               //printf("Port %d is already in use\r\n",port);
               return ERR_SOCKET_IN_USE; //in use
            }
        }
        
        for (int i = 0; i < MAX_LISTEN; i++)
        {
            li = (SocketListener*)(_listeners+i);
            if (( li->Type == 0 )&& (li->port == 0)) {
                li->ID = i+1;
                li->Type = type;
                li->port = port;
                li->Callback = callback;
                li->userData = userData;
                //printf("Listening on port %d \r\n",port);
                return 0;
            }
        }
        //printf("Max listen ports reached\r\n",port);
        return ERR_SOCKET_NONE_LEFT;           
    }
    
    int InUse(int type, int port)
    {
        SocketListener* li;
        SocketHandler* h = GetHandler(type);
        if (!h) return ERR_SOCKET_TYPE_NOT_FOUND;
        for (int i = 0; i < MAX_LISTEN; i++)
        {
            li = (SocketListener*)(_listeners+i);
            if (( li->Type == type )&& (li->port == port)) {

                //printf("Listen check on port %d OK\r\n",port);
                return 0;
            }
        }
        //printf("Listen check on port %d NOK\r\n",port);
        return ERR_SOCKET_NONE_LEFT; 
     }
    

    int Accept(int socket, SocketCallback callback, void* userData)
    {
        SocketInternal* si = GetInternal(socket);
        if (!si || si->ID != socket)
            return ERR_SOCKET_NOT_FOUND;
            
        si->Callback = callback;
        si->userData = userData;
        
        //printf("Call to Socket Manager Accept \r\n");        
        return 0;
         
    }

    int Send(int socket, const u8* data, int len)
    {
        //printf("Call to Socket Manager Send \r\n"); 
        SocketInternal* si = GetInternal(socket);
        //printf("socket = %d si->ID = %d si->Type = %d \r\n", socket, si->ID, si->Type); 
        if (!si || si->ID != socket){
            //printf("send: socket not found \r\n");
            return ERR_SOCKET_NOT_FOUND;
        }
        //printf("Calling l2cap send \r\n");
        
        SocketHandler* h = GetHandler(si->Type);
        if (!h) {
                //printf("Send: no socket type found \r\n");
                return ERR_SOCKET_TYPE_NOT_FOUND;
        }
        return h->Send(si,data,len);
       
    }

    int Close(int socket)
    {
        SocketInternal* si = GetInternal(socket);
        if (!si || si->ID != socket)
            return ERR_SOCKET_NOT_FOUND;
        si->ID = 0;
        return GetHandler(si->Type)->Close(si);
    }
};

SocketManager gSocketManager;

int Socket_Open(int type, SocketAddrHdr* addr, SocketCallback callback, void* userData)
{
    //printf("Call to Socket Open \r\n");
    return gSocketManager.Open(type,addr,callback,userData);
}

SocketInternal* Socket_Create(int type, SocketAddrHdr* addr, int port)
{
    return gSocketManager.Create(type, addr, port);
}

int Socket_Send(int socket, const u8* data, int len)
{
    //printf("Call to Socket_Send \r\n");
    return gSocketManager.Send(socket,data,len);
}

int Socket_Close(int socket)
{
    return gSocketManager.Close(socket);
}

int Socket_Listen(int type, int port,SocketCallback callback, void* userData)
{
    //printf("Call to Socket_Listen \r\n");
    return gSocketManager.Listen(type, port,callback,userData);
}

int Socket_InUse(int type, int port)
{
    //printf("Call to Socket_InUse \r\n");
    return gSocketManager.InUse(type, port);
}

int Socket_Accept(int socket, SocketCallback callback, void* userData)
{
    //printf("Call to Socket_Accept \r\n");
    return gSocketManager.Accept(socket, callback, userData);
}

int RegisterSocketHandler(int type, SocketHandler* handler)
{
    return gSocketManager.RegisterSocketHandler(type,handler);
}

SocketInternal* GetSocketInternal(int socket)
{
    return gSocketManager.GetInternal(socket);
}

