/* SAA1064 4 Digit Library
 * Copyright (c) 2015 Marcel (mc-b) Bernet
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "SAA1064.h"

SAA1064::SAA1064( PinName sda, PinName scl, uint8_t deviceAddress )
{
    i2c = new I2C( sda, scl );
    slaveAddress = deviceAddress;
    init();
}


SAA1064::SAA1064( I2C *i2c, uint8_t deviceAddress )
{
    this->i2c = i2c;
    slaveAddress = deviceAddress;
    init();
}

/** Write digits
*
* @param digit1  LED segment pattern for digit1 (MSB)
* @param digit2  LED segment pattern for digit2
* @param digit3  LED segment pattern for digit3
* @param digit4  LED segment pattern for digit4 (LSB)
*/
void SAA1064::write( uint8_t digit1, uint8_t digit2, uint8_t digit3, uint8_t digit4 )
{
    digit1 = SAA1064_SEGM[digit1];
    digit2 = SAA1064_SEGM[digit2];
    digit3 = SAA1064_SEGM[digit3];
    digit4 = SAA1064_SEGM[digit4];

    data[0] = 1;
    data[1] = ((digit4<<4) & 0xF0) | (digit2 & 0x0F);
    data[2] = ((digit3<<4) & 0xF0) | (digit1 & 0x0F);
    data[3] = ((digit2>>4) & 0x0F) | (digit4 & 0xF0);
    data[4] = ((digit1>>4) & 0x0F) | (digit3 & 0xF0);
    i2c->write( slaveAddress, (char*) data, 5 );
}

/** Write Integer
*
* @param value     integer value to display, valid range -999...9999
*/
void SAA1064::writeInt( int value )
{
    write( value / 1000, (value % 1000) / 100, (value % 100) / 10, value % 10 );
}

/** Init I2C bus 
*/
void SAA1064::init( void )
{
    // init
    data[0] = 0x00;
    data[1] = 0x47;
    i2c->write( slaveAddress, (char*) data, 2 );

    // blank display
    data[0] = 1;
    data[1] = 0;
    data[2] = 0;
    data[3] = 0;
    data[4] = 0;
    i2c->write( slaveAddress, (char*) data, 5 );
}