#ifndef LMDriver_H
#define LMDriver_H
#include <deque>
#include <string>

#define reg_noop         0x00
#define reg_column1      0x01
#define reg_column2      0x02
#define reg_column3      0x03
#define reg_column4      0x04
#define reg_column5      0x05
#define reg_column6      0x06
#define reg_column7      0x07
#define reg_column8      0x08
#define reg_decodeMode   0x09
#define reg_intensity    0x0a
#define reg_scanLimit    0x0b
#define reg_shutdown     0x0c
#define reg_displayTest  0x0f

#define MHz 1000000;

const uint8_t chars[]= {
    0xFF,0xC3,0xC3,0xC3,0xC3,0xC3,0xC3,0xFF
    ,0xFF,0xC0,0xC0,0xC0,0xC0,0xC0,0xC0,0xC0
    ,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0xFF
    ,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0xFF
    ,0x0C,0x18,0x30,0x7E,0x0C,0x18,0x30,0x00
    ,0xFF,0xC3,0xE7,0xDB,0xDB,0xE7,0xC3,0xFF
    ,0x00,0x01,0x03,0x06,0xCC,0x78,0x30,0x00
    ,0x3C,0x66,0xC3,0xC3,0xFF,0x24,0xE7,0x00
    ,0x00,0x00,0x30,0x60,0xFF,0x60,0x30,0x00
    ,0x00,0x00,0x0C,0x06,0xFF,0x06,0x0C,0x00
    ,0x18,0x18,0x18,0x18,0xDB,0x7E,0x3C,0x18
    ,0x18,0x3C,0x7E,0xDB,0x18,0x18,0x18,0x18
    ,0x18,0x5A,0x3C,0x99,0xDB,0x7E,0x3C,0x18
    ,0x00,0x03,0x33,0x63,0xFE,0x60,0x30,0x00
    ,0x3C,0x66,0xFF,0xDB,0xDB,0xFF,0x66,0x3C
    ,0x3C,0x66,0xC3,0xDB,0xDB,0xC3,0x66,0x3C
    ,0xFF,0xC3,0xC3,0xFF,0xC3,0xC3,0xC3,0xFF
    ,0x3C,0x7E,0xDB,0xDB,0xDF,0xC3,0x66,0x3C
    ,0x3C,0x66,0xC3,0xDF,0xDB,0xDB,0x7E,0x3C
    ,0x3C,0x66,0xC3,0xFB,0xDB,0xDB,0x7E,0x3C
    ,0x3C,0x7E,0xDB,0xDB,0xFB,0xC3,0x66,0x3C
    ,0x00,0x01,0x33,0x1E,0xCE,0x7B,0x31,0x00
    ,0x7E,0x66,0x66,0x66,0x66,0x66,0x66,0xE7
    ,0x03,0x03,0x03,0xFF,0x03,0x03,0x03,0x00
    ,0xFF,0x66,0x3C,0x18,0x18,0x3C,0x66,0xFF
    ,0x18,0x18,0x3C,0x3C,0x3C,0x3C,0x18,0x18
    ,0x3C,0x66,0x66,0x30,0x18,0x00,0x18,0x00
    ,0x3C,0x66,0xC3,0xFF,0xC3,0xC3,0x66,0x3C
    ,0xFF,0xDB,0xDB,0xDB,0xFB,0xC3,0xC3,0xFF
    ,0xFF,0xC3,0xC3,0xFB,0xDB,0xDB,0xDB,0xFF
    ,0xFF,0xC3,0xC3,0xDF,0xDB,0xDB,0xDB,0xFF
    ,0xFF,0xDB,0xDB,0xDB,0xDF,0xC3,0xC3,0xFF
    ,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
    ,0x18,0x18,0x18,0x18,0x18,0x00,0x18,0x00
    ,0x6C,0x6C,0x6C,0x00,0x00,0x00,0x00,0x00
    ,0x6C,0x6C,0xFE,0x6C,0xFE,0x6C,0x6C,0x00
    ,0x18,0x3E,0x58,0x3C,0x1A,0x7C,0x18,0x00
    ,0x00,0xC6,0xCC,0x18,0x30,0x66,0xC6,0x00
    ,0x38,0x6C,0x38,0x76,0xDC,0xCC,0x76,0x00
    ,0x18,0x18,0x30,0x00,0x00,0x00,0x00,0x00
    ,0x0C,0x18,0x30,0x30,0x30,0x18,0x0C,0x00
    ,0x30,0x18,0x0C,0x0C,0x0C,0x18,0x30,0x00
    ,0x00,0x66,0x3C,0xFF,0x3C,0x66,0x00,0x00
    ,0x00,0x18,0x18,0x7E,0x18,0x18,0x00,0x00
    ,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x30
    ,0x00,0x00,0x00,0x7E,0x00,0x00,0x00,0x00
    ,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x00
    ,0x06,0x0C,0x18,0x30,0x60,0xC0,0x80,0x00
    ,0x7C,0xC6,0xCE,0xD6,0xE6,0xC6,0x7C,0x00
    ,0x18,0x38,0x18,0x18,0x18,0x18,0x7E,0x00
    ,0x3C,0x66,0x06,0x3C,0x60,0x66,0x7E,0x00
    ,0x3C,0x66,0x06,0x1C,0x06,0x66,0x3C,0x00
    ,0x1C,0x3C,0x6C,0xCC,0xFE,0x0C,0x1E,0x00
    ,0x7E,0x62,0x60,0x7C,0x06,0x66,0x3C,0x00
    ,0x3C,0x66,0x60,0x7C,0x66,0x66,0x3C,0x00
    ,0x7E,0x66,0x06,0x0C,0x18,0x18,0x18,0x00
    ,0x3C,0x66,0x66,0x3C,0x66,0x66,0x3C,0x00
    ,0x3C,0x66,0x66,0x3E,0x06,0x66,0x3C,0x00
    ,0x00,0x00,0x18,0x18,0x00,0x18,0x18,0x00
    ,0x00,0x00,0x18,0x18,0x00,0x18,0x18,0x30
    ,0x0C,0x18,0x30,0x60,0x30,0x18,0x0C,0x00
    ,0x00,0x00,0x7E,0x00,0x00,0x7E,0x00,0x00
    ,0x60,0x30,0x18,0x0C,0x18,0x30,0x60,0x00
    ,0x3C,0x66,0x66,0x0C,0x18,0x00,0x18,0x00
    ,0x7C,0xC6,0xDE,0xDE,0xDE,0xC0,0x7C,0x00
    ,0x18,0x3C,0x66,0x66,0x7E,0x66,0x66,0x00
    ,0xFC,0x66,0x66,0x7C,0x66,0x66,0xFC,0x00
    ,0x3C,0x66,0xC0,0xC0,0xC0,0x66,0x3C,0x00
    ,0xF8,0x6C,0x66,0x66,0x66,0x6C,0xF8,0x00
    ,0xFE,0x62,0x68,0x78,0x68,0x62,0xFE,0x00
    ,0xFE,0x62,0x68,0x78,0x68,0x60,0xF0,0x00
    ,0x3C,0x66,0xC0,0xC0,0xCE,0x66,0x3E,0x00
    ,0x66,0x66,0x66,0x7E,0x66,0x66,0x66,0x00
    ,0x7E,0x18,0x18,0x18,0x18,0x18,0x7E,0x00
    ,0x1E,0x0C,0x0C,0x0C,0xCC,0xCC,0x78,0x00
    ,0xE6,0x66,0x6C,0x78,0x6C,0x66,0xE6,0x00
    ,0xF0,0x60,0x60,0x60,0x62,0x66,0xFE,0x00
    ,0xC6,0xEE,0xFE,0xFE,0xD6,0xC6,0xC6,0x00
    ,0xC6,0xE6,0xF6,0xDE,0xCE,0xC6,0xC6,0x00
    ,0x38,0x6C,0xC6,0xC6,0xC6,0x6C,0x38,0x00
    ,0xFC,0x66,0x66,0x7C,0x60,0x60,0xF0,0x00
    ,0x38,0x6C,0xC6,0xC6,0xDA,0xCC,0x76,0x00
    ,0xFC,0x66,0x66,0x7C,0x6C,0x66,0xE6,0x00
    ,0x3C,0x66,0x60,0x3C,0x06,0x66,0x3C,0x00
    ,0x7E,0x5A,0x18,0x18,0x18,0x18,0x3C,0x00
    ,0x66,0x66,0x66,0x66,0x66,0x66,0x3C,0x00
    ,0x66,0x66,0x66,0x66,0x66,0x3C,0x18,0x00
    ,0xC6,0xC6,0xC6,0xD6,0xFE,0xEE,0xC6,0x00
    ,0xC6,0x6C,0x38,0x38,0x6C,0xC6,0xC6,0x00
    ,0x66,0x66,0x66,0x3C,0x18,0x18,0x3C,0x00
    ,0xFE,0xC6,0x8C,0x18,0x32,0x66,0xFE,0x00
    ,0x3C,0x30,0x30,0x30,0x30,0x30,0x3C,0x00
    ,0xC0,0x60,0x30,0x18,0x0C,0x06,0x02,0x00
    ,0x3C,0x0C,0x0C,0x0C,0x0C,0x0C,0x3C,0x00
    ,0x18,0x3C,0x7E,0x18,0x18,0x18,0x18,0x00
    ,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF
    ,0x30,0x18,0x0C,0x00,0x00,0x00,0x00,0x00
    ,0x00,0x00,0x78,0x0C,0x7C,0xCC,0x76,0x00
    ,0xE0,0x60,0x7C,0x66,0x66,0x66,0xDC,0x00
    ,0x00,0x00,0x3C,0x66,0x60,0x66,0x3C,0x00
    ,0x1C,0x0C,0x7C,0xCC,0xCC,0xCC,0x76,0x00
    ,0x00,0x00,0x3C,0x66,0x7E,0x60,0x3C,0x00
    ,0x1C,0x36,0x30,0x78,0x30,0x30,0x78,0x00
    ,0x00,0x00,0x3E,0x66,0x66,0x3E,0x06,0x7C
    ,0xE0,0x60,0x6C,0x76,0x66,0x66,0xE6,0x00
    ,0x18,0x00,0x38,0x18,0x18,0x18,0x3C,0x00
    ,0x06,0x00,0x0E,0x06,0x06,0x66,0x66,0x3C
    ,0xE0,0x60,0x66,0x6C,0x78,0x6C,0xE6,0x00
    ,0x38,0x18,0x18,0x18,0x18,0x18,0x3C,0x00
    ,0x00,0x00,0x6C,0xFE,0xD6,0xD6,0xC6,0x00
    ,0x00,0x00,0xDC,0x66,0x66,0x66,0x66,0x00
    ,0x00,0x00,0x3C,0x66,0x66,0x66,0x3C,0x00
    ,0x00,0x00,0xDC,0x66,0x66,0x7C,0x60,0xF0
    ,0x00,0x00,0x76,0xCC,0xCC,0x7C,0x0C,0x1E
    ,0x00,0x00,0xDC,0x76,0x60,0x60,0xF0,0x00
    ,0x00,0x00,0x3C,0x60,0x3C,0x06,0x7C,0x00
    ,0x30,0x30,0x7C,0x30,0x30,0x36,0x1C,0x00
    ,0x00,0x00,0x66,0x66,0x66,0x66,0x3E,0x00
    ,0x00,0x00,0x66,0x66,0x66,0x3C,0x18,0x00
    ,0x00,0x00,0xC6,0xD6,0xD6,0xFE,0x6C,0x00
    ,0x00,0x00,0xC6,0x6C,0x38,0x6C,0xC6,0x00
    ,0x00,0x00,0x66,0x66,0x66,0x3E,0x06,0x7C
    ,0x00,0x00,0x7E,0x4C,0x18,0x32,0x7E,0x00
    ,0x0E,0x18,0x18,0x70,0x18,0x18,0x0E,0x00
    ,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x00
    ,0x70,0x18,0x18,0x0E,0x18,0x18,0x70,0x00
    ,0x76,0xDC,0x00,0x00,0x00,0x00,0x00,0x00
    ,0xCC,0x33,0xCC,0x33,0xCC,0x33,0xCC,0x33
    ,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
    ,0xF0,0xF0,0xF0,0xF0,0x00,0x00,0x00,0x00
    ,0x0F,0x0F,0x0F,0x0F,0x00,0x00,0x00,0x00
    ,0xFF,0xFF,0xFF,0xFF,0x00,0x00,0x00,0x00
    ,0x00,0x00,0x00,0x00,0xF0,0xF0,0xF0,0xF0
    ,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0
    ,0x0F,0x0F,0x0F,0x0F,0xF0,0xF0,0xF0,0xF0
    ,0xFF,0xFF,0xFF,0xFF,0xF0,0xF0,0xF0,0xF0
    ,0x00,0x00,0x00,0x00,0x0F,0x0F,0x0F,0x0F
    ,0xF0,0xF0,0xF0,0xF0,0x0F,0x0F,0x0F,0x0F
    ,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F
    ,0xFF,0xFF,0xFF,0xFF,0x0F,0x0F,0x0F,0x0F
    ,0x00,0x00,0x00,0x00,0xFF,0xFF,0xFF,0xFF
    ,0xF0,0xF0,0xF0,0xF0,0xFF,0xFF,0xFF,0xFF
    ,0x0F,0x0F,0x0F,0x0F,0xFF,0xFF,0xFF,0xFF
    ,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF
    ,0x00,0x00,0x00,0x18,0x18,0x00,0x00,0x00
    ,0x18,0x18,0x18,0x18,0x18,0x00,0x00,0x00
    ,0x00,0x00,0x00,0x1F,0x1F,0x00,0x00,0x00
    ,0x18,0x18,0x18,0x1F,0x0F,0x00,0x00,0x00
    ,0x00,0x00,0x00,0x18,0x18,0x18,0x18,0x18
    ,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18
    ,0x00,0x00,0x00,0x0F,0x1F,0x18,0x18,0x18
    ,0x18,0x18,0x18,0x1F,0x1F,0x18,0x18,0x18
    ,0x00,0x00,0x00,0xF8,0xF8,0x00,0x00,0x00
    ,0x18,0x18,0x18,0xF8,0xF0,0x00,0x00,0x00
    ,0x00,0x00,0x00,0xFF,0xFF,0x00,0x00,0x00
    ,0x18,0x18,0x18,0xFF,0xFF,0x00,0x00,0x00
    ,0x00,0x00,0x00,0xF0,0xF8,0x18,0x18,0x18
    ,0x18,0x18,0x18,0xF8,0xF8,0x18,0x18,0x18
    ,0x00,0x00,0x00,0xFF,0xFF,0x18,0x18,0x18
    ,0x18,0x18,0x18,0xFF,0xFF,0x18,0x18,0x18
    ,0x10,0x38,0x6C,0xC6,0x00,0x00,0x00,0x00
    ,0x0C,0x18,0x30,0x00,0x00,0x00,0x00,0x00
    ,0x66,0x66,0x00,0x00,0x00,0x00,0x00,0x00
    ,0x3C,0x66,0x60,0xF8,0x60,0x66,0xFE,0x00
    ,0x38,0x44,0xBA,0xA2,0xBA,0x44,0x38,0x00
    ,0x7E,0xF4,0xF4,0x74,0x34,0x34,0x34,0x00
    ,0x1E,0x30,0x38,0x6C,0x38,0x18,0xF0,0x00
    ,0x18,0x18,0x0C,0x00,0x00,0x00,0x00,0x00
    ,0x40,0xC0,0x44,0x4C,0x54,0x1E,0x04,0x00
    ,0x40,0xC0,0x4C,0x52,0x44,0x08,0x1E,0x00
    ,0xE0,0x10,0x62,0x16,0xEA,0x0F,0x02,0x00
    ,0x00,0x18,0x18,0x7E,0x18,0x18,0x7E,0x00
    ,0x18,0x18,0x00,0x7E,0x00,0x18,0x18,0x00
    ,0x00,0x00,0x00,0x7E,0x06,0x06,0x00,0x00
    ,0x18,0x00,0x18,0x30,0x66,0x66,0x3C,0x00
    ,0x18,0x00,0x18,0x18,0x18,0x18,0x18,0x00
    ,0x00,0x00,0x73,0xDE,0xCC,0xDE,0x73,0x00
    ,0x7C,0xC6,0xC6,0xFC,0xC6,0xC6,0xF8,0xC0
    ,0x00,0x66,0x66,0x3C,0x66,0x66,0x3C,0x00
    ,0x3C,0x60,0x60,0x3C,0x66,0x66,0x3C,0x00
    ,0x00,0x00,0x1E,0x30,0x7C,0x30,0x1E,0x00
    ,0x38,0x6C,0xC6,0xFE,0xC6,0x6C,0x38,0x00
    ,0x00,0xC0,0x60,0x30,0x38,0x6C,0xC6,0x00
    ,0x00,0x00,0x66,0x66,0x66,0x7C,0x60,0x60
    ,0x00,0x00,0x00,0xFE,0x6C,0x6C,0x6C,0x00
    ,0x00,0x00,0x00,0x7E,0xD8,0xD8,0x70,0x00
    ,0x03,0x06,0x0C,0x3C,0x66,0x3C,0x60,0xC0
    ,0x03,0x06,0x0C,0x66,0x66,0x3C,0x60,0xC0
    ,0x00,0xE6,0x3C,0x18,0x38,0x6C,0xC7,0x00
    ,0x00,0x00,0x66,0xC3,0xDB,0xDB,0x7E,0x00
    ,0xFE,0xC6,0x60,0x30,0x60,0xC6,0xFE,0x00
    ,0x00,0x7C,0xC6,0xC6,0xC6,0x6C,0xEE,0x00
    ,0x18,0x30,0x60,0xC0,0x80,0x00,0x00,0x00
    ,0x18,0x0C,0x06,0x03,0x01,0x00,0x00,0x00
    ,0x00,0x00,0x00,0x01,0x03,0x06,0x0C,0x18
    ,0x00,0x00,0x00,0x80,0xC0,0x60,0x30,0x18
    ,0x18,0x3C,0x66,0xC3,0x81,0x00,0x00,0x00
    ,0x18,0x0C,0x06,0x03,0x03,0x06,0x0C,0x18
    ,0x00,0x00,0x00,0x81,0xC3,0x66,0x3C,0x18
    ,0x18,0x30,0x60,0xC0,0xC0,0x60,0x30,0x18
    ,0x18,0x30,0x60,0xC1,0x83,0x06,0x0C,0x18
    ,0x18,0x0C,0x06,0x83,0xC1,0x60,0x30,0x18
    ,0x18,0x3C,0x66,0xC3,0xC3,0x66,0x3C,0x18
    ,0xC3,0xE7,0x7E,0x3C,0x3C,0x7E,0xE7,0xC3
    ,0x03,0x07,0x0E,0x1C,0x38,0x70,0xE0,0xC0
    ,0xC0,0xE0,0x70,0x38,0x1C,0x0E,0x07,0x03
    ,0xCC,0xCC,0x33,0x33,0xCC,0xCC,0x33,0x33
    ,0xAA,0x55,0xAA,0x55,0xAA,0x55,0xAA,0x55
    ,0xFF,0xFF,0x00,0x00,0x00,0x00,0x00,0x00
    ,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03
    ,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xFF
    ,0xC0,0xC0,0xC0,0xC0,0xC0,0xC0,0xC0,0xC0
    ,0xFF,0xFE,0xFC,0xF8,0xF0,0xE0,0xC0,0x80
    ,0xFF,0x7F,0x3F,0x1F,0x0F,0x07,0x03,0x01
    ,0x01,0x03,0x07,0x0F,0x1F,0x3F,0x7F,0xFF
    ,0x80,0xC0,0xE0,0xF0,0xF8,0xFC,0xFE,0xFF
    ,0xAA,0x55,0xAA,0x55,0x00,0x00,0x00,0x00
    ,0x0A,0x05,0x0A,0x05,0x0A,0x05,0x0A,0x05
    ,0x00,0x00,0x00,0x00,0xAA,0x55,0xAA,0x55
    ,0xA0,0x50,0xA0,0x50,0xA0,0x50,0xA0,0x50
    ,0xAA,0x54,0xA8,0x50,0xA0,0x40,0x80,0x00
    ,0xAA,0x55,0x2A,0x15,0x0A,0x05,0x02,0x01
    ,0x01,0x02,0x05,0x0A,0x15,0x2A,0x55,0xAA
    ,0x00,0x80,0x40,0xA0,0x50,0xA8,0x54,0xAA
    ,0x7E,0xFF,0x99,0xFF,0xBD,0xC3,0xFF,0x7E
    ,0x7E,0xFF,0x99,0xFF,0xC3,0xBD,0xFF,0x7E
    ,0x38,0x38,0xFE,0xFE,0xFE,0x10,0x38,0x00
    ,0x10,0x38,0x7C,0xFE,0x7C,0x38,0x10,0x00
    ,0x6C,0xFE,0xFE,0xFE,0x7C,0x38,0x10,0x00
    ,0x10,0x38,0x7C,0xFE,0xFE,0x10,0x38,0x00
    ,0x00,0x3C,0x66,0xC3,0xC3,0x66,0x3C,0x00
    ,0x00,0x3C,0x7E,0xFF,0xFF,0x7E,0x3C,0x00
    ,0x00,0x7E,0x66,0x66,0x66,0x66,0x7E,0x00
    ,0x00,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x00
    ,0x0F,0x07,0x0D,0x78,0xCC,0xCC,0xCC,0x78
    ,0x3C,0x66,0x66,0x66,0x3C,0x18,0x7E,0x18
    ,0x0C,0x0C,0x0C,0x0C,0x0C,0x3C,0x7C,0x38
    ,0x18,0x1C,0x1E,0x1B,0x18,0x78,0xF8,0x70
    ,0x99,0x5A,0x24,0xC3,0xC3,0x24,0x5A,0x99
    ,0x10,0x38,0x38,0x38,0x38,0x38,0x7C,0xD6
    ,0x18,0x3C,0x7E,0xFF,0x18,0x18,0x18,0x18
    ,0x18,0x18,0x18,0x18,0xFF,0x7E,0x3C,0x18
    ,0x10,0x30,0x70,0xFF,0xFF,0x70,0x30,0x10
    ,0x08,0x0C,0x0E,0xFF,0xFF,0x0E,0x0C,0x08
    ,0x00,0x00,0x18,0x3C,0x7E,0xFF,0xFF,0x00
    ,0x00,0x00,0xFF,0xFF,0x7E,0x3C,0x18,0x00
    ,0x80,0xE0,0xF8,0xFE,0xF8,0xE0,0x80,0x00
    ,0x02,0x0E,0x3E,0xFE,0x3E,0x0E,0x02,0x00
    ,0x38,0x38,0x92,0x7C,0x10,0x28,0x28,0x28
    ,0x38,0x38,0x10,0xFE,0x10,0x28,0x44,0x82
    ,0x38,0x38,0x12,0x7C,0x90,0x28,0x24,0x22
    ,0x38,0x38,0x90,0x7C,0x12,0x28,0x48,0x88
    ,0x00,0x3C,0x18,0x3C,0x3C,0x3C,0x18,0x00
    ,0x3C,0xFF,0xFF,0x18,0x0C,0x18,0x30,0x18
    ,0x18,0x3C,0x7E,0x18,0x18,0x7E,0x3C,0x18
    ,0x00,0x24,0x66,0xFF,0x66,0x24,0x00,0x00
};


class LMDriver
{

    SPI* spi;
    DigitalOut* cs;
    PinName _msoi,_miso,_sclk,_cs;
    float wtime;
    float cwtime;
    int freq;
    void writeToReg(int reg, int value) {
        cs->write(0);
        spi->write(reg);
        spi->write(value);
        cs->write(1);
    }
    void zeroFill() {
        for(int i=1; i<=8; i++)
            writeToReg(i,0);
    }
    void writeArray(const uint8_t* array, uint8_t size=8) {
        int count=0;
        for(int i=1; i<=8; i++)
            writeToReg(i,array[count++]);
    }

    void displayDeque(const deque<uint8_t>& output) {
        int count=0;
        for(int i=1; i<=8; i++)
            writeToReg(i,output[count++]);
    }
    void allocate(const LMDriver& lm) {
        this->spi=new SPI(lm._msoi,lm._miso,lm._sclk);
        this->cs=new DigitalOut(lm._cs);
        this->_msoi=lm._msoi;
        this->_miso=lm._miso;
        this->_cs=lm._cs;
        this->freq=lm.freq;
        spi->format(8,0);
        spi->frequency(freq);
        this->cwtime=lm.cwtime;
        this->wtime=lm.wtime;
    }
    void deallocate() {
        delete this->spi;
        this->spi=0;
        delete this->cs;
        this->cs=0;
    }
    void fillDeque(deque<uint8_t>& regs, char c, int shift,bool test=true) {
        int bin=0;
        if(test)
            bin=0xff>>(7-shift);
        else
            bin=0xff<<(7-shift);
        for(int i=0; i<8; i++) {
            int value=chars[c*8+i];
            value=value&bin;
            if(test) {
                value=value<<(7-shift);
                value=value&128;
                regs[i]=regs[i]>>1;
            } else {
                value=value>>(7-shift);
                value=value&1;
                regs[i]=regs[i]<<1;
            }
            regs[i]=regs[i]|value;
        }
    }
public:
    LMDriver(PinName msoi, PinName miso, PinName sclk, PinName chip_select);
    LMDriver(const LMDriver& lm);
    LMDriver& operator= (const LMDriver& lm);
    ~LMDriver();
    void Setup();
    void DisplayChar(char c);
    void DisplayString(const string& str);
    void DisplayStringCont(const string& str);
    void DisplayStringStep(const string& str, bool test=true);
    void SetSPIFreq(int freq) {
        if(freq<0)
            return;
        this->spi->frequency(freq);
    }
    void SetWaitTime(float time, bool test=true) {
        if(test) wtime=time;
        else cwtime=time;
    }
    void operator=(char c) {
        DisplayChar(c);
    }
    void operator=(const string& str) {
        DisplayStringCont(str);
    }
    void Clear() {
        zeroFill();
    }


};

void LMDriver:: DisplayStringStep(const string& str, bool test)
{
    deque<uint8_t> regs;
    for(int i=0; i<8; i++)
        regs.push_back(0);

    displayDeque(regs);

    for(int i=0; i<str.length(); i++) {
        for(int j=0; j<=8; j++) {
            if(j<8)
                fillDeque(regs,str[i],j,test);
            else if(j==8)
                fillDeque(regs,' ',7,test);
            displayDeque(regs);
            wait(cwtime);
        }
    }
}

LMDriver& LMDriver:: operator=(const LMDriver& lm)
{
    if(this==&lm)
        return *this;
    if(this->spi!=0 && this->cs!=0)
        deallocate();
    allocate(lm);
    return *this;
}

LMDriver::LMDriver(const LMDriver& lm)
{
    allocate(lm);
}



void LMDriver::DisplayStringCont(const string& str)
{
    if(str.length()==0)
        return;

    if(str.length()==1) {
        DisplayChar(str[0]);
        return;
    }

    deque<uint8_t> output;

    for(int i=0; i<8; i++)
        output.push_back(0);

    for(int i=0; i<str.size(); i++) {
        for(int j=0; j<10; j++) {
            displayDeque(output);
            wait(cwtime);
            output.pop_front();
            if(j<2)
                output.push_back(0);
            else if(j>=2)
                output.push_back(*(chars+str[i]*8+(j-2)));
        }
        displayDeque(output);
    }
}


void LMDriver::DisplayString(const string& str)
{
    if(str.length()==0)
        return;
    for(int i=0; i<str.size(); i++) {
        DisplayChar(str[i]);
        wait(wtime);
    }
    zeroFill();
}

void LMDriver::DisplayChar(char c)
{
    if(c>255)
        return;
    const uint8_t* array=chars+c*8;
    writeArray(array);
}

void LMDriver::Setup()
{
    writeToReg(reg_decodeMode,0x00);
    writeToReg(reg_intensity,0x0f);
    writeToReg(reg_scanLimit,0x07);
    writeToReg(reg_shutdown,0x01);
    writeToReg(0xff,0x00);
    zeroFill();
}
LMDriver::LMDriver(PinName msoi, PinName miso, PinName sclk,PinName chip_select):spi(0),cs(0)
{
    spi=new SPI(msoi,miso,sclk);
    spi->format(8,0);
    freq=1*MHz;
    spi->frequency(freq);
    cs= new DigitalOut(chip_select);
    _miso=miso;
    _msoi=msoi;
    _sclk=sclk;
    _cs=chip_select;
    wtime=0.5;
    cwtime=0.1;
}
LMDriver::~LMDriver()
{
    deallocate();
}

#endif
