/** Testprogramm fuer den SMD IoTKit Shield.
  */
#include "mbed.h"
#include "rtos.h"
#include "ReceiverIR.h"
#include "Servo.h"
#include "Motor.h"
#include "StepperMotorUni.h"
#include "MFRC522.h"
#include <string>
#include "ESP8266.h"
#include "RCSwitch.h"
#include "TMP175.h"
#include "SAA1064.h"
// Trace
#include "trace.h"

/** aktuell laufender Test */
int test = 0;
// Test vor- und rueckwaerts schalten
InterruptIn button1( SW2 );
InterruptIn button2( SW3 );

void next()
{
    test++;
    if  ( test > 16 )
        test = 0;
}
void previous()
{
    if  ( test > 0 )
        test--;
    if  ( test == 0 )
        test = 16;
}

/*------------------------------------------------------------------------
 * Testfunktionen
 ------------------------------------------------------------------------*/
 
/** Analoge Sensoren Werte ausgeben */
void doPrintAnalogeValues( )
{
    INFO( "Analog Thread" );
    
    // Analoge Sensoren
    AnalogIn poti( A0 );
    AnalogIn lightSensor( A1 );
    AnalogIn hallSensor( A2 );
    AnalogIn audioSensor( A3 );    
    
    printf( "Poti %f, Light %f, Hall %f Audio %f\n", poti.read(), lightSensor.read(), hallSensor.read(), audioSensor.read() );
}

/** LED's fortschalten */
int led = 1;

void doLedTicker()
{
    TRACE1( "LED Thread %d", led );
    
    // LED's (Aktoren)
    DigitalOut led1(D10);
    DigitalOut led2(D11);
    DigitalOut led3(D12);
    DigitalOut led4(D13);
    
    switch  ( led ) 
    {
        case 1:
            led1 = 1;
            break;
        case 2:
            led2 = 1;
            break;
        case 3:
            led3 = 1;
            break;
        case 4:
            led4 = 1;
            break;
        default:
            led1 = led2 = led3 = led4 = 0;
            led = 0;
            break;
    }
    led++;
}

/** Servo Thread */
void doServoThread()
{
    INFO( "Servo Thread" );
    
    Servo bottom( D9 );
    Servo arm( D10 );

    printf( "Servo 0 - 180\n" );
    for ( float p = 0.1f; p < 1.0f; p += 0.001f ) 
    {
        bottom = p;
        arm = p / 2;
        Thread::wait( 1 ); // ACHTUNG: Millisekunden!!!
    }

    printf( "Servo 180 - 0\n" );
    for ( float p = 1.0f; p >= 0.1f; p -= 0.001f ) 
    {
        bottom = p;
        arm = p / 2;
        Thread::wait( 1 );
    }
}
/** Motor 1 Test */
void doMotorThread()
{
    INFO( "Motor Thread" ); 
    
    Motor m1(D3, D2, D4); // pwm, fwd, rev
    Motor m2(D6, D5, D7); // pwm, fwd, rev
    
    printf( "Motor rueckwaerts\n" );
    for (double s= 0.5; s < 1.0 ; s += 0.01) 
    {
        m1.speed(s * -1);
        m2.speed(s );
        Thread::wait ( 40 );
    }
    Thread::wait( 100 );
    m1.speed( 0 );
    m2.speed( 0 );
    
    printf( "Motor vorwaerts\n" );
    for (double s= 0.5; s < 1.0 ; s += 0.01) 
    {
        m1.speed(s);
        m2.speed(s * -1);
        Thread::wait( 40 );
    }
    Thread::wait(  100 );
    m1.speed( 0 );
    m2.speed( 0 );
    
}

/** Schrittmotor Thread */
void doStepperThread()
{
    INFO( "Stepper Thread" );     
    
    StepperMotorUni motor1( D2, D3, D4, D5 );
    StepperMotorUni motor2( PTC5, PTC7, PTC0, PTC9 );
    StepperMotorUni motor3( PTC8, PTC1, PTB19, PTB18 );
        
    // Motordrehzahl
    motor1.set_pps( 100 );
    motor2.set_pps( 200 );
    motor3.set_pps( 300 );

    printf( "Schrittmotor vorwaerts\n" );
    motor1.move_steps( 512 );
    motor2.move_steps( 512 );
    motor3.move_steps( 512 );
    Thread::wait( 3000 );

    printf( "Schrittmotor  rueckwaerts\n" );
    motor1.move_steps( -512 );
    motor2.move_steps( -512 );
    motor3.move_steps( -512 );
    Thread::wait( 3000 );
}

/** MOSFET on/off */
void doMosfet()
{
    INFO( "MOSFET Thread" ); 
    
    DigitalOut mosfet( D7 );
            
    mosfet = 1;
    Thread::wait( 1000 );
    mosfet = 0;
}

/** ESP8266 Modem an Serieller Schnittstelle */
void doESP8266()
{
    INFO("ESP8266 Test");
    ESP8266 esp( A3, A2, 9600 ); // tx, rx (Bluetooth Header)
    char snd[255], rcv[1000]; // Strings for sending and receiving commands / data / replies
    wait(1);

    printf("Reset ESP\n");
    esp.Reset();
    esp.RcvReply(rcv, 400);
    printf("%s", rcv);
    wait(2);
    
    printf("Sending AT\n");
    strcpy(snd, "AT");
    esp.SendCMD(snd);
    esp.RcvReply(rcv, 400);
    printf("%s", rcv);
    wait(2);
    printf("Set mode to Client\n");
    esp.SetMode(1);
    esp.RcvReply(rcv, 1000);
    printf("%s", rcv);

    printf("Receiving Wifi List\n");
    esp.GetList(rcv);
    printf("%s", rcv);

    /*printf("Connecting to AP\n");
    esp.Join("MyAP", "MyPasswd" ); // Replace MyAP and MyPasswd with your SSID and password
    esp.RcvReply(rcv, 1000);
    printf("%s", rcv);
    wait(8);
    printf("Getting IP\n");
    esp.GetIP(rcv);
    printf("%s", rcv);
    printf("Setting multiple connections\r\n");
    esp.SetMultiple();
    esp.RcvReply(rcv, 1000);
    printf("%s", rcv);*/

    printf("THE END\n");
}

/** RFID Reader MFRC522 Test */
void doRFIDReader()
{
    INFO("RFID Reader MFRC522 Test");
    // NFC/RFID Reader an WiFi Header von K64F
    MFRC522    rfidReader( D11, D12, D13, D10, D8 );
    rfidReader.PCD_Init(); 

    // RFID Reader
    if ( rfidReader.PICC_IsNewCardPresent())
        if ( rfidReader.PICC_ReadCardSerial()) 
        {
            // Print Card UID
            printf("Card UID: ");
            for (uint8_t i = 0; i < rfidReader.uid.size; i++)
                printf("%02X:", rfidReader.uid.uidByte[i]);
            printf("\n");
            // Print Card type
            uint8_t piccType = rfidReader.PICC_GetType(rfidReader.uid.sak);
            printf("PICC Type: %s \n", rfidReader.PICC_GetTypeName(piccType));
        }
}

 /** 3 x 3 Werte */
unsigned int strip[9];
 
void writeLED( SPI &spi )
{
    for ( int p = 0; p < 9; p++ )
        spi.write( strip[p] );
}
 
void clearLED( SPI &spi )
{
    for ( int p = 0; p < 9; p++ ) 
    {
        strip[p] = 0;
        spi.write( strip[p] );
    }
}

/** LED Strip am SPI Bus */
void doSPILedStrip()
{
    INFO( "LED Strip Test" );

    SPI spi(D11, NC, D13); // mosi, miso, sclk
    spi.format( 8,0 );
    spi.frequency( 800000 );

    // Gruen, Rot, Blau - von Dunkel bis Hell
    for ( int i = 0; i < 128; i+=32 ) 
    {
        // LED 1
        strip[0] = i;
        strip[1] = 0;
        strip[2] = 0;
        // LED 2
        strip[3] = 0;
        strip[4] = i;
        strip[5] = 0;
        // LED 3
        strip[6] = 0;
        strip[7] = 0;
        strip[8] = i;
        writeLED( spi );
        wait( 0.1 );
    }
    wait( 1.0 );
    clearLED( spi );
}

void LEDStripOff( PwmOut& red, PwmOut& green, PwmOut& blue )
{
    printf( "off \n" );
    red = 0;
    green = 0;
    blue = 0;
    wait ( 0.2 );
}
 
void LEDStripDim( PwmOut& pin )
{
    printf( "dim\n" );
    for ( float i = 0.0f; i < 1.0f; i += .01f )
    {
        pin = i;
        wait( 0.02 );
    }
}
 
/** LED Strip 12Volt Test */
void doLEDStrip()
{
    INFO( "LED Strip V12 Test" );
    PwmOut green( D5 );
    PwmOut red( D6 );
    PwmOut blue( D7 );
    
    LEDStripDim( red );
    LEDStripOff( red, green, blue );
    LEDStripDim( green );
    LEDStripOff( red, green, blue );
    LEDStripDim( blue );
    LEDStripOff( red, green, blue );

    red = 1;
    wait( 0.2 );
    LEDStripOff( red, green, blue );

    green = 1;
    wait( 0.2 );
    LEDStripOff( red, green, blue );

    blue = 1;
    wait( 0.2 );
    LEDStripOff( red, green, blue );

    red = 1;
    blue = 1;
    green = 1;
    wait( 0.2 );
    LEDStripOff( red, green, blue );
}

/** Funksteckdose ein/aus */
void doRC()
{
    INFO( "RCSwitch Test" );
    RCSwitch mySwitch = RCSwitch( D9, D8 ); // Sender / Empfaenger (optional)
    
    printf( "Sende nach: Adresse 1 - 5 alle On, Geraet B aus A-D " );
    // Adresse 1 - 5 alle On, Geraet B aus A-D
    mySwitch.switchOff("11111", "01000");
    wait( 0.5 );
    
    printf( "- ON\n" );
    mySwitch.switchOn ("11111", "01000");
}

/*------------------------------------------------------------------------
 * Infrarot Funktionen
 ------------------------------------------------------------------------*/

/** Receive.
 * @param format Pointer to a format.
 * @param buf Pointer to a buffer.
 * @param bufsiz Size of the buffer.
 * @return Bit length of the received data.
 */
int receiveIR(ReceiverIR &ir_rx, RemoteIR::Format *format, uint8_t *buf, int bufsiz, int timeout = 100) 
{
    int cnt = 0;
    while (ir_rx.getState() != ReceiverIR::Received) 
    {
        cnt++;
        if (timeout < cnt) 
            return -1;
    }
    return ir_rx.getData(format, buf, bufsiz * 8);
}

/**
 * Display a format of a data.
 */
void displayIRFormat(RemoteIR::Format format) 
{
    switch (format) 
    {
        case RemoteIR::UNKNOWN:
            printf("????????\n");
            break;
        case RemoteIR::NEC:
            printf("NEC     \n");
            break;
        case RemoteIR::NEC_REPEAT:
            printf("NEC  (R)\n");
            break;
        case RemoteIR::AEHA:
            printf("AEHA    \n");
            break;
        case RemoteIR::AEHA_REPEAT:
            printf("AEHA (R)\n");
            break;
        case RemoteIR::SONY:
            printf("SONY    \n");
            break;
    }
}

/** Display a data.
 * @param buf Pointer to a buffer.
 * @param bitlength Bit length of a data.
 */
void displayIRData(uint8_t *buf, int bitlength) 
{
    const int n = bitlength / 8 + (((bitlength % 8) != 0) ? 1 : 0);
    
    for (int i = 0; i < n; i++) 
        printf("%02X", buf[i]);
    for (int i = 0; i < 8 - n; i++) 
        printf("--");
}

/** IR Empfaenger abfragen */
void onReceiveIR()
{
    INFO( "IR Receive Test" );
    ReceiverIR ir_rx( PTB20 );

    uint8_t buf1[32];
    uint8_t buf2[32];
    int bitlength1;
    RemoteIR::Format format;

    memset(buf1, 0x00, sizeof(buf1));
    memset(buf2, 0x00, sizeof(buf2));

    bitlength1 = receiveIR( ir_rx, &format, buf1, sizeof(buf1));
    if (bitlength1 < 0)
        return;

    displayIRData(buf1, bitlength1);
    displayIRFormat(format);
}

/** Buttons auf Shield */
void doButtons()
{
    INFO ( "Buttons Test -> press button" );
    DigitalIn button3( A0, PullUp );
    DigitalIn button4( A1, PullUp );
    DigitalIn button5( A2, PullUp );
    
    //while   ( 1 )
    //{
        if  ( button1 == 0 )
        {
            printf( "button 1 pressed\n" );
            return;
        }
        if  ( button2 == 0 )
        {
            printf( "button 2 pressed\n" );
            return;
        }                
        if  ( button3 == 0 )
        {
            printf( "button 3 pressed\n" );
            return;
        }
        if  ( button4 == 0 )
        {
            printf( "button 4 pressed\n" );
            return;
        }
        if  ( button5 == 0 )
        {
            printf( "button 5 pressed\n" );
            return;
        } 
        wait    ( 0.2 );               
    //}
}

/** I2C Temperatorsensor */
void doTempSensor()
{
    INFO( "Temperator Sensor Test" );
    TMP175 mySensor( D14, D15 );
    
    float Temp;
    mySensor.vSetConfigurationTMP175(SHUTDOWN_MODE_OFF|COMPARATOR_MODE|POLARITY_0|FAULT_QUEUE_6|RESOLUTION_12,0x48); // Address -> A0,A1,A2=0
    mySensor.vSetTemperatureLowTMP175(25.0);
    mySensor.vSetTemperatureHighTMP175(35.0);

    for( int k = 0; k < 100; k++ ) 
    {
        Temp = mySensor;
        printf( "Temperature: %f C\n",Temp);
    }
}

/** LED Digit Test */
void do4DigitLED()
{
    INFO( "LED Digit Test" );
    SAA1064 display;
    
    for ( int i = 0; i < 10; i++ ) 
    {
        display.write( i, i+1, i+2, i+3 );
        wait( 0.2 );
    }
    for ( int i = 1000; i < 1234+1; i++ ) 
    {
        display.writeInt( i );
        wait( 0.2 );
    }    
}

/** I2C Kommunikation mit Arduino */
void doI2CArduino()
{
    INFO  ( "I2C Master Test\n" );
    I2C i2c( D14, D15 );

    // CMD Buffer (cmd, addr, arg)
    char cmd[4] = { 0x00, 0x00, 0x00, 0x00 };
    int slave = 0x10 << 1;
    int status = 0;

    // write I2C
    for ( int i = 4; i <= 7; i++ ) 
    {
        cmd[0] = 'w';
        cmd[1] = i;
        cmd[2] = 1;
        printf( "write to %d, %d - ", slave, (int) cmd[2] );
        status = i2c.write( slave, cmd, 3 );
        printf("Status %d\n", status );
        wait( 1.0 );

        cmd[0] = 'w';
        cmd[1] = i;
        cmd[2] = 0;
        printf( "write to %d, %d - ", slave, (int) cmd[2] );
        status = i2c.write( slave, cmd, 3 );
        printf("Status %d\n", status );
        wait( 0.5 );
    }
    // read I2C
    i2c.read( slave, cmd, 2 );  // Arduino liefert einen Wert von 0 - 1024
    printf  ( "Analog %d\n", (int) (cmd[0] * 256 + cmd[1])  );
}

/** Buzzer (Summer) */
void doBuzzer()
{
    INFO( "Buzzer Test" );
    PwmOut buzzer( D7 );
    buzzer.period_us( 250 );
    
    buzzer = 0.5;
    wait( 1.0 );
    buzzer = 0.0;
}

/*------------------------------------------------------------------------
 * Hauptprogramm
 ------------------------------------------------------------------------*/
 
int main()
{
    // SW2 + 3 auf K64F schalten Test vor- und zurueck.
    button1.fall( &next );
    button2.fall( &previous );
    
    while   ( 1 )
    {
        switch  ( test )
        {
            case 0:
                doPrintAnalogeValues(); // Analoge Sensoren
                break;
            case 1:
                //doLedTicker();          // Lauflicht
                break;
            case 2:
                doServoThread();        // Kamara Plattform mit Servos
                break;
            case 3:
                //doMotorThread();        // 2 Motoren
                break;
            case 4:
                doStepperThread();      // Schrittmotor
                break;                                
            case 5:
                doButtons();            // Buttons auf Shield
                break;   
            case 6:
                doMosfet();             // MOSFET LED oder LED Strip V12
                break;
            case 7:
                doLEDStrip();           // LED Strip 12 Volt Variante
                break;   
            case 8:
                doSPILedStrip();        // LED Strip am SPI Bus
                break;   
            case 9:
                doRFIDReader();         // RFID Reader
                break;                             
            case 10:
                doTempSensor();         // Temperatur Sensor am I2C Bus 0x40
                break;                             
            case 11:
                doBuzzer();             // Buzzer (Summer)
                break;  
            case 12:
                do4DigitLED();          // LCD im I2C Bus
                break;                                              
            case 13:
                doI2CArduino();         // Arduino am I2C Bus
                break;                                              
            case 14:
                doRC();                 // RC Switch (Funk)
                break;   
            case 15:
                onReceiveIR();          // IR Receiver
                break;   
            case 16:
                doESP8266();            // WLAN Modem
                break;     
            default:
                break;
        }
        test++;
        if  ( test > 16 )
            test = 0;
            
        doLedTicker();          // Lauflicht
        wait( 1.0 );
    }        
}