/**
 * Bosch BMP085 Barometer driver
 */

#ifndef BMP085_H
#define BMP085_H

#include <stdint.h>
#include "mbed.h"

#define BMP085_ADDR     (0x77<<1)

typedef enum {
    BMP085_CAL_AC1            = 0xAA,
    BMP085_CAL_AC2            = 0xAC,
    BMP085_CAL_AC3            = 0xAE,
    BMP085_CAL_AC4            = 0xB0,
    BMP085_CAL_AC5            = 0xB2,
    BMP085_CAL_AC6            = 0xB4,
    BMP085_CAL_B1             = 0xB6,
    BMP085_CAL_B2             = 0xB8,
    BMP085_CAL_MB             = 0xBA,
    BMP085_CAL_MC             = 0xBC,
    BMP085_CAL_MD             = 0xBE,
    BMP085_CHIPID             = 0xD0,
    BMP085_VERSION            = 0xD1,
    BMP085_SOFTRESET          = 0xE0,
    BMP085_CTRL               = 0xF4,
    BMP085_DATA_TEMP          = 0xF6,
    BMP085_DATA_PRESSURE      = 0xF6,
    BMP085_CMD_READ_TEMP      = 0x2E,
    BMP085_CMD_READ_PRESSURE  = 0x34
} BMP085_Register;

typedef enum {
    BMP085_MODE_ULTRA_LOW_POWER          = 0,
    BMP085_MODE_STANDARD                 = 1,
    BMP085_MODE_HIGH_RESOLUTION          = 2,
    BMP085_MODE_ULTRA_HIGH_RESOLUTION    = 3
} BMP085_Mode;

typedef enum {
    BMP085_IDLE,
    BMP085_BUSY
} BMP085_State;

typedef struct {
    int16_t  ac1;
    int16_t  ac2;
    int16_t  ac3;
    uint16_t ac4;
    uint16_t ac5;
    uint16_t ac6;
    int16_t  b1;
    int16_t  b2;
    int16_t  mb;
    int16_t  mc;
    int16_t  md;
} BMP085_Calibration_Data;

class BMP085 {
public:
    BMP085(PinName sda, PinName scl, BMP085_Mode mode = BMP085_MODE_STANDARD) : _i2c(sda, scl), _mode(mode), _state(BMP085_IDLE) { init(); };
    BMP085(I2C & i2c, BMP085_Mode mode = BMP085_MODE_STANDARD) : _i2c(i2c), _mode(mode), _state(BMP085_IDLE) { init(); };
    
    float get_temperature(void) { return _temperature; };
    float get_pressure(void) { return _pressure; };
    void measure();
    void _callback_handler();
protected:
    void init(void);
    uint16_t read_cal_register(BMP085_Register reg);
    uint16_t read_uc_temperature(void);
    uint32_t read_uc_pressure(void);
    I2C _i2c;
    BMP085_Mode _mode;
    BMP085_State _state;
    float _temperature;
    float _pressure;
    BMP085_Calibration_Data _cal_data;
private:
    event_callback_t _callback;
    void _callback_handler(int event);
    uint8_t _tx_buf[2];
    uint8_t _rx_buf[3];
};

#endif /* BMP085_H */