#include "mbed.h"
#include "security.h"
#include "easy-connect.h"
#include "simple-mbed-client.h"

SimpleMbedClient client;

// Declare peripherals
DigitalOut connectivityLed(LED2);                   // Blinks while connecting, turns solid when connected
DigitalOut augmentedLed(LED1, BUILTIN_LED_OFF);     // LED that can be controlled through Connector
InterruptIn btn(MBED_CONF_APP_BUTTON);              // Button that sends it's count to Connector

Ticker connectivityTicker;

// Callback function when the pattern resource is updated
void patternUpdated(string v) {
    printf("New pattern: %s\n", v.c_str());
}

// Define resources here. They act as normal variables, but are exposed to the internet...
SimpleResourceInt btn_count = client.define_resource("button/0/clicks", 0, M2MBase::GET_ALLOWED);
SimpleResourceString pattern = client.define_resource("led/0/pattern", "500:500:500:500:500:500:500", &patternUpdated);

void fall() {
    btn_count = btn_count + 1;
    printf("Button count is now %d\r\n", static_cast<int>(btn_count));
}

void toggleConnectivityLed() {
    connectivityLed = !connectivityLed;
}

void registered() {
    printf("Registered\r\n");

    connectivityTicker.detach();
    connectivityLed = BUILTIN_LED_ON;
}

void play(void* args) {
    connectivityLed = BUILTIN_LED_OFF;

    // Parse the pattern string, and toggle the LED in that pattern
    string s = static_cast<string>(pattern);
    size_t i = 0;
    size_t pos = s.find(':');
    while (pos != string::npos) {
        wait_ms(atoi(s.substr(i, pos - i).c_str()));
        augmentedLed = !augmentedLed;

        i = ++pos;
        pos = s.find(':', pos);

        if (pos == string::npos) {
            wait_ms(atoi(s.substr(i, s.length()).c_str()));
            augmentedLed = !augmentedLed;
        }
    }

    augmentedLed = BUILTIN_LED_OFF;
    connectivityLed = BUILTIN_LED_ON;
}

int main() {
    // SimpleClient gives us an event queue which we can use, running on a separate thread (see https://docs.mbed.com/docs/mbed-os-handbook/en/5.1/concepts/events/)
    EventQueue* eventQueue = client.eventQueue();

    // Handle interrupts on the eventQueue - so we don't do things in interrupt context
    btn.fall(eventQueue->event(&fall));

    // Toggle the built-in LED every second (until we're connected, see `registered` function)
    connectivityTicker.attach(eventQueue->event(&toggleConnectivityLed), 1.0f);

    // Functions can be executed through mbed Device Connector via POST calls (also defer to the event thread, so we never block)
    client.define_function("led/0/play", eventQueue->event(&play));

    // Connect to the internet (see mbed_app.json for the connectivity method)
    NetworkInterface* network = easy_connect(true);
    if (!network) {
        return 1;
    }

    // Connect to mbed Device Connector
    struct MbedClientOptions opts = client.get_default_options(); // opts contains information like the DeviceType
    bool setup = client.setup(opts, network);
    if (!setup) {
        printf("Client setup failed\n");
        return 1;
    }
    client.on_registered(&registered);
}
