/* LoRaWAN API Class
 * Copyright (c) 2016 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef LORAWANINTERFACE_H
#define LORAWANINTERFACE_H
 
#include "mbed.h"

#include <string>
#include <vector>

class LoRaWAN
{
public:
    // Sequence counter is a security mechanism in LoRaWAN which prevents
    // replay attacks. Adding a simple way to restore the counters from
    // non-volatile storage.
    LoRaWAN(int16_t seq_down_counter = 0, int16_t seq_up_counter = 0);

    // Events that fire whenever one of the sequence counters change,
    // so it's easy to hook up non-volatile storage API to them.
    void (*on_seq_down_change)(int16_t);
    void (*on_seq_up_change)(int16_t);

    // Set up this node for personalized activation. The reason why I'm
    // choosing strings over buffers is because there are a ton of problems
    // with endian-ness, and transforming the keys into bytes. I've not seen
    // anyone get this right the first time they try LoRa.
    // Strings are much simpler, and easy to copy from a website.
    int8_t init_personalized_node(std::string network_address,
                                  std::string network_session_key,
                                  std::string application_session_key,
                                  std::string network_id = "");

    // Same as init_personalized_node, but because OTA is asynchronous we
    // can also pass in a callback.
    int8_t init_ota_node(std::string device_eui,
                         std::string application_eui,
                         std::string application_key,
                         std::string network_id = "",
                         void (*on_joined)(int16_t, std::string) = 0);
    // Same function as we pass into init_ota_node. Gives a callback when OTA
    // succeeded or failed. String as second argument to give more info
    // if a JOIN failed.
    void (*on_ota_callback)(int16_t, std::string);

    // Spread factors (taken from Multitechs implementation)
    enum DataRate {
        SF_12, SF_11, SF_10, SF_9, SF_8, SF_7, SF_7H, SF_50
    };

    // The standard spread factor / data rate to be used
    // need return type non-void, as spread factors differ per US/EU freq.
    // so need to validate.
    int8_t set_default_data_rate(DataRate default_data_rate);
    DataRate get_default_data_rate(void);

    // Adaptive data rating
    void set_adr(bool enable_adr);
    bool get_adr(void);

    // Transmission power, need return type non-void,
    // as we need to validate the input param
    int8_t set_tx_power(int8_t tx_power);
    uint8_t get_tx_power(void);

    // Receive power, need return type non-void,
    // as we need to validate the input param
    int8_t set_rx_power(int8_t rx_power);
    uint8_t get_rx_power(void);

    // Send a message over the network, we need a port and data
    // ACK can be enabled, and when it is we provide a callback method.
    // The callback method will also be invoked if timeout happens and we no
    // longer listen for ACK.
    int8_t send(int8_t port,
                std::vector<uint8_t> data,
                bool ack = false,
                void(*on_ack)(int8_t) = 0);

    // Register a function which will be invoked when we receive a message.
    // Pass in a callback which will receive two parameters: port and data.
    void on_receive_message(void(*on_receive_message)(int8_t, std::vector<uint8_t>));

    // Duty cycle limitations makes that we cannot send constantly.
    // Get the time until the next time we can send.
    // (Failure to obide will make send function fail).
    int32_t get_ms_until_next_free_channel(void);

    // Register a callback which will fire whenever a free channel is available.
    // We should be able to combine this with deep-sleep methods in mbed to
    // make a device go to sleep until the next window is available.
    void on_next_free_channel(void(*on_free_channel)(void));

    // Frequency bands which LoRa currently operates.
    enum FrequencyBand {
        F_433, F_868, F915
    };
    // get the frequency band the module is working on.
    // This does not require a setter, as it's hardware property.
    FrequencyBand get_frequency_band(void);

    // US only. Get/set frequency sub band.
    enum FrequencySubBand {
        FSB_ALL, FSB_1, FSB_2, FSB_3, FSB_4, FSB_5, FSB_6, FSB_7, FSB_8
    };
    void set_frequency_sub_band(FrequencySubBand sub_band);
    FrequencySubBand get_frequency_sub_band(void);
};

#endif

