// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.

#ifndef __SAFECIRCBUF_H__
#define __SAFECIRCBUF_H__

#include "mbed.h"
#include "rtos.h"

#include <CircularBuffer.h>

template<typename T, uint32_t BufferSize, typename CounterType = uint32_t>
class SafeCircBuf
{
public:
    SafeCircBuf() {
    }

    ~SafeCircBuf() {
    }

    /** Push the transaction to the buffer. This overwrites the buffer if it's
     *  full
     *
     * @param data Data to be pushed to the buffer
     */
    void push(const T& data) {
        _mutex.lock();
        _circBuf.push(data);
        _mutex.unlock();
    }

    /** Pop the transaction from the buffer
     *
     * @param data Data to be pushed to the buffer
     * @return True if the buffer is not empty and data contains a transaction, false otherwise
     */
    bool pop(T& data) {
        bool rc;
        _mutex.lock();
        rc = _circBuf.pop(data);
        _mutex.unlock();
        return rc;
    }

    /** Check if the buffer is empty
     *
     * @return True if the buffer is empty, false if not
     */
    bool empty() {
        bool rc;
        _mutex.lock();
        rc = _circBuf.empty();
        _mutex.unlock();
        return rc;
    }

    /** Check if the buffer is full
     *
     * @return True if the buffer is full, false if not
     */
    bool full() {
        bool rc;
        _mutex.lock();
        rc = _circBuf.full();
        _mutex.unlock();
        return rc;
    }

    /** Reset the buffer
     *
     */
    void reset() {
        _mutex.lock();
        _circBuf.reset();
        _mutex.unlock();
    }

private:
    Mutex _mutex;
    CircularBuffer<T, BufferSize, CounterType> _circBuf;
};

#endif
