#include "mbed.h"
#include "rtos.h"

#define GPS_ENABLE      1
#define WIFI_ENABLE     1
#define ETHER_ENABLE    0

#if WIFI_ENABLE
#include "ESP8266.h"
#endif // WIFI_ENABLE

#if ETHER_ENABLE
#include "EthernetInterface.h"
#endif // ETHER_ENABLE

#if GPS_ENABLE
#include "gps.h"
#endif // GPS_ENABLE

/****************************************************************************/
/* Ether関連の定義                                                          */
/****************************************************************************/
#if ETHER_ENABLE
EthernetInterface eth;
#define HOST_NAME       "104.199.222.173"               // HOST Address
#define HOST_URL        "/r-king/space/webapi.php?"     // HOST URL
#define HOST_PORT       (80)                            // HOST Port
#endif // ETHER_ENABLE

/****************************************************************************/
/* Wi-Fi関連の定義                                                          */
/****************************************************************************/
#if WIFI_ENABLE
SerialBuffered serial_buffered(P8_14, P8_15);
ESP8266 wifi;
//#define WLAN_SSID       "4CE67630E22B"                  // AP Name
//#define WLAN_PASS       "t3340pn5mkmkh"                 // AP PassWord
//#define HOST_NAME       "104.199.222.173"               // HOST Address
//#define HOST_URL        "/r-king/space/webapi.php?"     // HOST URL


#define WLAN_SSID       "DIRECT-W4Q0:DSC-QX10"                  // AP Name
#define WLAN_PASS       "kXocAsgx"                 // AP PassWord
#define HOST_NAME       "10.0.0.1"               // HOST Address
#define HOST_PORT       (10000)                            // HOST Port
#define ESP_Serial      serial_buffered                 // WiFi Module Port
#define ESP_Baud        115200                          // WiFi Module Port
#define WIFI_BUF        256                             // WiFi Buffer Length

#define CAMERA_SET_PICMODE      "{\"method\": \"setShootMode\",\"params\": [\"still\"],\"id\": 1,\"version\": \"1.0\"}"
#define CAMERA_SET_MOVIEMODE    "{\"method\": \"setShootMode\",\"params\": [\"movie\"],\"id\": 1,\"version\": \"1.0\"}"
#define CAMERA_SHOT             "{\"method\": \"actTakePicture\",\"params\": [],\"id\": 1,\"version\": \"1.0\"}"
#define CAMERA_REC_START        "{\"method\": \"startMovieRec\",\"params\": [],\"id\": 1,\"version\": \"1.0\"}"
#define CAMERA_REC_STOP         "{\"method\": \"stopMovieRec\",\"params\": [],\"id\": 1,\"version\": \"1.0\"}"
int     cameraStat = 0;
#endif // WIFI_ENABLE

/****************************************************************************/
/* グローバル変数の定義                                                     */
/****************************************************************************/
int     isEnableWiFi =      false;
int     isEnableGPS  =      false;
double  g_lat        =      12.3f;
double  g_lon        =      45.6f;
double  g_alt        =      78.9f;
int     g_year       =      2016;
int     g_month      =      01;
int     g_day        =      01;
int     g_hour       =      00;
int     g_min        =      00;
int     g_sec        =      00;

/****************************************************************************/
/* プロトタイプ宣言                                                         */
/****************************************************************************/
static void loop(void);
#if ETHER_ENABLE
static int setupEther();
static int sendEther();
#endif // ETHER_ENABLE
#if WIFI_ENABLE
static int setupWiFi();
static int sendWiFi();
static void resetWiFi();
#endif // WIFI_ENABLE
#if GPS_ENABLE
int setupGPS();
static void onGpsGdaReceivedCallback(SerialGPS::NMEA_GGA* data);
static void onGpsZdaReceivedCallback(SerialGPS::NMEA_ZDA* data);
static SerialGPS    s_SerialGPS(P2_14, P2_15);    // ★PIN番号は機器にあわせること★
static SerialGPS::GpsCallbackTable s_GpsCallbackTable = {
    /*.log_cbfunc = */ NULL,
    /*.gga_cbfunc = */ onGpsGdaReceivedCallback,
    /*.gll_cbfunc = */ NULL,
    /*.gsa_cbfunc = */ NULL,
    /*.gsv_cbfunc = */ NULL,
    /*.rmc_cbfunc = */ NULL,
    /*.vtg_cbfunc = */ NULL,
    /*.zda_cbfunc = */ onGpsZdaReceivedCallback
};
#endif //GPS_ENABLE

int main() {
        printf("[Serial Start]\r\n");
        
        // Setup Wi-Fi
#if WIFI_ENABLE
        isEnableWiFi = setupWiFi();
#endif // WIFI_ENABLE

#if ETHER_ENABLE
        setupEther();
#endif // ETHER_ENABLE

        // Setup GPS
        isEnableGPS = setupGPS();
        
        //mc.baud(115200);
        /*
        char data1[7] = {0xFF,0x1C,0x04,0x03,0x01,0x01,0x28};//時計回り
        mc.printf(data1);
        printf(data1);
        
        wait(3);
        
        char data2[7] = {0xFF,0x1C,0x04,0x03,0x01,0x02,0x28};//反時計まわり
        mc.printf(data2);
        printf(data2);
        
        wait(3);
        
        char data3[7] = {0xFF,0x1C,0x04,0x03,0x01,0x00,0x3C};//モータ停止TODO:00
//        mc.printf(data3);
        for(int i = 0 ; i < sizeof(data3) ; i++){
            mc.putc(data3[i]);
        }
//        printf(data3);
        
        printf("[Serial End]\n");
        */
        
        loop();
}
int cnt = 0;
void loop(void)
{
    while(true)
    {
#if GPS_ENABLE
        s_SerialGPS.Processing();//
#endif  //GPS_ENABLE   
        wait_ms(10);
        
        if(cnt >= 1000)
        {
#if ETHER_ENABLE
            sendEther();
#endif // ETHER_ENABLE
#if WIFI_ENABLE
            sendWiFi();
#endif // WIFI_ENABLE
            cnt = 0;   
        }
        cnt++;
    }
}

#if ETHER_ENABLE
/****************************************************************************/
//  機能      : Ether通信機能のセットアップ
//  引数      : なし
//  返り値     : ret           FALSE(0):失敗 TRUE(1):成功
//  備考      : なし
/****************************************************************************/
static int setupEther()
{
    eth.init(); //Use DHCP
    eth.connect();
    printf("IP Address is %s\n", eth.getIPAddress());
    
    return 1;
}

/****************************************************************************/
//  機能      : Ether送信処理
//  引数      : なし
//  返り値     : ret           FALSE(0):失敗 TRUE(1):成功
//  備考      : なし
/****************************************************************************/
static int sendEther()
{
    TCPSocketConnection sock;
    sock.connect(HOST_NAME, HOST_PORT);
    
    // build GET Request
    char buf[1024] = {'\0'};
    // sprintf(buf, "GET %saction=add&token=001&lat=%lf&lon=%lf&height=%lf HTTP/1.1\r\nHost: %s\r\nConnection: close\r\n\r\n",
    sprintf(buf, "GET %saction=add&token=001&lat=%lf&lon=%lf&height=%lf HTTP/1.1\r\nHost: %s\r\n\r\n",
        HOST_URL,
        g_lat,
        g_lon,
        g_alt,
        HOST_NAME);

    printf(buf);

    sock.send_all(buf, sizeof(buf)-1);
    
    char buffer[300];
    int ret;
    while (true) {
        ret = sock.receive(buffer, sizeof(buffer)-1);
        if (ret <= 0)
            break;
        buffer[ret] = '\0';
        printf("Received %d chars from server:\n%s\n", ret, buffer);
    }
      
    sock.close();
    
    return 1;
}
#endif // ETHER_ENABLE

#if WIFI_ENABLE
/****************************************************************************/
//  機能      : Wi-Fi通信機能のセットアップ
//  引数      : なし
//  返り値     : ret           FALSE(0):失敗 TRUE(1):成功
//  備考      : なし
/****************************************************************************/
static int setupWiFi()
{
    int ret    = false;

    // Wi-Fiとのシリアル通信を開始する
    wifi.begin(ESP_Serial, ESP_Baud);
    
    // Wi-Fiモジュールをリセット
    resetWiFi();
    
    // FWバージョンを取得
    printf("FW Version:");
    printf(wifi.getVersion().c_str());            // AT+GMR
    printf("\r\n");
    
    // 既に接続済みであれば切断
    wifi.leaveAP();                               // AT+CWQAP
    
    // 接続の準備
    if (wifi.setOprToStationSoftAP()) {           // AT+CWMODE=3
        printf("to station + softap ok\r\n");
    } else {
        printf("to station + softap err\r\n");
        //return ret;
    }
    
    // 接続可能なAP一覧を表示
    //printf("%s\r\n", wifi.getAPList().c_str()); // AT+CWLAP
    
    // APに接続する
    if (wifi.joinAP(WLAN_SSID, WLAN_PASS)) {      // AT+CWJAP="SSID","PASSWORD"
        printf("Join AP success\r\n");
        printf("IP:");
        printf( wifi.getLocalIP().c_str());       
        printf("\r\n");
    } else {
        printf("Join AP failure\r\n");
        return ret;
    }

    // コネクションを1つのみに設定
    
    if (wifi.disableMUX()) {                        // AT+CIPMUX=0
        printf("multi ok\r\n");
    } else {
        printf("multi err\r\n");
        return ret;
    }
    
    printf("setup end\r\n");
    ret = true;

    return ret;
}

/****************************************************************************/
//  機能      : Wi-Fi送信機能
//  引数      : buffer        送信データ
//  返り値     : ret           FALSE(0):失敗 TRUE(1):成功
//  備考      : なし
/****************************************************************************/
static int sendWiFi()
{
    int ret = false;
    
    uint8_t buffer[2048] = {0};
    bool sendRet = false;

    printf("sendWiFi\r\n");
    if (wifi.createTCP(HOST_NAME, HOST_PORT)) {         // AT+CIPSTART="TCP","104.199.222.173",80
        printf("create tcp ok\r\n");
    } else {
        printf("create tcp err\r\n");
        // システムリセット
        NVIC_SystemReset();
        return ret;
    }
    wait(1);
    
    // build POST Request
    char buf[1024] = {'\0'};
    // 動画の撮影
    char dat[128] = {'\0'};
    
    //sprintf(dat, "%s", CAMERA_SHOT);
    
    switch(cameraStat)
    {
        case 0:
            sprintf(dat, "%s", CAMERA_SET_MOVIEMODE);
            cameraStat++;
        break;
        
        case 1:
            sprintf(dat, "%s", CAMERA_REC_START);
            cameraStat++;
        break;
        
        case 2:
            sprintf(dat, "%s", CAMERA_REC_STOP);
            cameraStat++;
        break;
        
        default:
            //
        break;
    }
    
    sprintf(buf, "POST /sony/camera HTTP/1.1\r\nHost: 10.0.0.1\r\nContent-Type: application/x-www-form-urlencoded\r\nContent-Length: %d\r\n\r\n%s\r\n",(strlen(dat)+1), dat);

    /*
    sprintf(buf, "GET %saction=add&token=001&lat=%lf&lon=%lf&height=%lf HTTP/1.1\r\nHost: %s\r\n\r\n",
        HOST_URL,
        g_lat,
        g_lon,
        g_alt,
        HOST_NAME);
        */

    printf(buf);
    
    // Send GET Request
    sendRet = wifi.send((const uint8_t*)buf, strlen(buf));  // AT+CIPSEND=送信バイト数
    if(sendRet == true) {
        printf("Success!!\r\n");
        ret = true;
    } else {
        printf("Fail!!\r\n");
        // Wi-Fiモジュールをリセット
        resetWiFi();
        return ret;
    }
    wait(1);
    
    // wait Response (T/O 10Sec)
    uint32_t len = wifi.recv(buffer, sizeof(buffer), 10000);
    if (len > 0) {
        printf("Received:[");
        for(uint32_t i = 0; i < len; i++) {
            printf("%c", buffer[i]);
        }
        printf("]\r\n");
    }

    // release TCP
    if (wifi.releaseTCP()) {
        printf("release tcp ok\r\n");
    } else {
        printf("release tcp err\r\n");
    }
    
    

    return ret;
}

/****************************************************************************/
//  機能      : Wi-Fiリセット機能
//  引数      : buffer        送信データ
//  返り値     : ret           FALSE(0):失敗 TRUE(1):成功
//  備考      : なし
/****************************************************************************/
static void resetWiFi()
{
    printf("Wi-Fi kick\r\n");
    wifi.kick();                                   // AT
    printf("Wi-Fi reset\r\n");
    wifi.restart();                                // AT+RST
    wait(10);
}
#endif // WIFI_ENABLE

#if GPS_ENABLE
/****************************************************************************
モジュール名  ： GPS Setup
関数名         ： setupGPS
引数          ： なし
戻り値         ： GPS受信データ
処理概要        ： GPS受信処理
注意事項        ： なし
****************************************************************************/
int setupGPS()
{
    int ret = false;
#if GPS_ENABLE
    // GPSライブラリの呼び出しサンプルコード
    s_SerialGPS.Init(); // GPS初期化
    s_SerialGPS.SetGpsCallback(&s_GpsCallbackTable);  // GPSコールバック登録
    s_SerialGPS.RcvIsrEnable();   // GPS受信有効
    ret = true;
#endif // GPS_ENABLE
    return ret;
}


/****************************************************************************
モジュール名  ： GPS受信コールバック
関数名         ： onGpsGdaReceivedCallback
引数          ： なし
戻り値         ： GPS受信データ
処理概要        ： GPS受信処理
注意事項        ： なし
****************************************************************************/
void onGpsGdaReceivedCallback(SerialGPS::NMEA_GGA* data)
{
    double d_lat,d_lon,m_lat,m_lon;
    
    // lat
    d_lat=int(data->latitude/100);
    m_lat=(data->latitude - d_lat*100)/60;
    g_lat=d_lat+m_lat;
    
    // lon
    d_lon=int(data->longitude/100);
    m_lon=(data->longitude - d_lon*100)/60;
    g_lon=d_lon+m_lon;
    
    // alt
    g_alt = data->altitude;

    printf("lat=%lf, lon=%lf, alt=%lf\r\n", g_lat, g_lon, g_alt);
}

/****************************************************************************
モジュール名  ： GPS受信コールバック
関数名         ： onGpsZdaReceivedCallback
引数          ： なし
戻り値         ： GPS受信データ
処理概要        ： GPS受信処理
注意事項        ： なし
****************************************************************************/
void onGpsZdaReceivedCallback(SerialGPS::NMEA_ZDA* data)
{
    g_year      = data->year;
    g_month     = data->month;
    g_day       = data->day;
    g_hour      = (data->hour) + (data->hour_diff);
    g_min       = (data->min)  + (data->min_diff);
    g_sec       = data->sec;
    
    printf("TIME:%04d/%02d/%02d %02d:%02d:%02d\r\n",
        g_year,
        g_month,
        g_day,
        g_hour,
        g_min,
        g_sec);
} 
#endif //GPS_ENABLE