/*
 * Nucleo STM32F4(or L4) quadrature decoder, ADCs and DACs
 * with serial communication over the USB virtual serial port

 * Developed for Elliptec X15 piezoelectric motor control, on a L432KC board
 *
 * Using STM32's counter peripherals to interface rotary encoders.
 * Encoders are supported on F4xx's TIM1,2,3,4,5. TIM2 & TIM5 have 32bit count,
 * others 16bit.
 * Take into account that on F4xx Mbed uses TIM5 for system timer, SPI needs TIM1,
 * others are used for PWM.
 * Check your platform's PeripheralPins.c & PeripheralNames.h if you need
 * both PWM & encoders. This project does not use PWM.
 *
 * On L432KC, for example, only TIM2 has 32bit count, others have 16bit.
 * However, mbed has TIM2 assigned by default to the system ticker
 * For L432KC to work with TIM2 encoder input one needs to reassign
 * the system ticker from TIM2 to TIM7, for example,
 * in mbed-dev/targets/TARGET_STM/TARGET_STM32L4/TARGET_STM32L432xC/device/hal_tick.h
 *
 * Edit HAL_TIM_Encoder_MspInitFx(Lx).cpp to suit your mcu & board's available
 *  pinouts & pullups/downs.
 *
 * Thanks to:
 * David Lowe (for the quadrature encoder code)
 * https://developer.mbed.org/users/gregeric/code/Nucleo_Hello_Encoder/
 *
 * And Frederic Blanc
 * https://developer.mbed.org/users/fblanc/code/AnalogIn_Diff/
 *
 * And Eric Lewiston /  STM32L4xx_HAL_Driver
 * https://developer.mbed.org/users/EricLew/code/STM32L4xx_HAL_Driver/docs/tip/group__ADC__LL__EF__Configuration__Channels.html
 *
 * References:
 * http://www.st.com/resource/en/datasheet/stm32l432kc.pdf
 * https://developer.mbed.org/platforms/ST-Nucleo-L432KC/
 * http://www.st.com/web/en/resource/technical/document/application_note/DM00042534.pdf
 * http://www.st.com/web/en/resource/technical/document/datasheet/DM00102166.pdf
 *
 * Tonny-Leonard Farauanu, 2017
 */

#include "mbed.h"
#include "Encoder.h"

//Defining the timer and its coresponding encoder
TIM_HandleTypeDef timer2;
TIM_Encoder_InitTypeDef encoder1;

//Timer to be used for the speed computation function
Timer timer1;

//The input for the encoder's index channel
InterruptIn event(PA_8);

//LED1 to signal USB serial RX interrupt reading
DigitalOut led1(LED1);

//LED2 to signal the encoder's index impulses
//only for feedback, to calibrate the position
//of the AVAGO encoder with respect to the shaft
DigitalOut led2(PB_4);

//Relay port to power on and off the Digitally Controlled Power Source (DCPS)
DigitalOut relay1(PB_5);

//Defining the ADCs
//For reading the Phase Comparator output
AnalogIn adc1(PA_3); //ADC1_IN8
//For reading the DCPS output
AnalogIn adc2(PA_4); //ADC1_IN9
//For reading the current value from the shunt monitor
AnalogIn adc3(PA_6); //ADC1_IN11

//Defining the DAC for the input of DCPS
//(DCPS - Digitally Controlled Power Source)
//DAC1_OUT2 on pin PA_5 is at least twices as fast as DAC1_OUT1 on pin PA_4
AnalogOut dac1(PA_5); // DAC1_OUT2

//Defining the serial object to be used for communicating with Raspi
Serial raspi(USBTX, USBRX);

//Counter for the absolute position from start
int32_t count1 = 0;
//Counter for the index passes
volatile int32_t count2 = 0;
//Records the wheel offset position to the index
volatile int32_t count3 = 0;
//Counter recording absolute position at last index pulse
volatile int32_t count4 = 0;
//Used to filter the first index pulse, so that the offset position
//to the index may be recorded
volatile bool adjustOffset = false;

//Last data written to dac1 (using subunitary values)
float dac_val = 0.0;

//Array with adc1, adc2 & adc3 correction addition,
//experimentally determined (hardware offset voltage dependent)
const float adc_corr[] = { 0, 0.022f, 0.022f, 0.022f};

//Enable ADC reading and printing to the serial interface
bool adc_en = true;

//Enables speed computing and printing to the serial interface
bool speed_en = true;

//Enable position reading and printing to the serial interface
bool pos_en = true;

//Enable alternative position computation and printing to the serial interface,
//relative to the encoder index pulses counter
bool posIndex_en = true;

//Function invoked by the encoder's index interrupt pulses
//It counts the index pulses, incrementing or decrementing the number,
//and registers the offset position of the wheel at start in count3;
//The index counter is used to correct possible reading errors of the
//encoder's counter
//led2 is used for visual feedback.
void atint()
{
    count4 =__HAL_TIM_GET_COUNTER(&timer2);
    if (__HAL_TIM_IS_TIM_COUNTING_DOWN(&timer2)) {
        if (count2 == 0 && adjustOffset == false) { //catch first index pulse
            count3 = count4;
            adjustOffset = true;
        }
        count2--;
        led2 =!led2;
    } else {
        if (count2 == 0 && adjustOffset == false) { //catch first index pulse
            count3 = count4;
            adjustOffset = true;
        }
        count2++;
        led2 =!led2;
    }
}

//Function for adaptive speed computation
float speedRead()
{
    //counter for the waiting time window
    uint16_t i = 0;
    //sample time for speed computation
    uint32_t deltaT;

    //Computer speed
    float speed;

    //First and second registered position
    int32_t pos1;
    int32_t pos2;
    //position difference to be used for speed computation
    int32_t pos;
    //Direction of rotation, read from the encoder, for the first
    //and the second registered positions
    int8_t sens1;
    int8_t sens2;

    timer1.start();
    pos1=__HAL_TIM_GET_COUNTER(&timer2);
    sens1 = __HAL_TIM_IS_TIM_COUNTING_DOWN(&timer2);
    //Minumum waiting time for this project would be 100 microseconds,
    //for 10 kHz quadrature frequency. We will set it 100 times longer
    //wait_ms(10);
    // Avoiding wait(), since it uses interrupts and timing here
    // is not critical. This takes 10 ms, if not interrupted
    for (uint32_t j=0; j<199950; j++) {}
    pos2=__HAL_TIM_GET_COUNTER(&timer2);
    sens2 = __HAL_TIM_IS_TIM_COUNTING_DOWN(&timer2);

    //The speed computation method adapts to slow/fast speeds, in order to
    //optimize the rapport between computation precision and time windows size
    while (pos2 == pos1 && i<9) { // The accepted max delay time is 0.1 seconds
        //wait_ms(10);
        // Avoiding wait(), since it uses interrupts and timing here
        // is not critical. This takes 10 ms, if not interrupted
        for (uint32_t j=0; j<199950; j++) {}
        i++;
        pos2=__HAL_TIM_GET_COUNTER(&timer2);
        sens2 = __HAL_TIM_IS_TIM_COUNTING_DOWN(&timer2);
    }
    pos2=__HAL_TIM_GET_COUNTER(&timer2);
    sens2 = __HAL_TIM_IS_TIM_COUNTING_DOWN(&timer2);
    timer1.stop();
    deltaT = timer1.read_us();
    timer1.reset();
    if (sens1 == sens2) {
        pos = pos2 - pos1;
    } else {
        printf("E:Speed computation error, change of direction between readings!\n");
    }

    if (deltaT > 0) {
        //speed computation in rot/s
        speed = ((float) pos)*125.f/((float) deltaT); // (pulses/us)*1000000/8000 -> rot/s
    } else {
        printf("E:Error, time interval not greater than zero, speed not calculated!\n");
    }
    return speed;
}

//Function attached to the serial RX interrupt event, it parses
//the serial messages and invoques the corresponding commands
void readData(void)
{
    led1 = 1;
    char message[15];
    if (raspi.readable()) {
        // Signalling the beginning of serial read
        int i = 0;
        bool rx = true;
        while(rx && i<14) {
            message[i] = raspi.getc();
            if (message[i] == '\r') {
                message[i] = NULL;
                rx = false;
            }
            i++;
        }
        //Message received printed for debugging
        //printf("M:%d %s\n", strlen(message), message);
        int p1 = 0;
        if (strcmp(message, "adcOn") == 0) {
            adc_en = true;
            printf("M:ADC true\n");
        } else if (strcmp(message, "adcOff") == 0) {
            adc_en = false;
            printf("M:ADC false\n");
        } else if (p1=strstr(message, "v=") != NULL) {
            //Writing the dac1 value read from serial
            //The DCPS has 1V offset, so we have to remove it
            dac_val = (atof(message+p1+1)-1.f)/15.51;
            dac1.write(dac_val);
        } else if (strcmp(message, "reset") == 0) {
            //encoder related counters reset command
            TIM2->CNT = 0x0000;
            count1 = 0;
            count2 = 0;
            count3 = 0;
            count4 = 0;
            adjustOffset = false;
            printf("M:Encoder counters reset!\n");
        } else if (strcmp(message, "powerOn") == 0) {
            //command to power on the DCPS
            relay1.write(1);
            printf("M:DCPS on\n");
        } else if (strcmp(message, "powerOff") == 0) {
            //command to power off the DCPS
            relay1.write(0);
            printf("M:DCPS off\n");
        } else if (strcmp(message, "posOn") == 0) {
            //command to enable the encoder position notification
            pos_en = true;
            printf("M:Position notification enabled\n");
        } else if (strcmp(message, "posOff") == 0) {
            //command to disable the encoder position notification
            pos_en = false;
            printf("M:Position notification disabled\n");
        } else if (strcmp(message, "posIndexOn") == 0) {
            //command to enable the index related encoder position notification
            posIndex_en = true;
            printf("M:Index related position notification enabled\n");
        } else if (strcmp(message, "posIndexOff") == 0) {
            //command to disable the index related encoder position notification
            posIndex_en = false;
            printf("M:Index related position notification disabled\n");
        } else if (strcmp(message, "speedOn") == 0) {
            //command to enable speed computation and notification
            speed_en = true;
            printf("M:Speed enabled\n");
        } else if (strcmp(message, "speedOff") == 0) {
            //command to disable speed computation and notification
            speed_en = false;
            printf("M:Speed disabled\n");
        }
    }
    
    //Signalling the end of searial read
    led1 = 0;
}

//Function to compute the average of 1000 ADC readings
//Current intensity measurement should be as precise
//as possible, the time delay introduced my multiplying
//the readings is tolerable for this project
float ADC_read(AnalogIn adc)
{
    float Voltage;
    float Voltage_total = 0.0;

    // Voltage is summed then averaged
    for (int i=0; i<1000; i++) { // do 1000 readings
        Voltage = adc.read();
        Voltage_total = Voltage_total+ Voltage;
    }
    Voltage = Voltage_total/1000.f;
    return Voltage;
}

//The main function
int main()
{
    //Make sure the DCPS is off
    relay1.write(0);
    //Set the DCPS output to (0.32x3.3)x4.7+1 = ~6V
    //These parameters are set for DCPS on load
    dac1.write(0.32);

    //counting on both A&B inputs (A at PA0, B at PA1), 4 ticks per cycle,
    //full 32-bit count
    //For L432KC to work with TIM2 one needs to reassign the system ticker
    //from TIM2 to TIM7 in TARGET/../device/hal_tick.h\
    //Initialise encoder
    EncoderInit(&encoder1, &timer2, TIM2, 0xffffffff, TIM_ENCODERMODE_TI12);

    //Define function to call for interrupt event rise
    //This is triggered by the encoder's index pulses
    //and it resets the encoder counter
    event.rise(&atint);

    //Set serial baud rate
    raspi.baud(921600);

    //Attach functin to call for serial interrupt event
    //The wait() function ensures that not false serial RX
    //interrupt is triggered at power up, because of initial
    //serial port install by the Raspi OS
    wait(1);
    raspi.attach(&readData, Serial::RxIrq);

    //Message to mark the initialisation of the program
    printf("M:STM HAL encoder with ADC and DAC\n");
    printf("M:Running at %u MHz\n", HAL_RCC_GetSysClockFreq()/1000000);

    //The main loop
    while(1) {

        //Variable for the direction of the counter
        int8_t dir1;

        //Prints the timestamp in miliseconds
        printf("T:%u", us_ticker_read()/1000);

        if (pos_en) {
            //It gets the position and the direction of the encoder
            count1=__HAL_TIM_GET_COUNTER(&timer2);
            dir1 = __HAL_TIM_IS_TIM_COUNTING_DOWN(&timer2);
            printf("P:%ldD:%sC:%d", count1, dir1==0 ? "+":"-", count2);
            if (posIndex_en) {
                if (count2 > 1) {
                    printf("Pi:%ld", (count1-count4) + (count2-1)*8000 + count3);
                } else if (count2 < -1) {
                    printf("Pi:%ld", (count1-count4) + (count2+1)*8000 + count3);
                } else {
                    printf("Pi:%ld", (count1-count4) + count3);
                }
            }
        }

        if (speed_en) {
            //Print speed in rot/s
            printf("S:%.3f%", speedRead());
        }

        if (adc_en) {
            //Print phase shift in Degrees, DCPS output voltage in Volts
            // and current intensity in Ampers
            // The dividing factors are hardware dependant
            printf("F:%3.3fU:%3.3fI:%3.3f", (3.3f*ADC_read(adc1)+adc_corr[1])*125.62f,
                   (3.3f*ADC_read(adc2)+adc_corr[2])/0.207f, 
                    3.3f*ADC_read(adc3)+adc_corr[3]);
        }
        printf("\n");

        wait(0.005);
    }

}