/* mbed Microcontroller Library
 * Copyright (c) 2006-2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include <cstdlib>
#include <ctime>
#include <math.h>
#include "mbed.h"
#include "ble/BLE.h"
#include "ble/DiscoveredCharacteristic.h"
#include "ble/DiscoveredService.h"

BLE        ble;
DigitalOut led(P0_19,1);
DigitalOut myled(P0_8);
DigitalOut neo(P0_6,1);
DigitalOut pin4(P0_4);
DigitalOut pin5(P0_5);
DigitalOut pin15(P0_15);
DigitalOut pin29(P0_29);

static const unsigned NUM_ROBOTS = 8;
static const unsigned CLASS = 0;

uint16_t customServiceUUID  = 0xA000;
uint16_t readwriteCharUUID  = 0xA001;
static const uint16_t secure_code[] = {0xABCD};

uint8_t info[NUM_ROBOTS]= {0};
uint8_t x[22] = {1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22};
float y[22];

ReadWriteArrayGattCharacteristic<uint8_t, NUM_ROBOTS> readwriteChar(readwriteCharUUID, info);

GattCharacteristic *characteristics[] = {&readwriteChar};

GattService        customService(customServiceUUID, characteristics, sizeof(characteristics) / sizeof(GattCharacteristic *));

DiscoveredCharacteristic characteristics_discovered;

uint8_t com_p;
bool movingTrigger = false;
uint8_t info_t[NUM_ROBOTS]= {0,};
uint8_t point;

float IRRead(PinName Pin) {
        AnalogIn AIn(Pin);
        float IRIn;
        IRIn=AIn;
        return(IRIn);
}

void moveRandom(void)
{
    uint8_t move_r1 = static_cast <int> (rand())%2; //0 or 1
    float move_r2 = 3 * static_cast <float> (rand()) / static_cast <float> (RAND_MAX); // 0~3

    if( move_r1 > 0.5) {
        move_r1 = 1;
    } else {
        move_r1 = 0;
    }

    if(move_r1==0) {
        pin4 = 1;
        pin5 = 1;
        pin15 = 0;
        pin29 = 0;
    } else {
        pin4 = 0;
        pin5 = 0;
        pin15 = 1;
        pin29 = 1;
    }
    wait(move_r2);

    pin4 = 1;
    pin5 = 1;
    pin15 = 1;
    pin29 = 1;
    float move_r3 = 3 * static_cast <float> (rand()) / static_cast <float> (RAND_MAX); // 0~3
    wait(move_r3);
}

void RGB_Show(uint8_t r, uint8_t g, uint8_t b)
{
    uint8_t rgb[3] = {g, r, b};
    uint8_t *p = rgb;
    uint8_t *end = p + 3;
    while (p < end) {
        uint8_t pix = *p++;
        for (uint8_t mask = 0x80; mask; mask >>= 1) {
            if (pix & mask) {
                // T1H 760ns
                NRF_GPIO->OUTSET = (1UL << 8);
                NRF_GPIO->OUTSET = (1UL << 8);
                NRF_GPIO->OUTSET = (1UL << 8);
                // T1L 660ns
                NRF_GPIO->OUTCLR = (1UL << 8);
            } else {
                // T0H 380ns
                NRF_GPIO->OUTSET = (1UL << 8);
                // T0L 840ns
                NRF_GPIO->OUTCLR = (1UL << 8);
                NRF_GPIO->OUTCLR = (1UL << 8);
                NRF_GPIO->OUTCLR = (1UL << 8);
                NRF_GPIO->OUTCLR = (1UL << 8);
            }
        }
    }
    NRF_GPIO->OUTCLR = (1UL << 8);
    wait_us(100);
}

void periodicCallback(void)
{
    led=!led;
    point = 0;
    for(int i=0; i<NUM_ROBOTS; i++) {
        point+=info[i];
    }
    if(point==0) {
        RGB_Show(0x00,0x00,0xff);
    } else if(point==1) {
        RGB_Show(0xff,0xff,0xff);
    } else if(point==2) {
        RGB_Show(0x00,0xff,0x00);
    } else if(point==3) {
        RGB_Show(0x00,0xff,0xff);
    } else if(point==4) {
        RGB_Show(0xff,0x00,0x00);
    } else if(point==5) {
        RGB_Show(0xff,0x00,0xff);
    } else if(point==6) {
        RGB_Show(0xff,0xff,0x00);
    } else if(point==7) {
        RGB_Show(0xff,0xff,0xff);
    } else if(point==8) {
        RGB_Show(0x00,0x00,0xff);
    }
}

void advertisementCallback(const Gap::AdvertisementCallbackParams_t *params)
{
    float ran=static_cast <float> (rand()) / static_cast <float> (RAND_MAX); // 0~1 probability
    printf("com_p : %f OOOOOOOOOOOOOO ran : %f \r\n",y[com_p],ran);
    if( (params->rssi > -48) && (y[com_p] > ran) && (params->advertisingData[1] == 0xAF) ) {
        ble.gap().connect(params->peerAddr, Gap::ADDR_TYPE_RANDOM_STATIC, NULL, NULL);
    } else {
        movingTrigger = true;
    }
}

void characteristicDiscoveryCallback(const DiscoveredCharacteristic *characteristicP)
{
    if (characteristicP->getUUID() == 0xA001) { /* !ALERT! Alter this filter to suit your device. */
        characteristics_discovered        = *characteristicP;
        printf("characteristic discovered\r\n");
        characteristics_discovered.read();
    }
}

void serviceDiscoveryCallback(const DiscoveredService *service)
{
}

void connectionCallback(const Gap::ConnectionCallbackParams_t *params)
{
    info[7] += 1;
    ble.gap().stopAdvertising();
    ble.gap().stopScan();
    
    printf("adv peerAddr[%02x %02x %02x %02x %02x %02x]\r\n",
                   params->peerAddr[5], params->peerAddr[4], params->peerAddr[3], params->peerAddr[2], params->peerAddr[1], params->peerAddr[0]);
    movingTrigger = false;
    printf("connected\r\n");
    if (params->role == Gap::CENTRAL) {
        printf("CENTRAL\r\n");
        ble.gattClient().launchServiceDiscovery(params->handle, serviceDiscoveryCallback, characteristicDiscoveryCallback, 0xA000, 0xA001);
    } else {
        printf("PERIPHERAL\r\n");
    }
}

void onDataReadCallback(const GattReadCallbackParams *response)
{
    if (response->handle == characteristics_discovered.getValueHandle()) {
        printf("info_t : ");
        for(int i=0; i < NUM_ROBOTS; i++) {
            info_t[i] = response->data[i];
            printf("%u \t",info_t[i]);
        }
        printf("\r\n");
        
        if(info[0] == 0) {
            if(response->data[0] == 0) {
                info[1] += 1; // +1 point
                info[2] = 0; // connection with normal
                info_t[1] += 1;
                info_t[2] = 0;
                characteristics_discovered.write(NUM_ROBOTS,info_t);
            } else {
                info[3] += 1; // -1 point
                info[2] = 1; // connection with malicious
                characteristics_discovered.write(NUM_ROBOTS,info_t);
            }
        } else {
            if(response->data[0] == 0) {
                info_t[3] += 1;
                info_t[2] = 1;
                characteristics_discovered.write(NUM_ROBOTS,info_t);
                printf("aaa\r\n");
            } else {
                characteristics_discovered.write(NUM_ROBOTS,info_t);
            }
        }
    }
}

void onDataWriteCallback(const GattWriteCallbackParams *params)
{
    printf("bbb\r\n");
    ble.updateCharacteristicValue(readwriteChar.getValueHandle(),info,NUM_ROBOTS);
}

void onDataWrittenCallback(const GattWriteCallbackParams *params)
{
    printf("ccc\r\n");
    for(int i=0; i<NUM_ROBOTS-1; i++) {
        info[i] = params->data[i];
    }
    printf("writecallbackparam : ");
    for(int i=0; i<NUM_ROBOTS; i++) {
        printf("%u \t",params->data[i]);
    }
    printf("\r\n");
    ble.gattServer().write(readwriteChar.getValueHandle(),info,NUM_ROBOTS);
    ble.updateCharacteristicValue(readwriteChar.getValueHandle(),params->data,params->len);
    printf("data is written well\r\n");
    
    ble.gap().disconnect(params->connHandle,Gap::REMOTE_USER_TERMINATED_CONNECTION);
}

void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason)
{
    printf("Current info : ");
    for(int i=0; i<NUM_ROBOTS; i++) {
        printf("%u \t",info[i]);
    }
    printf("\r\n");
    printf("disconnected\r\n");
    if(info[2] == 0) {
        if(com_p == 21) {
            com_p = 21;
        } else {
            com_p += 1;
        }
    } else if(info[2] == 1) {
        if(com_p == 0) {
            com_p = 0;
        } else {
            com_p -= 1;
        }
    }
    ble.gap().startAdvertising();
    ble.gap().startScan(advertisementCallback);
}

int main(void)
{   
    srand (static_cast <unsigned> (time(0))); //seeding for random
    for(int i=0; i<22;i++){
        y[i] = 0.1 + 0.9*(1-exp(-0.2*x[i]));
        printf("%f ",y[i]);
    }
    printf("\r\n");
    info[0] = CLASS; // normal : 0 & malicious : 1
    info[1] = 0; // point
    com_p = 11;
    
    srand (static_cast <unsigned> (time(0)));
    
    Ticker ticker;
    
    for(int i=0; i<NUM_ROBOTS; i++) {
        printf("%u \t",info[i]);
    }
    printf("\r\n");
    
    printf("communication for connection is started\r\n");
    ble.init();
    ble.gap().onConnection(connectionCallback);
    ble.gap().onDisconnection(disconnectionCallback);

    ble.gattClient().onDataRead(onDataReadCallback);
    ble.gattClient().onDataWrite(onDataWriteCallback);
    ble.gattServer().onDataWritten(onDataWrittenCallback);
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::RABOT_REDBEAR_BLE_NANO, (uint8_t *)secure_code, sizeof(secure_code));
    ble.gap().setAdvertisingInterval(59);

    ble.addService(customService);
    
    ble.gap().startAdvertising();
    ble.gap().setScanParams(125 /* scan interval */, 79 /* scan window */);
    ble.gap().startScan(advertisementCallback);

    while (true) {
        /*
                if(connectionState == false) {
                    ble.gap().startAdvertising();
                    ble.gap().startScan(advertisementCallback);
                } else {
                    ble.gap().stopAdvertising();
                    ble.gap().stopScan();
                }
        */
        ble.waitForEvent();
        if(movingTrigger == true) {
            moveRandom();
        } else {
            pin4 = 0;
            pin5 = 0;
            pin15 = 0;
            pin29 = 0;
        }
        if(info[1] == 20) {
            printf("Game over");
            break;
        }
    }
    ble.shutdown();
    pin4 = 0;
    pin5 = 0;
    pin15 = 0;
    pin29 = 0;
}