#include "mbed.h"
#include "main.h"
#include "sx1272-hal.h"
#include "debug.h"

/* Set this flag to '1' to display debug messages on the console */
#define DEBUG_MESSAGE   1

/* Set this flag to '1' to use the LoRa modulation or to '0' to use FSK modulation */
#define USE_MODEM_LORA  1
#define USE_MODEM_FSK   !USE_MODEM_LORA

#define RF_FREQUENCY                                    868000000 // Hz
#define TX_OUTPUT_POWER                                 14        // 14 dBm

#if USE_MODEM_LORA == 1

#define LORA_BANDWIDTH                              2         // [0: 125 kHz,
//  1: 250 kHz,
//  2: 500 kHz,
//  3: Reserved]
#define LORA_SPREADING_FACTOR                       7         // [SF7..SF12]
#define LORA_CODINGRATE                             1         // [1: 4/5,
//  2: 4/6,
//  3: 4/7,
//  4: 4/8]
#define LORA_PREAMBLE_LENGTH                        8         // Same for Tx and Rx
#define LORA_SYMBOL_TIMEOUT                         5         // Symbols
#define LORA_FIX_LENGTH_PAYLOAD_ON                  false
#define LORA_FHSS_ENABLED                           false
#define LORA_NB_SYMB_HOP                            4
#define LORA_IQ_INVERSION_ON                        false
#define LORA_CRC_ENABLED                            true

#elif USE_MODEM_FSK == 1

#define FSK_FDEV                                    25000     // Hz
#define FSK_DATARATE                                19200     // bps
#define FSK_BANDWIDTH                               50000     // Hz
#define FSK_AFC_BANDWIDTH                           83333     // Hz
#define FSK_PREAMBLE_LENGTH                         5         // Same for Tx and Rx
#define FSK_FIX_LENGTH_PAYLOAD_ON                   false
#define FSK_CRC_ENABLED                             true

#else
#error "Please define a modem in the compiler options."
#endif

#define RX_TIMEOUT_VALUE                                3500000   // in us
#define BUFFER_SIZE                                     32        // Define the payload size here

#if( defined ( TARGET_KL25Z ) || defined ( TARGET_LPC11U6X ) )
DigitalOut led(LED2);
#else
DigitalOut led(LED1);
#endif
typedef unsigned char byte;
unsigned short crc16(byte byteArray[], int length)
{
    unsigned char x;
    unsigned short crc = 0xFFFF;
    for (int i = 0; i < length; i++) {
        x = crc >> 8 ^ byteArray[i];
        x ^= x >> 4;
        crc = (crc << 8) ^ ((unsigned short)(x << 12)) ^ ((unsigned short)(x << 5)) ^ ((unsigned short)x);
    }
    return crc;
}

void float2Bytes(float val, byte* bytes_array)
{
    union {
        float float_variable;
        byte temp_array[4];
    } u;
    u.float_variable = val;
    memcpy(bytes_array, u.temp_array, 4);
}

class Position
{
public:
    bool north;
    bool south;
    bool east;
    bool west;
    short degLon;
    short degLat;
    float minutesLong;
    float minutesLat;
    float secondesLong;
    float secondesLat;

    Position(bool north, bool south, bool east, bool west, short degLon, short degLat, float minutesLong, float minutesLat, float secondesLong, float secondesLat) {
        this->north = north;
        this->south = south;
        this->east = east;
        this->west = west;
        this->degLon = degLon;
        this->degLat = degLat;
        this->minutesLong = minutesLong;
        this->minutesLat = minutesLat;
        this->secondesLong = secondesLong;
        this->secondesLat = secondesLat;
    }

    Position(byte* posBytes) {
        this->north = (posBytes[0] & 1);
        this->south = (posBytes[0] & 2) >> 1;
        this->east = (posBytes[0] & 4) >> 2;
        this->west = (posBytes[0] & 8) >> 3;
        this->degLon = posBytes[1] * 256 + posBytes[2];
        this->degLat = posBytes[3] * 256 + posBytes[4];
        this->minutesLong = *(float *)(posBytes + 5);
        this->minutesLat = *(float *)(posBytes + 9);
        this->secondesLong = *(float *)(posBytes + 13);
        this->secondesLat = *(float *)(posBytes + 17);
    }

    byte* GetBytes() {
        byte* posByte = (byte*)malloc(21 * sizeof(byte));
        posByte[0] = 0;
        if (this->north) {
            posByte[0] = posByte[0] | 1;
        }
        if (this->south) {
            posByte[0] = posByte[0] | 2;
        }
        if (this->east) {
            posByte[0] = posByte[0] | 4;
        }
        if (this->west) {
            posByte[0] = posByte[0] | 8;
        }
        posByte[1] = (this->degLon & 0xFF00) >> 8; //Bits de poids fort du short
        posByte[2] = (this->degLon & 0x00FF); //Bits de poids faible
        posByte[3] = (this->degLat & 0xFF00) >> 8; //Bits de poids fort du short
        posByte[4] = (this->degLat & 0x00FF); //Bits de poids faible
        float2Bytes(this->minutesLong, posByte + 5);
        float2Bytes(this->minutesLat, posByte + 9);
        float2Bytes(this->secondesLong, posByte + 13);
        float2Bytes(this->secondesLat, posByte + 17);
        return posByte;
    }
};
class Date
{
public:
    byte jour;
    byte mois;
    byte annee;
    byte heure;
    byte minutes;
    byte secondes;

    byte* GetBytes() {
        byte* dateByte = (byte*)malloc(6 * sizeof(byte));
        dateByte[0] = this->jour;
        dateByte[1] = this->mois;
        dateByte[2] = this->annee;
        dateByte[3] = this->heure;
        dateByte[4] = this->minutes;
        dateByte[5] = this->secondes;
        return dateByte;
    }

    Date(byte jour, byte mois, byte annee, byte heure, byte minutes, byte secondes) {
        this->jour = jour;
        this->mois = mois;
        this->annee = annee;
        this->heure = heure;
        this->minutes = minutes;
        this->secondes = secondes;
    }

    Date(byte* dateBytes) {
        this->jour = dateBytes[0];
        this->mois = dateBytes[1];
        this->annee = dateBytes[2];
        this->heure = dateBytes[3];
        this->minutes = dateBytes[4];
        this->secondes = dateBytes[5];
    }

};

class Content
{
public:
    short IDLapin;
    Position* positionLapin;
    Date* dateLapin;

    byte* GetBytes() {
        byte* contByte = (byte*)malloc(29 * sizeof(byte));
        contByte[0] = (this->IDLapin & 0xFF00) << 8;
        contByte[1] = (this->IDLapin & 0x00FF);
        byte* dateBytes = this->dateLapin->GetBytes();
        byte* positionBytes = this->positionLapin->GetBytes();
        memcpy(contByte + 2, dateBytes, 6);
        memcpy(contByte + 8, positionBytes, 21);
        free(positionBytes);
        free(dateBytes);
        return contByte;
    }

    Content(short IDLapin, Position* positionLapin, Date* dateLapin) {
        this->IDLapin = IDLapin;
        this->positionLapin = positionLapin;
        this->dateLapin = dateLapin;
    }

    Content(byte* contentBytes) {
        this->IDLapin = contentBytes[0] * 256 + contentBytes[1];
        byte* positionBytes = (byte*)malloc(21 * sizeof(byte));
        byte* dateBytes = (byte*)malloc(6 * sizeof(byte));
        memcpy(dateBytes, contentBytes + 2, 6);
        memcpy(positionBytes, contentBytes + 8, 21);
        this->positionLapin = (Position*)(new Position(positionBytes));
        this->dateLapin =  (Date*)(new Date(dateBytes));
        free(dateBytes);
        free(positionBytes);
    }
};

byte* CreateMessageBytes(Content* messageContent)
{
    byte* messByte = (byte*)malloc(32 * sizeof(byte));
    messByte[0] = 20;
    byte* contentBytes = messageContent->GetBytes();
    unsigned short gotHash = crc16(contentBytes, 29);
    messByte[1] = (gotHash & 0xFF00) >> 8;
    messByte[2] = (gotHash & 0x00FF);
    memcpy(messByte + 3, contentBytes, 29);
    free(contentBytes);
    return messByte;
}

enum MessageCheckResult { CORRECT = 1, CORRUPTED = 0, NOTFORUS = -1 };
MessageCheckResult IsCorrectMessage(byte* bytesReceived, int mLength)
{
    if (mLength != 32) {
        return NOTFORUS;
    }
    if (bytesReceived[0] != 20) {
        return NOTFORUS;
    }
    if (crc16(bytesReceived + 3, 29) != (bytesReceived[1] * 256 + bytesReceived[2])) {
        return CORRUPTED;
    }
    return CORRECT;
}


/*
 *  Global variables declarations
 */
typedef enum {
    LOWPOWER = 0,
    IDLE,

    RX,
    RX_TIMEOUT,
    RX_ERROR,

    TX,
    TX_TIMEOUT,

    CAD,
    CAD_DONE
} AppStates_t;

volatile AppStates_t State = LOWPOWER;

/*!
 * Radio events function pointer
 */
static RadioEvents_t RadioEvents;

/*
 *  Global variables declarations
 */
SX1272MB2xAS Radio( NULL );

uint16_t BufferSize = BUFFER_SIZE;
uint8_t Buffer[BUFFER_SIZE];

int16_t RssiValue = 0.0;
int8_t SnrValue = 0.0;

int main()
{
    // Initialize Radio driver
    RadioEvents.TxDone = OnTxDone;
    RadioEvents.RxDone = OnRxDone;
    RadioEvents.RxError = OnRxError;
    RadioEvents.TxTimeout = OnTxTimeout;
    RadioEvents.RxTimeout = OnRxTimeout;
    Radio.Init( &RadioEvents );

    // verify the connection with the board
    while( Radio.Read( REG_VERSION ) == 0x00  ) {
        debug( "Radio could not be detected!\n\r", NULL );
        wait( 1 );
    }

    debug_if( ( DEBUG_MESSAGE & ( Radio.DetectBoardType( ) == SX1272MB2XAS ) ) , "\n\r > Board Type: SX1272MB2xAS < \n\r" );

    Radio.SetChannel( RF_FREQUENCY );

#if USE_MODEM_LORA == 1

    debug_if( LORA_FHSS_ENABLED, "\n\n\r             > LORA FHSS Mode < \n\n\r");
    debug_if( !LORA_FHSS_ENABLED, "\n\n\r             > LORA Mode < \n\n\r");

    Radio.SetTxConfig( MODEM_LORA, TX_OUTPUT_POWER, 0, LORA_BANDWIDTH,
                       LORA_SPREADING_FACTOR, LORA_CODINGRATE,
                       LORA_PREAMBLE_LENGTH, LORA_FIX_LENGTH_PAYLOAD_ON,
                       LORA_CRC_ENABLED, LORA_FHSS_ENABLED, LORA_NB_SYMB_HOP,
                       LORA_IQ_INVERSION_ON, 2000000 );

    Radio.SetRxConfig( MODEM_LORA, LORA_BANDWIDTH, LORA_SPREADING_FACTOR,
                       LORA_CODINGRATE, 0, LORA_PREAMBLE_LENGTH,
                       LORA_SYMBOL_TIMEOUT, LORA_FIX_LENGTH_PAYLOAD_ON, 0,
                       LORA_CRC_ENABLED, LORA_FHSS_ENABLED, LORA_NB_SYMB_HOP,
                       LORA_IQ_INVERSION_ON, true );

#elif USE_MODEM_FSK == 1

    debug("\n\n\r              > FSK Mode < \n\n\r");
    Radio.SetTxConfig( MODEM_FSK, TX_OUTPUT_POWER, FSK_FDEV, 0,
                       FSK_DATARATE, 0,
                       FSK_PREAMBLE_LENGTH, FSK_FIX_LENGTH_PAYLOAD_ON,
                       FSK_CRC_ENABLED, 0, 0, 0, 2000000 );

    Radio.SetRxConfig( MODEM_FSK, FSK_BANDWIDTH, FSK_DATARATE,
                       0, FSK_AFC_BANDWIDTH, FSK_PREAMBLE_LENGTH,
                       0, FSK_FIX_LENGTH_PAYLOAD_ON, 0, FSK_CRC_ENABLED,
                       0, 0, false, true );

#else

#error "Please define a modem in the compiler options."

#endif

    Radio.Rx(RX_TIMEOUT_VALUE  );

    while( 1 ) {
       // Radio.Rx( 100*RX_TIMEOUT_VALUE  );
        //SendStr("Bonjour Baudouin");
    }

}

void SendStr(char* str)
{
    int len = strlen(str);
    uint8_t MyBuffer[len];
    strcpy( ( char* )MyBuffer, str );
    debug(str);
    Radio.Send(MyBuffer, len);
}

void OnTxDone( void )
{
    Radio.Sleep();
    State = TX;
    debug_if( DEBUG_MESSAGE, "\n\r> Message transmis\n\r" );
}

void OnRxDone( uint8_t *payload, uint16_t size, int16_t rssi, int8_t snr)
{
    Radio.Sleep();
    BufferSize = size;
    memcpy( Buffer, payload, BufferSize );
    RssiValue = rssi;
    SnrValue = snr;
    State = RX;
    debug_if( DEBUG_MESSAGE, "\n\r> Message recu\n\r" );
    //debug("\r\n");
    //debug("SNR = %d\r\n",snr);
    //debug("RSSI = %d\r\n",rssi);
    if (IsCorrectMessage(Buffer,BufferSize) == 1){
        Content contentReceived = Content(Buffer+3);
        debug("%d %d %d %d %d %d", contentReceived.dateLapin->jour, contentReceived.dateLapin->mois,contentReceived.dateLapin->annee ,contentReceived.dateLapin->heure, contentReceived.dateLapin->minutes, contentReceived.dateLapin->secondes);
    }
    
    Radio.Rx(RX_TIMEOUT_VALUE  );
}

void OnTxTimeout( void )
{
    Radio.Sleep( );
    State = TX_TIMEOUT;
    debug_if( DEBUG_MESSAGE, "\n\r> Delai d'attente depasse\n\r" );
}

void OnRxTimeout( void )
{
    Radio.Sleep( );
    Buffer[ BufferSize ] = 0;
    State = RX_TIMEOUT;
    debug_if( DEBUG_MESSAGE, "> OnRxTimeout\n\r" );
    Radio.Rx(RX_TIMEOUT_VALUE  );
}

void OnRxError( void )
{
    Radio.Sleep( );
    State = RX_ERROR;
    debug_if( DEBUG_MESSAGE, "> OnRxError\n\r" );
    Radio.Rx( 10*RX_TIMEOUT_VALUE  );
}

