/*******************************************************************************************
 *
 * University of York Robot Lab Pi Swarm Robot Library
 *
 * (C) Dr James Hilder, Dept. Electronics & Computer Science, University of York
 * 
 * Version 0.6  February 2014
 *
 * Change notes:
 * 0.6  :  Added new IR sensor functions to improve efficiency (see manual for details)
 * 0.5  :  Initial release
 *
 * Designed for use with the Pi Swarm Board (enhanced MBED sensor board) v1.2
 *
 ******************************************************************************************/

#ifndef PISWARM_H
#define PISWARM_H

#include "mbed.h"
#include "platform.h"
#include "alpha433.h"
#include "main.h"

#ifdef MBED_RPC
#include "rpc.h"
#endif

// The following defines are user parameters and may be edited 
// SWARM_SIZE - size of swarm (+1).  Max 32.
// USE_COMMUNICATION_STACK - 1 to enable, 0 to bypass
// CALIBRATE_COLOURS - 1 to increase blue intensity, 0 to not
// CENTER_LED_BRIGHTNESS - Range 0 to 100
// OUTER_LED_BRIGHTNESS - Range 0 to 100
// PC_BAUD - PC Serial Baudrate.  Max 115200
// RF_ALLOW_COMMANDS - 1 to allow remote RF commands, 0 to block
// RF_USE_LEDS - 1 to use MBED LEDs to display RF status
// RF_USE_TDMA - 1 to use time-delayed responses to broadcast messages 
// RF_TDMA_TIME_PERIOD_US 15625 - Period of delay to use (per ID) in us.
// PISWARM_DEBUG - 1 to send general Pi Swarm debug information over USB serial interface
// RF_DEBUG - 1 to send communication stack debug information over USB serial interface
// RF_VERBOSE - 1 to send verbose RF debug information over USB serial interface
// START_RADIO_ON_BOOT - 1 to enable RF transceiver and comms stack on boot
// RF_FREQUENCY - Frequency of RF chip.  See datasheet for details (default 435000000)
// RF_DATARATE - Baud rate of the RF chip. Max 115200, default 57600.

#define SWARM_SIZE 32
#define USE_COMMUNICATION_STACK 1
#define CALIBRATE_COLOURS 1
#define CENTER_LED_BRIGHTNESS 30
#define OUTER_LED_BRIGHTNESS 50
#define PC_BAUD 115200
#define RF_ALLOW_COMMANDS 1
#define RF_USE_LEDS 1
#define RF_USE_TDMA 1
#define RF_TDMA_TIME_PERIOD_US 15625
#define PISWARM_DEBUG 1
#define RF_DEBUG 1
#define RF_VERBOSE 0
#define START_RADIO_ON_BOOT 1
#define RF_FREQUENCY 435000000
#define RF_DATARATE 57600

// The remaining defines give the addresses of hardware and 3-Pi instructions etc. and should not be changed
#define SEND_SIGNATURE 0x81
#define SEND_RAW_SENSOR_VALUES 0x86
#define SEND_TRIMPOT 0xB0
#define SEND_BATTERY_MILLIVOLTS 0xB1
#define DO_PLAY 0xB3
#define PI_CALIBRATE 0xB4
#define DO_CLEAR 0xB7
#define DO_PRINT 0xB8
#define DO_LCD_GOTO_XY 0xB9
#define LINE_SENSORS_RESET_CALIBRATION 0xB5
#define SEND_LINE_POSITION 0xB6
#define AUTO_CALIBRATE 0xBA
#define SET_PID 0xBB
#define STOP_PID 0xBC
#define M1_FORWARD 0xC1
#define M1_BACKWARD 0xC2
#define M2_FORWARD 0xC5
#define M2_BACKWARD 0xC6
#define EXPANSION_IC_ADDRESS 0x40
#define MAGNETOMETER_ADDRESS 0x1C
#define ADC_ADDRESS 0x46
#define EEPROM_ADDRESS 0XA0

class PiSwarm :  public Stream {

    // Public functions
public:

    // Create the Pi Swarm object connected to the default pins [see end of file]
    PiSwarm();
    
    
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////  
    // LED Functions
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    // Returns the RGB Colour of the Outer LED as an int (r<<16 g<<8 b)
    int get_oled_colour(void);
   
    // Returns the enable state of an single outer led 
    char get_oled_state(char oled);

    // Set colour of outer LEDs
    void set_oled_colour ( char red, char green, char blue );

    // Toggle the state of a single outer LED without affected the others
    void set_oled(char oled, char value);
   
    // Toggle the states of all 10 outer LEDs
    void set_oleds(char oled_0, char oled_1, char oled_2, char oled_3, char oled_4, char oled_5, char oled_6, char oled_7, char oled_8, char oled_9);
    
    // Disable all the LEDs and the dedicated power supply
    void turn_off_all_oleds(void);
     
    // Set brightness of outer LEDs (works by increasing total period for PWM)
    void set_oled_brightness ( char brightness );
     
    // Sends the message to the GPIO expander to update which OLEDs are active
    void activate_oleds(void);
    
    // Returns the RGB Colour of the Center LED as an int (r<<16 g<<8 b)
    int get_cled_colour(void);
 
    // Returns the enable state of the center LED 
    char get_cled_state(void);

    // Set colour of central LED
    void set_cled_colour ( char red, char green, char blue );
       
    // Enable or disable the center LED
    void enable_cled ( char enable );
    
    // Set brightness of central LED (works by increasing total period for PWM)
    void set_cled_brightness ( char brightness );

    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////  
    // IR Functions
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
 
    // Returns an approximate distance of any reflected obstacle in front of the given IR sensor (in millimeters, range 0 - 100)
    float read_reflected_ir_distance ( char index );
    
    // Returns the stored value of the reflected obstacle based on last call of either read_reflected_ir_distance or store_reflected_distances
    // Introduced in API 0.6
    float get_reflected_ir_distance ( char index );
    
    // Returns the stored value of the non-illuminated sensor based on last call of store_background_raw_ir_values
    // Introduced in API 0.6
    unsigned short get_background_raw_ir_value ( char index );
    
    // Returns the stored value of the illuminated sensor based on last call of store_illuminated_raw_ir_values
    // Introduced in API 0.6
    unsigned short get_illuminated_raw_ir_value ( char index );
  
    // Stores the reflected distances for all 8 IR sensors
    // Introduced in API 0.6
    void store_reflected_ir_distances ( void );
    
    // Stores the raw ADC values for all 8 IR sensors without enabling IR emitters
    // Introduced in API 0.6
    void store_background_raw_ir_values ( void );
    
    // Stores the raw ADC values for all 8 IR sensors with a 500us emitter pulse
    // Introduced in API 0.6
    void store_illuminated_raw_ir_values ( void );
    
    // Converts a background and illuminated value into a distance
    // Introduced in API 0.6 - used by read_reflected_ir_distance and store_reflected_ir_distances
    float calculate_reflected_distance ( unsigned short background_value, unsigned short illuminated_value );
    
    // Returns the illuminated raw ADC value for a given IR sensor (500us pulse)
    unsigned short read_illuminated_raw_ir_value ( char index );
    
    // Returns the raw ADC value for a given IR sensor
    unsigned short read_adc_value( char index );
    
    // Turns on or off the LDO outputs based on the current enable variables (enable_ir_ldo and enable_led_ldo)
    void enable_ldo_outputs ( void );
    
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////  
    // Sensor Functions
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    // Read the gyroscope yaw value.  Return value is degrees per second clockwise.
    float read_gyro ( void );
    
    // Read the X value from the accelerometer.  Return value is mps2.
    float read_accelerometer_x ( void );
    
    // Read the Y value from the accelerometer.  Return value is mps2.
    float read_accelerometer_y ( void );
    
    // Read the Z value from the accelerometer.  Return value is mps2.  Generally this will have g against it.
    float read_accelerometer_z ( void );
    
    // Read the I2C magnetometer and store values locally: use before get_magnetometer_x etc...
    char read_magnetometer ( void );
    
    // Return the magnetometer x value stored on last call of read_magnetometer()
    signed short get_magnetometer_x ( void );
    
    // Return the magnetometer y value stored on last call of read_magnetometer()
    signed short get_magnetometer_y ( void );
    
    // Return the magnetometer z value stored on last call of read_magnetometer()
    signed short get_magnetometer_z ( void );
    
    // Read temperature from the temperature IC; return is temperature in degrees Celsius
    float read_temperature ( void );
   
    // Read the value from the light sensor.  This will return a value ranging from 0 to a peak of around 100 in direct, bright light.
    float read_light_sensor ( void );
    
    // Read the raw line sensors into a 5-int array (range 0-2000)
    void read_raw_sensors ( int * raw_ls_array ); 
    
    // Returns the uptime of the system in seconds    
    float get_uptime (void);
    
    // Returns the battery voltage
    float battery(void);
    
    // Read the voltage of the user potentiometer on the 3-pi
    float pot_voltage(void);

    // Read the robot ID set using the DIP switches (these are read during setup and not updated).
    char get_id ( void );
    
    // Read the switches value from the last call of the ISR
    char get_switches ( void );
    
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////  
    // Motor and Actuator Functions
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    // Returns the last speed value set for the left motor
    float get_left_motor (void);
     
    // Returns the last speed value set for the right motor
    float get_right_motor (void);

    // Directly control the speed and direction of the left motor (range -1.0 to 1.0)
    void left_motor (float speed);

    // Directly control the speed and direction of the right motor (range -1.0 to 1.0)
    void right_motor (float speed);

    // Drive both motors forward as the same speed (range -1.0 to 1.0)
    void forward (float speed);

    // Drive both motors backward as the same speed (range -1.0 to 1.0)
    void backward (float speed);

    // Drive left motor backwards and right motor forwards at the same speed to turn on the spot (range -1.0 to 1.0)
    void left (float speed);

    // Drive left motor forward and right motor backwards at the same speed to turn on the spot (range -1.0 to 1.0)
    void right (float speed);

    // Stop both motors
    void stop (void);
    
    // Play a tune on the 3-pi buzzer (see notes at end of file for more details)
    void play_tune (char * tune, int length);
    
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////  
    // EEPROM Functions
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
   
    // Write single byte to EEPROM
    void write_eeprom_byte ( int address, char data );  
  
    // Write (length) bytes to EEPROM
    void write_eeprom_block ( int address, char length, char * data);
    
    // Read single byte from EEPROM
    char read_eeprom_byte ( int address );

    // Read next byte from EEPROM at current address 
    char read_next_eeprom_byte ( void ); 

    // Read (length) from EEPROM
    char read_eeprom_block ( int address, char length );
    
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////  
    // Display Functions
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
   
    // Locate the cursor on the 8x2 LCD [to position x (0-7), y (0-1)]
    void locate(int x, int y);

    // Clear the LCD
    void cls(void);

    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////  
    // Setup and Internal Functions
    // Generally these functions should not be called by user code
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    // Send a message of given length over the RF interface
    void send_rf_message(char* message, char length);
 
    // The default setup routine: called when the Pi Swarm is created
    void setup ( void );
       
    // Sets up the IO expansion IC using I2C commands; returns a zero if successful
    int setup_expansion_ic ( void );
    
    // Called after a delay from the IO expansion interrupt; delay helps debounce switches
    void interrupt_timeout_event ( void );

    // The interrupt handler for the IO expansion IC; calls the timeout event to debounce switches
    void expansion_interrupt_handler ( void );
    
    // Turns off interrupt LED after a delay to indicate switch has been pressed
    void led_timeout_event ( void );
    
    // Routine to calibrate gyro; by default run in setup, robot should be stationary to avoid errors (Returns 1 if successful, 0 if failed)
    char calibrate_gyro ( void );
    
    // Routine to calibrate accelerometer, by default run after successful gyro calibration, robot should be flat and stationary to avoid errors (Returns 1 if successful, 0 if failed)
    char calibrate_accelerometer ( void );

    // Routine to calibrate magnetometer (Returns 1 if successful, 0 if failed)    
    char calibrate_magnetometer ( void );
    
    // Send instructions to the I2C ADC (for IR measurement)
    char setup_adc( void );
    
    // Routine to initialise radio
    void setup_radio( void );
    
    // Blinks the calibration LED during calibration (turns off if calibration successful)
    void calibrate_ticker_routine ( void );
    
    // Used to debug/test the OLED array
    void test_oled(void);

    // Starts the system timer
    void start_system_timer(void);
    
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////  
    // Native m3Pi Functions
    // These should preserve functionality of (most) m3pi code but may not be fully functional on Pi Swarm
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    // Force a hardware reset of the 3-pi
    void reset (void);
    
    // Read the position of the detected line [using line-following IR]
    float line_position (void);

    // Calibrate the sensors. This turns the robot left then right, looking for a line [using line-following IR]
    char sensor_auto_calibrate (void);

    // Set calibration manually to the current settings.
    void calibrate(void);

    // Clear the current calibration settings
    void reset_calibration (void);

    // Start the PID contoller
    void PID_start(int max_speed, int a, int b, int c, int d);

    // Stop the PID controller
    void PID_stop();

    // Send a character directly to the 3pi serial interface
    int putc(int c);

    // Receive a character directly to the 3pi serial interface
    int getc();

    // Send a string buffer to the 3pi serial interface
    int print(char* text, int length);

    #ifdef MBED_RPC
    virtual const struct rpc_method *get_rpc_methods();
    #endif

    private :

    Serial _ser;
    PwmOut _oled_r, _oled_g, _oled_b, _cled_r, _cled_g, _cled_b;
    AnalogIn _gyro, _accel_x, _accel_y, _accel_z, _temperature, _light;
    I2C _i2c;
    DigitalOut _irpulse_1, _irpulse_2;
    Alpha433 _rf;
    Timer _system_timer;
 
    void motor (int motor, float speed);
    virtual int _putc(int c);
    virtual int _getc();

};

void display_system_time(void);
void init(void);
extern Serial pc;
extern PiSwarm piswarm;

#endif


/********************************************************************************
 *     MBED Pin connection layout                                               *
 *                                                                              *
 *   * Pin 05 : Alpha Transceiver SDI                                           *
 *   * Pin 06 : Alpha Transceiver SDO                                           *
 *   * Pin 07 : Alpha Transceiver CLK                                           *
 *   * Pin 08 : Alpha Transceiver FSS                                           *
 *   * Pin 09 : Serial (3pi) transmit                                           *
 *   * Pin 10 : Serial (3pi) receive                                            *
 *   * Pin 11 : Alpha Transceiver NIRQ                                          *
 *   * Pin 12 : IR Pulse Output for IR1,2,7,8                                   *
 *   * Pin 13 : IR Pulse Output for IR3,4,5,6                                   *
 *   * Pin 14 : Echo Pin for Sonar                                              *
 *   * Pin 15 : Light Sensor Analogue Input                                     *
 *   * Pin 16 : Gyrometer Analogue Input                                        *
 *   * Pin 17 : Accelerometer Z Analogue Input                                  *
 *   * Pin 18 : Accelerometer Y Analogue Input                                  *
 *   * Pin 19 : Accelerometer X Analogue Input                                  *
 *   * Pin 20 : Temperature Sensor Analogue Input                               *
 *   * Pin 21 : Center LED Blue PWM Output                                      *
 *   * Pin 22 : Center LED Green PWM Output                                     *
 *   * Pin 23 : Center LED Red PWM Output                                       *
 *   * Pin 24 : Outer LED Blue PWM Output                                       *
 *   * Pin 25 : Outer LED Green PWM Output                                      *
 *   * Pin 26 : Outer LED Red PWM Output                                        *
 *   * Pin 27 : I2C SCL                                                         *
 *   * Pin 28 : I2C SDA                                                         *
 *   * Pin 29 : GPIO Expander Interrupt                                         *
 *   * Pin 30 : Trigger Pin for Sonar                                           *
 *   * Pin 31 : USB D+ Connector                                                *
 *   * Pin 32 : USB D- Connector                                                *
 *                                                                              *
 *******************************************************************************/     

/********************************************************************************
 *     Using the 3-Pi Buzzer                                                    *
 *                                                                              * 
 *     play_tune (char * tune, int length)                                      *
 *                                                                              *
 *     tune - Pointer to a character sequence of the tune                       * 
 *     length - Length of characters in sequence                                *
 *                                                                              *
 *     The character sequence recognises the following characters:              *
 *                                                                              *
 *      A – G : used to specify the notes that will be played                   *
 *                                                                              *
 *      R : used to specify a rest (no sound for the duration of the note)      *
 *                                                                              *
 *      < or > : plays the next note one octave lower or higher                 *
 *                                                                              *
 *      + or # : raises the previous notes pitch by one semitone                *
 *                                                                              *
 *      - : lowers the previous note pitch by one semitone                      *
 *                                                                              *
 *      . : extend length of previous by 50% (each additional dot adds half as  *
 *          much as the previous dot, so that A.. is 1.75 times the length of A *
 *                                                                              *
 *      O followed by a number : sets the octave (default : O4)                 *
 *                                                                              *
 *      T followed by a number : sets the tempo in beats/min (default : T120)   *
 *                                                                              *
 *      L followed by a number : sets the default note duration specified by the*
 *                               number - 4 for quarter notes, 8 for eighth, 16 *
 *                               for sixteenth notes, etc. (default : L4)       *
 *                                                                              *
 *      V followed by a number : sets the volume (range 0 - 15, default : V15)  *
 *                                                                              *
 *      MS : sets all subsequent notes to play staccato (each note is played for*
 *           1/2 of its allotted time followed by an equal period of silence)   *
 *                                                                              *
 *      ML : sets all subsequent notes to play legato (each note is played for  *
 *           full length. This is the default setting.                          *
 *                                                                              *
 *      ! : resets the octave, tempo, duration, volume, and staccato setting to *
 *          their default values. These settings persist from one play_tune() to*
 *          the next, which allows you to more conveniently break up your music *
 *          into reusable sections.                                             *
 *                                                                              *
 *      1 - 2000 : when immediately following a note, a number determines the   *
 *                 duration of the note. For example, C16 specifies C played as *
 *                 a sixteenth note (1/16th the length of a whole note).        * 
 *                                                                              *
 *                                                                              *
 *      Some examples:                                                          *
 *                                                                              *
 *      Blue Danube Waltz (played badly!)                                       *
 *      play_tune("MSCEGGR>F>FR>D>DRCCEGGR>G>GR>E>ER<B<BDAAR>A>AR>F>F",51);     *
 *                                                                              *
 *******************************************************************************/
         
/********************************************************************************
 * COPYRIGHT NOTICE                                                             *
 *                                                                              *
 *                                                                              *
 * Parts of code based on the original m3pi library, Copyright (c) 2010 cstyles *
 *                                                                              *
 * Permission is hereby granted, free of charge, to any person obtaining a copy *
 * of this software and associated documentation files (the "Software"), to deal*
 * in the Software without restriction, including without limitation the rights * 
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell    *
 * copies of the Software, and to permit persons to whom the Software is        *
 * furnished to do so, subject to the following conditions:                     *
 *                                                                              *
 * The above copyright notice and this permission notice shall be included in   * 
 * all copies or substantial portions of the Software.                          *
 *                                                                              *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR   *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,     *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE  *
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER       *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,*
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN    *
 * THE SOFTWARE.                                                                *
 *                                                                              *
 *******************************************************************************/