#include "mbed.h"
#include "nexpaq_mdk.h"
#include "MAX30105.h"

PwmOut ledR(P2_4);
PwmOut ledG(P2_5);
PwmOut ledB(P2_6);
DigitalIn button(P0_1, PullUp);
I2C i2c(P1_6, P1_7);
MAX30105 max30105(i2c);

/***** Definitions *****/
#define		FUNCTION_TABLE_NUM					2
#define		UUID_NUM							16			// UUID number is 16, don't change it
#define     LOOP_DELAY                          100
#define     PROX_THRESHOLD                      10
#define     TEMP_INTERVAL                       10          // about once per second

/***** Globals *****/
void my_function_CMD_2700(unsigned char *pData, unsigned char len);
void my_function_CMD_2702(unsigned char *pData, unsigned char len);
const MDK_REGISTER_CMD my_cmd_func_table[FUNCTION_TABLE_NUM] = {
    {0x2700, my_function_CMD_2700},		// Command -> function
    {0x2702, my_function_CMD_2702},		// Command -> function
};

int tempCnt = 0;
int prxThrsh = PROX_THRESHOLD ;
int lastPrx = 0;
unsigned char prxPress = 0x02;
int lastBtn = 1;
unsigned char btnPress = 0x01;

/***** Functions *****/
void my_function_CMD_2700(unsigned char *pData, unsigned char len)
{
    unsigned char response = 0x00;
    ledR = 1.0f - (pData[0] / 255.0f);
    ledG = 1.0f - (pData[1] / 255.0f);
    ledB = 1.0f - (pData[2] / 255.0f);
    np_api_upload(0x2701, &response, 1);
}

void my_function_CMD_2702(unsigned char *pData, unsigned char len)
{
    unsigned char response = 0x00;
    prxThrsh = pData[0] ;
    np_api_upload(0x2703, &response, 1);
}

void sendTEMP()
{
    char dataBuf[2];
    unsigned char pData[3];
    pData[0] = 10;
    pData[1] = 0xFF;
    pData[2] = 0xFF;
    if (max30105.readReg(MAX30105::REG_TEMP_CONFIG, dataBuf) == 0) {
    	dataBuf[0] = MAX30105::REG_TEMP_INT;
    	if (i2c.write(MAX30105_I2C_ADDR, dataBuf, 1) == 0) {
	    	if (i2c.read(MAX30105_I2C_ADDR, dataBuf, 2) == 0) {
	    		pData[1] = dataBuf[0];
	    		pData[2] = dataBuf[1];
    		}
    	}
    }
    np_api_upload(0x2800, pData, 3);
    max30105.writeReg(MAX30105::REG_TEMP_CONFIG, 0x01);
}

/******************************************************************************/
void app_setup()
{
//	np_api_set_app_version(0, 0, 3);
    if ( np_api_register((MDK_REGISTER_CMD*)my_cmd_func_table, FUNCTION_TABLE_NUM) == MDK_REGISTER_FAILD ) {
        // Register failed handle code
    }
    max30105.softReset(); // reset the MAX30105
    max30105.shutDown();  // shut down while configuring
    max30105.enableIntr(MAX30105::INTR_PROX);  // enable proximity interrupt
    max30105.setProx(0x40, PROX_THRESHOLD);    // set proximity pulse amplitude and threshold
    max30105.setSingleLED();  // configure single LED mode to initiate proximity detection
    max30105.wakeUp();        // exit shutdown to start sensing
    ledR = 1.0f;
    ledG = 1.0f;
    ledB = 1.0f;
}

void app_loop()
{
    if (max30105.getIntr1() & MAX30105::INTR_PROX) { // if the proximity interrupt occurs
        if (!lastPrx) {
            np_api_upload(0x2800, &prxPress, 1);
        }
        max30105.writeReg(MAX30105::REG_MODE_CONFIG, MAX30105::MODE_1LED); // go back into proximity detection
        lastPrx = 1;
    } else {
        lastPrx = 0;
    }

    if (!button && lastBtn) {
        np_api_upload(0x2800, &btnPress, 1);
    }
    lastBtn = button;

    tempCnt += 1;
    if (tempCnt > TEMP_INTERVAL) {
        sendTEMP();
        tempCnt = 0;
    }
}

int main(void)
{
    np_api_init();
    app_setup();
    np_api_start();
    while(1) {
        app_loop();
        np_api_bsl_chk();
        Thread::wait(LOOP_DELAY);
    }
    return 0;
}
