/* Pmod Interface Library
 *
 */
#ifndef SERIALINTERFACE_H
#define SERIALINTERFACE_H

#include "mbed.h"
#include "SerialInterface.h"

#define MAX_NUM_ARGS      0x400
#define DBUF_MAX_LENGTH   0x400

/** Serial Interface Library, Provides utilities for remotely accessing peripherals
 *
 * Example:
 * @code
 * // Configure board to pass UART signals to peripheral connector.
 *
 * #include "SerialInterface.h"
 *
 * I2C i2c(P3_4, P3_5);
 * SPI spi(P5_1, P5_2, P5_0);
 * DigitalInOut gpio[] = {P5_3, P5_4, P5_5, P5_6, P3_0, P3_1, P3_2, P3_3};
 * AnalogIn ain[] = {AIN_0, AIN_1, AIN_2, AIN_3, AIN_4, AIN_5, AIN_6, AIN_7};
 * 
 * SerialInterface serInt(i2c, spi, gpio, ain);
 *
 * int main() {
 *    char ibuf[256];
 *    char obuf[256];
 *    while(1) {
 *        scanf("%s", ibuf);
 *        serInt.call(ibuf, obuf);
 *        printf("%s=", ibuf);
 *        printf("%s\n", obuf);
 * }
 * @endcode
 */
class SerialInterface
{
public:

    /** Create a SerialInterface interface
     *
     */
    SerialInterface(I2C *i2c, SPI *spi, DigitalInOut* gpio, AnalogIn* ain);

    ~SerialInterface();

    /** Name the I2C arguments
    */
    enum PINTi2cArgs {
        IA_CNT = 0, /**< Argument Count */
        IA_ADD,     /**< Device Address */
        IA_DATA,    /**< Data, Read = # bytes to read, Write = first data byte */
        IA_RDDA     /**< Read Data, data to write prior to read */
    };

    /** Process URI encoded commands
     *
     *  @param input a pointer to the string containing the command
     *  @param output a pointer to the string to write the result
     */
    void call(char* input, char* output);

private:

    // Types
    typedef union {
        struct {
            char csPin;
            char csPol;
            char format;
            char freq;
        };
        int merged;
    } spiConfig_t;

    // Internal Functions

    /** Process Analog Input Command
     *
     *  @param resp a pointer to the string to write the result
     */
    void fnc_ain(char* resp);

    /** Process Digital I/O Command
     *
     *  @param resp a pointer to the string to write the result
     */
    void fnc_dio(char* resp);

    /** Process I2C Command
     *
     *  @param resp a pointer to the string to write the result
     */
    void fnc_i2c(char* resp);

    /** Process SPI Command
     *
     *  @param resp a pointer to the string to write the result
     */
    void fnc_spi(char* resp);

    // Internal Resources
    I2C *_i2c;
    SPI *_spi;
    DigitalInOut *_gpio;
    AnalogIn *_ain;

    // Internal Buffers
    int _args[MAX_NUM_ARGS];
    char _dbuf[DBUF_MAX_LENGTH];

};

#endif