#include "YDwifiInterface.h"
#include "YDwifi_uartmsg.h"

using namespace murata_wifi;

#define MEMPOOL_BLOCK_SIZE  2048
typedef struct
{
    unsigned char buf[MEMPOOL_BLOCK_SIZE];
}tagMEMPOOL_BLOCK_T;
#define MEMPOOL_PAYLOAD_NUM 1
/** MemoryPool for payload of UART response */
MemoryPool<tagMEMPOOL_BLOCK_T, MEMPOOL_PAYLOAD_NUM>     gMEMPOOL_PAYLOAD;

#define UART_REQUEST_PAYLOAD_MAX    256

C_YDwifiInterface::C_YDwifiInterface( PinName tx, PinName rx, PinName cts, PinName rts, PinName reset, PinName alarm, int baud):
    C_YDwifi(tx, rx, cts, rts, reset, alarm, baud)
{
}

int C_YDwifiInterface::init()
{
    /* Initialize UART */
    initUart();
    return 0;
}

int C_YDwifiInterface::getFWVersion( unsigned char *version_p )
{
    // Get buffer for response payload from MemoryPool
    tagMEMPOOL_BLOCK_T *payload_buf = gMEMPOOL_PAYLOAD.alloc();
    if( payload_buf == NULL )
    {
        printf("getFWVersion payload_buf NULL\r\n");
        return -1;
    }

    tagGEN_FW_VER_GET_REQ_T req;
    unsigned char  payload_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned char  command_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned short payload_len;
    unsigned int   command_len;
    int ret;
    
    // Make request
    req.cmd_sid = UART_CMD_SID_GEN_FW_VER_GET_REQ;
    req.seq     = mUartRequestSeq++;
    
    // Make command payload
    payload_len = C_YD_UartMsg::makePayload( sizeof(tagGEN_FW_VER_GET_REQ_T), (unsigned char *)&req, payload_array );
    // Make all command request
    command_len = C_YD_UartMsg::makeRequest( UART_CMD_ID_GEN, payload_array, payload_len, command_array );

    // Set data for response
    mUartCommand.setCommandID( UART_CMD_ID_GEN );
    mUartCommand.setCommandSID( req.cmd_sid );
    mUartCommand.setResponseBuf( payload_buf->buf );
    
    // Send uart command request
    sendUart( command_len, command_array );
    
    // Wait UART response
    ret = mUartCommand.wait();
    printf( "getFWversion wait:%d\r\n", ret );
    if( ret != 0 )
    {
        printf( "getFWversion failed\r\n" );
        gMEMPOOL_PAYLOAD.free( payload_buf );
        return -1;
    }
    
    if( mUartCommand.getCommandStatus() == 0 )
    {
        unsigned char version_len = payload_buf->buf[3];
        memcpy( version_p, &payload_buf->buf[4], version_len );
    }
    gMEMPOOL_PAYLOAD.free( payload_buf );
    return 0;
}

int C_YDwifiInterface::connect(const char *ssid_p, unsigned char ssid_len, E_SECURITY sec_type
                            , const char *sec_key_p, unsigned char sec_key_len)
{
    // Parameter check(SSID)
    if( (ssid_p == NULL) || (ssid_len == 0) )
    {
        printf( "connect failed [ parameter NG:SSID ]\r\n" );
        return -1;
    }
    
    // Parameter check(Security key)
    if( (sec_type != e_SEC_OPEN) && ( (sec_key_len == 0) || (sec_key_p == NULL) ) )
    {
        printf( "connect failed [ parameter NG:Security key ]\r\n" );
        return -1;
    }
    
    // Get buffer for response payloadfrom MemoryPool
    tagMEMPOOL_BLOCK_T *payload_buf = gMEMPOOL_PAYLOAD.alloc();
    if( payload_buf == NULL )
    {
        printf("connect payload_buf NULL\r\n");
        return -1;
    }

    unsigned char  buf[UART_REQUEST_PAYLOAD_MAX];
    unsigned char  payload_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned char  command_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned int   buf_len = 0;
    unsigned short payload_len;
    unsigned int   command_len;
    int ret;

    memset( buf, 0, UART_REQUEST_PAYLOAD_MAX );
    // Make request
    buf[0] = UART_CMD_SID_WIFI_JOIN_REQ;
    buf_len++;
    buf[1] = mUartRequestSeq++;
    buf_len++;
    // SSID
    memcpy( &buf[2], ssid_p, ssid_len );
    buf_len += ssid_len;
    buf_len++;
    
    // Security mode
    buf[ buf_len ] = (unsigned char)sec_type;
    buf_len++;

    // Security key
    if( sec_type != e_SEC_OPEN )
    {
        buf[ buf_len ] = sec_key_len;
        buf_len++;
        if( sec_key_len > 0 )
        {
            memcpy( &buf[buf_len], sec_key_p, sec_key_len );
            buf_len += sec_key_len;
        }
    }
    // Make command payload
    payload_len = C_YD_UartMsg::makePayload( buf_len, (unsigned char *)buf, payload_array );
    // Make all command request
    command_len = C_YD_UartMsg::makeRequest( UART_CMD_ID_WIFI, payload_array, payload_len, command_array );
    // Set data for response
    mUartCommand.setCommandID( UART_CMD_ID_WIFI );
    mUartCommand.setCommandSID( UART_CMD_SID_WIFI_JOIN_REQ );
    mUartCommand.setResponseBuf( payload_buf->buf );
    
    // Send uart command request
    sendUart( command_len, command_array );
    
    // Wait UART response
    ret = mUartCommand.wait();
    if( ret != 0 )
    {
        printf( "join failed\r\n" );
        gMEMPOOL_PAYLOAD.free( payload_buf );
        return -1;
    }
    
    if( mUartCommand.getCommandStatus() != 0 )
    {
        printf("join status:%02x\r\n", mUartCommand.getCommandStatus());
        ret = -1;
    }
    gMEMPOOL_PAYLOAD.free( payload_buf );

    return ret;
}

int C_YDwifiInterface::disconnect()
{
    // Get buffer for response payloadfrom MemoryPool
    tagMEMPOOL_BLOCK_T *payload_buf = gMEMPOOL_PAYLOAD.alloc();
    if( payload_buf == NULL )
    {
        printf("disconnect payload_buf NULL\r\n");
        return -1;
    }

    tagWIFI_DISCONNECT_REQ_T req;
    unsigned char payload_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned char command_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned short payload_len;
    unsigned int   command_len;
    int ret;
    
    // Make request
    req.cmd_sid = UART_CMD_SID_WIFI_DISCONNECT_REQ;
    req.seq = mUartRequestSeq++;
    
    // Make command payload
    payload_len = C_YD_UartMsg::makePayload( sizeof(tagWIFI_DISCONNECT_REQ_T), (unsigned char *)&req, payload_array );
    // Make all command request
    command_len = C_YD_UartMsg::makeRequest( UART_CMD_ID_WIFI, payload_array, payload_len, command_array );

    // Set data for response
    mUartCommand.setCommandID( UART_CMD_ID_WIFI );
    mUartCommand.setCommandSID( req.cmd_sid );
    mUartCommand.setResponseBuf( payload_buf->buf );
    
    // Send uart command request
    sendUart( command_len, command_array );
    
    // Wait UART response
    ret = mUartCommand.wait();
    if( ret != 0 )
    {
        printf( "disconnect failed\r\n" );
        gMEMPOOL_PAYLOAD.free( payload_buf );
        return -1;
    }
    
    if( mUartCommand.getCommandStatus() != 0 )
    {
        printf("disconnect status:%02x\r\n", mUartCommand.getCommandStatus());
        ret = -1;
    }
    gMEMPOOL_PAYLOAD.free( payload_buf );
    return 0;
}

int C_YDwifiInterface::scan( const char *ssid_p, unsigned char *bssid_p
                        , void (*result_handler_p)(tagSCAN_RESULT_T *scan_result) )
{
    // Get buffer for response payloadfrom MemoryPool
    tagMEMPOOL_BLOCK_T *payload_buf = gMEMPOOL_PAYLOAD.alloc();
    if( payload_buf == NULL )
    {
        printf("scan payload_buf NULL\r\n");
        return -1;
    }
    
    tagWIFI_SCAN_REQ_T req;
    unsigned char payload_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned char command_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned short payload_len;
    unsigned int   command_len;
    unsigned int   buf_len = 0;
    int ret;
    
    memset( &req, 0, sizeof(tagWIFI_SCAN_REQ_T) );
    // Make request
    req.cmd_sid = UART_CMD_SID_WIFI_SCAN_REQ;
    buf_len++;
    req.seq = mUartRequestSeq++;
    buf_len++;
    
    // Set scan type(Active scan)
    req.scan_type = 0;
    buf_len++;
    // Set bss type(any)
    req.bss_type = 2;
    buf_len++;
    // Set BSSID
    if( bssid_p != NULL )
    {
        memcpy( req.bssid, bssid_p, BSSID_MAC_LENTH );
    }
    buf_len += BSSID_MAC_LENTH;
    // Set channel list(0)
    req.chan_list = 0;
    buf_len++;
    //Set SSID
    if( ssid_p != NULL )
    {
        strcpy( (char *)req.ssid, ssid_p );
        buf_len += strlen(ssid_p);
    }
    else
    {
        buf_len++;
    }
    
    // Make command payload
    payload_len = C_YD_UartMsg::makePayload( buf_len, (unsigned char *)&req, payload_array );
    // Make all command request
    command_len = C_YD_UartMsg::makeRequest( UART_CMD_ID_WIFI, payload_array, payload_len, command_array );

    // Set data for response
    mUartCommand.setCommandID( UART_CMD_ID_WIFI );
    mUartCommand.setCommandSID( req.cmd_sid );
    mUartCommand.setResponseBuf( payload_buf->buf );
    // Set scan result callback 
    mUartCommand.setScanResultHandler( result_handler_p );
    
    // Send uart command request
    sendUart( command_len, command_array );
    
    // Wait UART response
    ret = mUartCommand.wait();
    printf( "scan wait:%d\r\n", ret );
    if( ret != 0 )
    {
        printf( "scan failed\r\n" );
        gMEMPOOL_PAYLOAD.free( payload_buf );
        return -1;
    }
    
    if( mUartCommand.getCommandStatus() != 0 )
    {
        printf("scan status:%02x\r\n", mUartCommand.getCommandStatus());
        ret = -1;
    }
    gMEMPOOL_PAYLOAD.free( payload_buf );

    return 0;
}