/* Copyright (C) 2012 mbed.org, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
/******************* Murata Manufacturing Co.,Ltd. 2014 *****************
 *
 * Filename:   TCPSocketConnection.cpp
 *
 * Purpose:    This module has implementation of TCP connection.
 *              
 * $Author: kishino $
 *
 * $Date: 2014/03/26 $
 *
 * $Revision: 0.0.0.1 $
 * ***********************************************************************/
#include "TCPSocketConnection.h"
#include <cstring>

using namespace murata_wifi;

TCPSocketConnection::TCPSocketConnection()
{
}

int TCPSocketConnection::connect( unsigned int ip_addr, unsigned short port)
{
    int ret;

    // Socket create
    ret = createSocket();
    if( ret != 0 )
    {
        printf("createSocket error : %d\r\n", ret);
        return -1;
    }

    // Get buffer for response payload from MemoryPool
    tagMEMPOOL_BLOCK_T *payload_buf = mSnicWifi_p->getAlocCmdBuf();
    if( payload_buf == NULL )
    {
        printf("connect payload_buf NULL\r\n");
        return -1;
    }

    tagSNIC_TCP_CONNECT_TO_SERVER_REQ_T req;
    // Make request
    req.cmd_sid      = UART_CMD_SID_SNIC_TCP_CONNECT_TO_SERVER_REQ;
    req.seq          = mUartRequestSeq++;
    req.socket_id    = mSocketID;
    
    // set ip addr ( byte order )
    req.remote_addr[0] = ( (ip_addr & 0xFF000000) >> 24 );
    req.remote_addr[1] = ( (ip_addr & 0xFF0000) >> 16 );
    req.remote_addr[2] = ( (ip_addr & 0xFF00) >> 8 );
    req.remote_addr[3] = (ip_addr & 0xFF);
    req.remote_port[0] = ( (port & 0xFF00) >> 8 );
    req.remote_port[1] = (port & 0xFF);
    req.recv_bufsize[0] = ( (SNIC_UART_RECVBUF_SIZE & 0xFF00) >> 8 );
    req.recv_bufsize[1] = (SNIC_UART_RECVBUF_SIZE & 0xFF);
    req.timeout         = 60;

    unsigned char command_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned int  command_len;
    // Preparation of command
    command_len = mSnicWifi_p->preparationSendCommand( UART_CMD_ID_SNIC, req.cmd_sid, (unsigned char *)&req
                            , sizeof(tagSNIC_TCP_CONNECT_TO_SERVER_REQ_T), payload_buf->buf, command_array );

    // Send uart command request
    mSnicWifi_p->sendUart( command_len, command_array );

    mSnicWifi_p->mUartCommand.setCommandSID( UART_CMD_SID_SNIC_TCP_CONNECTION_STATUS_IND );

    // Wait UART response
    ret = mSnicWifi_p->mUartCommand.wait();
    if( ret != 0 )
    {
        printf( "connect failed\r\n" );
        mSnicWifi_p->freeCmdBuf( payload_buf );
        return -1;
    }
    
    if( mSnicWifi_p->mUartCommand.getCommandStatus() != UART_CMD_RES_SNIC_CONNECTION_UP )
    {
        printf("connect status:%02x\r\n", mSnicWifi_p->mUartCommand.getCommandStatus());
        mSnicWifi_p->freeCmdBuf( payload_buf );
        return -1;
    }
    mSnicWifi_p->freeCmdBuf( payload_buf );

    // Initialize connection information
    tagCONNECT_INFO_T *con_info_p = mSnicWifi_p->getConnectInfo( mSocketID );
    if( con_info_p->recvbuf_p == NULL )
    {
        printf( "create recv buffer[socket:%d]\r\n", mSocketID);
        con_info_p->recvbuf_p = new CircBuffer<unsigned char>(SNIC_UART_RECVBUF_SIZE);
    }
    con_info_p->is_connected = true;
    con_info_p->is_received  = false;
    
    return 0;
}

bool TCPSocketConnection::is_connected(void)
{
    tagCONNECT_INFO_T *con_info_p = mSnicWifi_p->getConnectInfo( mSocketID );
    return con_info_p->is_connected;
}

unsigned char gTCP_SEND_BUF[2048];
int TCPSocketConnection::send(unsigned char* data_p, int length)
{
    // Get buffer for response payload from MemoryPool
    tagMEMPOOL_BLOCK_T *payload_buf = mSnicWifi_p->getAlocCmdBuf();
    if( payload_buf == NULL )
    {
        printf("connect payload_buf NULL\r\n");
        return -1;
    }

    tagSNIC_TCP_SEND_FROM_SOCKET_REQ_T req;
    // Make request
    req.cmd_sid       = UART_CMD_SID_SNIC_SEND_FROM_SOCKET_REQ;
    req.seq           = mUartRequestSeq++;
    req.socket_id     = mSocketID;
    req.option        = 0;
    req.payload_len[0]= ( (length & 0xFF00) >> 8 );
    req.payload_len[1]= (length & 0xFF);
    
    int req_size = sizeof(tagSNIC_TCP_SEND_FROM_SOCKET_REQ_T);
    memcpy( gTCP_SEND_BUF, &req, req_size );
    memcpy( &gTCP_SEND_BUF[req_size], data_p, length );
    
    unsigned char command_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned int  command_len;
    // Preparation of command
    command_len = mSnicWifi_p->preparationSendCommand( UART_CMD_ID_SNIC, req.cmd_sid, gTCP_SEND_BUF
                            , req_size + length, payload_buf->buf, command_array );

    // Send uart command request
    mSnicWifi_p->sendUart( command_len, command_array );

//    mSnicWifi_p->mUartCommand.setCommandSID( req.cmd_sid );

    // Wait UART response
    int ret = mSnicWifi_p->mUartCommand.wait();
    if( ret != 0 )
    {
        printf( "send failed\r\n" );
        mSnicWifi_p->freeCmdBuf( payload_buf );
        return -1;
    }
    
    if( mSnicWifi_p->mUartCommand.getCommandStatus() != UART_CMD_RES_SNIC_SUCCESS )
    {
        printf("send status:%02x\r\n", mSnicWifi_p->mUartCommand.getCommandStatus());
        mSnicWifi_p->freeCmdBuf( payload_buf );
        return -1;
    }
    mSnicWifi_p->freeCmdBuf( payload_buf );

    // SNIC_SEND_FROM_SOCKET_REQ
    return 0;
}

int TCPSocketConnection::receive(unsigned char* data_p, int length)
{
    int i = 0;
    
    if( (data_p == NULL) || (length < 1) )
    {
        printf("TCPSocketConnection::receive parameter error\r\n");
        return -1;
    }
    
    // Initialize connection information
    tagCONNECT_INFO_T *con_info_p = mSnicWifi_p->getConnectInfo( mSocketID );
    if( con_info_p->recvbuf_p == NULL )
    {
        printf("TCPSocketConnection::receive Conncection info error\r\n");
        return -1;
    }

    // Check connection
    if( con_info_p->is_connected == false )
    {
        printf(" Socket id \"%d\" is not connected\r\n", mSocketID);
        return -1;
    }
    
    if( con_info_p->is_received == false )
    {
//        printf(" Socket id \"%d\" is not received\r\n", mSocketID);
        return 0;
    }
    
    // Get packet data from buffer for receive.
    for (i = 0; i < length; i ++) 
    {
        if (con_info_p->recvbuf_p->dequeue(&data_p[i]) == false)
        {
            break;
        }
    }

    if( con_info_p->recvbuf_p->isEmpty() )
    {
        con_info_p->is_received = false;
    }

    return i;
}
