/******************* Murata Manufacturing Co.,Ltd. 2014 *****************
 *
 * Filename:   SNIC_WifiInterface.cpp
 *
 * Purpose:    This module has implementation of API for SNIC UART of Wi-Fi. 
 *              
 * $Author: kishino $
 *
 * $Date: 2014/03/26 $
 *
 * $Revision: 0.0.0.1 $
 * ***********************************************************************/
#include "SNIC_WifiInterface.h"
#include "SNIC_UartMsgUtil.h"

using namespace murata_wifi;

C_SNIC_WifiInterface::C_SNIC_WifiInterface( PinName tx, PinName rx, PinName cts, PinName rts, PinName reset, PinName alarm, int baud)
{
    mSNICWifi_p = C_SNIC_Core::getInstance();

    mUART_tx     = tx;
    mUART_rx     = rx;
    mUART_cts    = cts;
    mUART_rts    = rts;;
    mUART_baud   = baud;
    mModuleReset = reset;
}

int C_SNIC_WifiInterface::init()
{    
    /* Initialize UART */
    mSNICWifi_p->initUart( mUART_tx, mUART_rx, mUART_baud );

    /* Initialize SNIC API */
    // Get buffer for response payload from MemoryPool
    tagMEMPOOL_BLOCK_T *payload_buf = mSNICWifi_p->getAlocCmdBuf();
    if( payload_buf == NULL )
    {
        printf("snic_init payload_buf NULL\r\n");
        return -1;
    }

    tagSNIC_INIT_REQ_T req;
    // Make request
    req.cmd_sid  = UART_CMD_SID_SNIC_INIT_REQ;
    req.seq      = mUartRequestSeq++;
    req.buf_size = 0x800;

    unsigned char command_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned int  command_len;
    // Preparation of command
    command_len = mSNICWifi_p->preparationSendCommand( UART_CMD_ID_SNIC, req.cmd_sid, (unsigned char *)&req
                            , sizeof(tagSNIC_INIT_REQ_T), payload_buf->buf, command_array );

    // Send uart command request
    mSNICWifi_p->sendUart( command_len, command_array );

    int ret;
    // Wait UART response
    ret = mSNICWifi_p->mUartCommand.wait();
    if( ret != 0 )
    {
        printf( "snic_init failed\r\n" );
        mSNICWifi_p->freeCmdBuf( payload_buf );
        return -1;
    }
    
    if( mSNICWifi_p->mUartCommand.getCommandStatus() != 0 )
    {
        printf("snic_init status:%02x\r\n", mSNICWifi_p->mUartCommand.getCommandStatus());
        ret = -1;
    }
    mSNICWifi_p->freeCmdBuf( payload_buf );
    
    return ret;
}

int C_SNIC_WifiInterface::getFWVersion( unsigned char *version_p )
{
    // Get buffer for response payload from MemoryPool
    tagMEMPOOL_BLOCK_T *payload_buf = mSNICWifi_p->getAlocCmdBuf();
    if( payload_buf == NULL )
    {
        printf("getFWVersion payload_buf NULL\r\n");
        return -1;
    }

    tagGEN_FW_VER_GET_REQ_T req;
    // Make request
    req.cmd_sid = UART_CMD_SID_GEN_FW_VER_GET_REQ;
    req.seq     = mUartRequestSeq++;
    
    unsigned char command_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned int  command_len;
    // Preparation of command
    command_len = mSNICWifi_p->preparationSendCommand( UART_CMD_ID_GEN, req.cmd_sid, (unsigned char *)&req
                        , sizeof(tagGEN_FW_VER_GET_REQ_T), payload_buf->buf, command_array );

    int ret;
    
    // Send uart command request
    mSNICWifi_p->sendUart( command_len, command_array );
    
    // Wait UART response
    ret = mSNICWifi_p->mUartCommand.wait();
    if( ret != 0 )
    {
        printf( "getFWversion failed\r\n" );
        mSNICWifi_p->freeCmdBuf( payload_buf );
        return -1;
    }
    
    if( mSNICWifi_p->mUartCommand.getCommandStatus() == 0 )
    {
        unsigned char version_len = payload_buf->buf[3];
        memcpy( version_p, &payload_buf->buf[4], version_len );
    }
    mSNICWifi_p->freeCmdBuf( payload_buf );
    return 0;
}

int C_SNIC_WifiInterface::connect(const char *ssid_p, unsigned char ssid_len, E_SECURITY sec_type
                            , const char *sec_key_p, unsigned char sec_key_len)
{
    // Parameter check(SSID)
    if( (ssid_p == NULL) || (ssid_len == 0) )
    {
        printf( "connect failed [ parameter NG:SSID ]\r\n" );
        return -1;
    }
    
    // Parameter check(Security key)
    if( (sec_type != e_SEC_OPEN) && ( (sec_key_len == 0) || (sec_key_p == NULL) ) )
    {
        printf( "connect failed [ parameter NG:Security key ]\r\n" );
        return -1;
    }
    
    // Get buffer for response payload from MemoryPool
    tagMEMPOOL_BLOCK_T *payload_buf = mSNICWifi_p->getAlocCmdBuf();
    if( payload_buf == NULL )
    {
        printf("connect payload_buf NULL\r\n");
        return -1;
    }

    unsigned char buf[UART_REQUEST_PAYLOAD_MAX];
    unsigned int  buf_len = 0;
    unsigned char command_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned int  command_len;

    memset( buf, 0, UART_REQUEST_PAYLOAD_MAX );
    // Make request
    buf[0] = UART_CMD_SID_WIFI_JOIN_REQ;
    buf_len++;
    buf[1] = mUartRequestSeq++;
    buf_len++;
    // SSID
    memcpy( &buf[2], ssid_p, ssid_len );
    buf_len += ssid_len;
    buf_len++;
    
    // Security mode
    buf[ buf_len ] = (unsigned char)sec_type;
    buf_len++;

    // Security key
    if( sec_type != e_SEC_OPEN )
    {
        buf[ buf_len ] = sec_key_len;
        buf_len++;
        if( sec_key_len > 0 )
        {
            memcpy( &buf[buf_len], sec_key_p, sec_key_len );
            buf_len += sec_key_len;
        }
    }

    // Preparation of command
    command_len = mSNICWifi_p->preparationSendCommand( UART_CMD_ID_WIFI, UART_CMD_SID_WIFI_JOIN_REQ, buf
                        , buf_len, payload_buf->buf, command_array );

    // Send uart command request
    mSNICWifi_p->sendUart( command_len, command_array );
    
    int ret;
    // Wait UART response
    ret = mSNICWifi_p->mUartCommand.wait();
    if( ret != 0 )
    {
        printf( "join failed\r\n" );
        mSNICWifi_p->freeCmdBuf( payload_buf );
        return -1;
    }
    
    if( mSNICWifi_p->mUartCommand.getCommandStatus() != 0 )
    {
        printf("join status:%02x\r\n", mSNICWifi_p->mUartCommand.getCommandStatus());
        ret = -1;
    }
    mSNICWifi_p->freeCmdBuf( payload_buf );

    return ret;
}

int C_SNIC_WifiInterface::disconnect()
{
    // Get buffer for response payload from MemoryPool
    tagMEMPOOL_BLOCK_T *payload_buf = mSNICWifi_p->getAlocCmdBuf();
    if( payload_buf == NULL )
    {
        printf("disconnect payload_buf NULL\r\n");
        return -1;
    }

    tagWIFI_DISCONNECT_REQ_T req;
    // Make request
    req.cmd_sid = UART_CMD_SID_WIFI_DISCONNECT_REQ;
    req.seq = mUartRequestSeq++;
    
    unsigned char command_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned int  command_len;
    // Preparation of command
    command_len = mSNICWifi_p->preparationSendCommand( UART_CMD_ID_WIFI, req.cmd_sid, (unsigned char *)&req
                        , sizeof(tagWIFI_DISCONNECT_REQ_T), payload_buf->buf, command_array );

    // Send uart command request
    mSNICWifi_p->sendUart( command_len, command_array );
    
    int ret;
    // Wait UART response
    ret = mSNICWifi_p->mUartCommand.wait();
    if( ret != 0 )
    {
        printf( "disconnect failed\r\n" );
        mSNICWifi_p->freeCmdBuf( payload_buf );
        return -1;
    }
    
    if( mSNICWifi_p->mUartCommand.getCommandStatus() != 0 )
    {
        printf("disconnect status:%02x\r\n", mSNICWifi_p->mUartCommand.getCommandStatus());
        ret = -1;
    }
    mSNICWifi_p->freeCmdBuf( payload_buf );
    return ret;
}

int C_SNIC_WifiInterface::scan( const char *ssid_p, unsigned char *bssid_p
                        , void (*result_handler_p)(tagSCAN_RESULT_T *scan_result) )
{
    // Get buffer for response payload from MemoryPool
    tagMEMPOOL_BLOCK_T *payload_buf = mSNICWifi_p->getAlocCmdBuf();
    if( payload_buf == NULL )
    {
        printf("scan payload_buf NULL\r\n");
        return -1;
    }
    
    tagWIFI_SCAN_REQ_T req;
    unsigned int  buf_len = 0;
    
    memset( &req, 0, sizeof(tagWIFI_SCAN_REQ_T) );
    // Make request
    req.cmd_sid = UART_CMD_SID_WIFI_SCAN_REQ;
    buf_len++;
    req.seq = mUartRequestSeq++;
    buf_len++;
    
    // Set scan type(Active scan)
    req.scan_type = 0;
    buf_len++;
    // Set bss type(any)
    req.bss_type = 2;
    buf_len++;
    // Set BSSID
    if( bssid_p != NULL )
    {
        memcpy( req.bssid, bssid_p, BSSID_MAC_LENTH );
    }
    buf_len += BSSID_MAC_LENTH;
    // Set channel list(0)
    req.chan_list = 0;
    buf_len++;
    //Set SSID
    if( ssid_p != NULL )
    {
        strcpy( (char *)req.ssid, ssid_p );
        buf_len += strlen(ssid_p);
    }
    buf_len++;

    unsigned char command_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned int  command_len;
    // Preparation of command
    command_len = mSNICWifi_p->preparationSendCommand( UART_CMD_ID_WIFI, req.cmd_sid, (unsigned char *)&req
                        , buf_len, payload_buf->buf, command_array );

    // Set scan result callback 
    mSNICWifi_p->mUartCommand.setScanResultHandler( result_handler_p );
    
    // Send uart command request
    mSNICWifi_p->sendUart( command_len, command_array );

    int ret;
    // Wait UART response
    ret = mSNICWifi_p->mUartCommand.wait();
    printf( "scan wait:%d\r\n", ret );
    if( ret != 0 )
    {
        printf( "scan failed\r\n" );
        mSNICWifi_p->freeCmdBuf( payload_buf );
        return -1;
    }
    
    if( mSNICWifi_p->mUartCommand.getCommandStatus() != 0 )
    {
        printf("scan status:%02x\r\n", mSNICWifi_p->mUartCommand.getCommandStatus());
        ret = -1;
    }

    mSNICWifi_p->freeCmdBuf( payload_buf );

    return ret;
}

int C_SNIC_WifiInterface::wifi_on( const char *country_p )
{
    // Parameter check
    if( country_p == NULL )
    {
        printf("wifi_on parameter error\r\n");
        return -1;
    }
    
    // Get buffer for response payload from MemoryPool
    tagMEMPOOL_BLOCK_T *payload_buf = mSNICWifi_p->getAlocCmdBuf();
    if( payload_buf == NULL )
    {
        printf("wifi_on payload_buf NULL\r\n");
        return -1;
    }

    tagWIFI_ON_REQ_T req;
    // Make request
    req.cmd_sid = UART_CMD_SID_WIFI_ON_REQ;
    req.seq = mUartRequestSeq++;
    memcpy( req.country, country_p, COUNTRYC_CODE_LENTH );
    
    unsigned char command_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned int  command_len;
    // Preparation of command
    command_len = mSNICWifi_p->preparationSendCommand( UART_CMD_ID_WIFI, req.cmd_sid, (unsigned char *)&req
                        , sizeof(tagWIFI_ON_REQ_T), payload_buf->buf, command_array );

    // Send uart command request
    mSNICWifi_p->sendUart( command_len, command_array );
    
    int ret;
    // Wait UART response
    ret = mSNICWifi_p->mUartCommand.wait();
    if( ret != 0 )
    {
        printf( "wifi_on failed\r\n" );
        mSNICWifi_p->freeCmdBuf( payload_buf );
        return -1;
    }
    
    if( mSNICWifi_p->mUartCommand.getCommandStatus() != 0 )
    {
        printf("wifi_on status:%02x\r\n", mSNICWifi_p->mUartCommand.getCommandStatus());
        ret = -1;
    }
    mSNICWifi_p->freeCmdBuf( payload_buf );

    return ret;
}

int C_SNIC_WifiInterface::wifi_off()
{
    // Get buffer for response payload from MemoryPool
    tagMEMPOOL_BLOCK_T *payload_buf = mSNICWifi_p->getAlocCmdBuf();
    if( payload_buf == NULL )
    {
        printf("wifi_off payload_buf NULL\r\n");
        return -1;
    }

    tagWIFI_OFF_REQ_T req;
    // Make request
    req.cmd_sid = UART_CMD_SID_WIFI_OFF_REQ;
    req.seq = mUartRequestSeq++;
    
    unsigned char command_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned int  command_len;
    // Preparation of command
    command_len = mSNICWifi_p->preparationSendCommand( UART_CMD_ID_WIFI, req.cmd_sid, (unsigned char *)&req
                        , sizeof(tagWIFI_OFF_REQ_T), payload_buf->buf, command_array );

    // Send uart command request
    mSNICWifi_p->sendUart( command_len, command_array );
    
    int ret;
    // Wait UART response
    ret = mSNICWifi_p->mUartCommand.wait();
    if( ret != 0 )
    {
        printf( "wifi_off failed\r\n" );
        mSNICWifi_p->freeCmdBuf( payload_buf );
        return -1;
    }
    
    if( mSNICWifi_p->mUartCommand.getCommandStatus() != 0 )
    {
        printf("wifi_off status:%02x\r\n", mSNICWifi_p->mUartCommand.getCommandStatus());
        ret = -1;
    }
    mSNICWifi_p->freeCmdBuf( payload_buf );

    return ret;
}

int C_SNIC_WifiInterface::getRssi( signed char *rssi_p )
{
    if( rssi_p == NULL )
    {
        printf("getRssi parameter error\r\n");
        return -1;
    }
    
    // Get buffer for response payload from MemoryPool
    tagMEMPOOL_BLOCK_T *payload_buf = mSNICWifi_p->getAlocCmdBuf();
    if( payload_buf == NULL )
    {
        printf("getRssi payload_buf NULL\r\n");
        return -1;
    }

    tagWIFI_GET_STA_RSSI_REQ_T req;
    
    // Make request
    req.cmd_sid = UART_CMD_SID_WIFI_GET_STA_RSSI_REQ;
    req.seq     = mUartRequestSeq++;
    
    unsigned char  command_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned int   command_len;
    command_len = mSNICWifi_p->preparationSendCommand( UART_CMD_ID_WIFI, req.cmd_sid, (unsigned char *)&req
                        , sizeof(tagWIFI_GET_STA_RSSI_REQ_T), payload_buf->buf, command_array );

    int ret;
    // Send uart command request
    mSNICWifi_p->sendUart( command_len, command_array );
    
    // Wait UART response
    ret = mSNICWifi_p->mUartCommand.wait();
    if( ret != 0 )
    {
        printf( "getRssi failed\r\n" );
        mSNICWifi_p->freeCmdBuf( payload_buf );
        return -1;
    }
    
    *rssi_p = (signed char)payload_buf->buf[2];

    mSNICWifi_p->freeCmdBuf( payload_buf );
    return 0;
}

int C_SNIC_WifiInterface::getWifiStatus( tagWIFI_STATUS_T *status_p)
{
    if( status_p == NULL )
    {
        printf("getWifiStatus parameter error\r\n");
        return -1;
    }
    
    // Get buffer for response payload from MemoryPool
    tagMEMPOOL_BLOCK_T *payload_buf = mSNICWifi_p->getAlocCmdBuf();
    if( payload_buf == NULL )
    {
        printf("getWifiStatus payload_buf NULL\r\n");
        return -1;
    }

    tagWIFI_GET_STATUS_REQ_T req;
    // Make request
    req.cmd_sid = UART_CMD_SID_WIFI_GET_STATUS_REQ;
    req.seq     = mUartRequestSeq++;
    req.interface = 0;
    
    unsigned char  command_array[UART_REQUEST_PAYLOAD_MAX];
    unsigned int   command_len;
    command_len = mSNICWifi_p->preparationSendCommand( UART_CMD_ID_WIFI, req.cmd_sid, (unsigned char *)&req
                        , sizeof(tagWIFI_GET_STATUS_REQ_T), payload_buf->buf, command_array );

    // Send uart command request
    mSNICWifi_p->sendUart( command_len, command_array );
    
    int ret;
    // Wait UART response
    ret = mSNICWifi_p->mUartCommand.wait();
    if( ret != 0 )
    {
        printf( "getWifiStatus failed\r\n" );
        mSNICWifi_p->freeCmdBuf( payload_buf );
        return -1;
    }
    
    // set status
    status_p->status = (E_WIFI_STATUS)payload_buf->buf[2];
    
    // set Mac address
    if( status_p->status != e_STATUS_OFF )
    {
        memcpy( status_p->mac_address, &payload_buf->buf[3], BSSID_MAC_LENTH );
    } 

    // set SSID
    if( ( status_p->status == e_STA_JOINED ) == ( status_p->status == e_AP_STARTED ) )
    {
        memcpy( status_p->ssid, &payload_buf->buf[9], strlen( (char *)&payload_buf->buf[9]) );
    } 

    mSNICWifi_p->freeCmdBuf( payload_buf );
    return 0;
}
