/*
Demo application for storage_on_flash library which is for STM32F4xx series.
*/
#include <stdio.h>
#include <stdlib.h>

#include "SOFBlock.h"
#include <string.h>

#include "mbed.h"

Serial pc(SERIAL_TX, SERIAL_RX);

#define TEST_BUF_SIZE 256

static char test_buf[TEST_BUF_SIZE];

static char *get_input()
{
	size_t i;
	int c;

	for (i = 0; i < TEST_BUF_SIZE-1; ++i)
	{
		c = pc.getc();
		printf("%c", c);	// echo
		if (c == '\r' || c == '\n')
		{
			printf("\n");
			break;
		}
		test_buf[i] = (char)c;
	}

	test_buf[i] = 0;
	

	return test_buf;
}

static int geti()
{
	return atoi(get_input());
}

static uint8_t ask_sector_no()
{
	printf(" sector number? ");

	return (uint8_t)geti();
}

static void show_spec(uint8_t sector_index)
{
	const SOF_SectorSpec_t *spec = SOF_block_get_info(sector_index);

	if (spec == NULL)
	{
		printf("SOF_block_get_info(%u) error\r\n", sector_index);
		return;
	}

	printf("sec_no=%u\r\n", spec->sec_no);
	printf("sec_addr=%#x\r\n", spec->sec_addr);
	printf("sec_size=%u\r\n", spec->sec_size);
}

static void foramt_sector(uint8_t sector_index)
{
	printf("format(%u)...\r\n", sector_index);
	printf("   %s\r\n", SOFBlock::format(sector_index) ? "ok" : "fail");
}

static void write_data(uint8_t sector_index)
{
	printf(" input? ");
	char *s = get_input();
	
	SOFWriter writer;

	if (writer.open(sector_index) != kSOF_ErrNone)
	{
		printf("open fail\r\n");
		return;
	}

	size_t len = strlen(s);
	size_t r;
	if ((r=writer.write_data((uint8_t*)s, len)) != len)
	{
		printf("write_data fail: %u/%u\r\n", r, len);
		printf("No free area exists, format required\r\n");
		return;
	}
	
	printf("write_data ok: %u bytes\r\n", r);
}

static void read_data(uint8_t sector_index)
{
	SOFReader reader;

	if (reader.open(sector_index) != kSOF_ErrNone)
	{
		printf("open fail\r\n");
		return;
	}

	printf("data %d bytes at %p :\r\n", reader.get_data_size(), reader.get_physical_data_addr());
	printf("%.*s", reader.get_data_size(), reader.get_physical_data_addr());
	printf("\r\n");
}

static void show_statics(uint8_t sector_index)
{
	SOF_Statics_t stats;

	if (SOF_block_get_statics(sector_index, &stats) !=  kSOF_ErrNone)
	{
		printf("SOF_block_get_statics(%u) error\r\n", sector_index);
		return;
	}

	printf("data_addr=%p\r\n", stats.data_addr);
	printf("data_size=%u\r\n", stats.data_size);
	printf("free_size=%u\r\n", stats.free_size);
}

static void demo()
{
	while (true)
	{
		printf("\r\n* MENU *\r\n");
		printf("1. see flash sector spec\r\n");
		printf("2. format sector\r\n");
		printf("3. write data to sector\r\n");
		printf("4. read data from sector\r\n");
		printf("5. see storage status\r\n");

		printf(" select? ");
		switch(geti())
		{
		case 1:
			show_spec(ask_sector_no());
			break;
		case 2:
			foramt_sector(ask_sector_no());
			break;
		case 3:
			write_data(ask_sector_no());
			break;
		case 4:
			read_data(ask_sector_no());
			break;
		case 5:
			show_statics(ask_sector_no());
			break;
		default:
			printf("invalid choice\r\n");
			break;
		}
	}
}

int main()
{
	demo();

	return 0;
}


