/* Copyright (c) 2010-2012 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#define __DEBUG__ 0
#ifndef __MODULE__
#define __MODULE__ "WANDongleInitializer.cpp"
#endif

#include "core/dbg.h"

#include <cstdint>
using std::uint16_t;

#include "WANDongleInitializer.h"

WANDongleInitializer::WANDongleInitializer(USBHost* pHost) : m_pHost(pHost)
{

}

WANDongleInitializer** WANDongleInitializer::getInitializers(USBHost* pHost)
{
  static Sprint598UInitializer sprint598U(pHost);
  static SprintMC5728VInitializer sprintMC5728V(pHost);
  const static WANDongleInitializer* list[] = { &sprint598U, &sprintMC5728V, NULL };
  return (WANDongleInitializer**)list;
}

//Sierra Wireless 598U (Sprint)
Sprint598UInitializer::Sprint598UInitializer(USBHost* pHost) : WANDongleInitializer(pHost)
{
  
}

uint16_t Sprint598UInitializer::getMSDVid()      { return 0x1199; }
uint16_t Sprint598UInitializer::getMSDPid()      { return 0x0FFF; } //0x0f25?

uint16_t Sprint598UInitializer::getSerialVid()   { return 0x1199; }
uint16_t Sprint598UInitializer::getSerialPid()   { return 0x0025; }

bool Sprint598UInitializer::switchMode(USBDeviceConnected* pDev)
{
  DBG("Device %p found, will now try to switch into serial mode", (void *)pDev);
  m_pHost->controlWrite(pDev, /*requestType = */ 0x40, /*request = */ 0x0B, /*value = */ 0x00000001, /*index = */ 0, /*buf = */ NULL, /*len = */ 0); 
  return true;
}

USBEndpoint* Sprint598UInitializer::getEp(USBDeviceConnected* pDev, int serialPortNumber, bool tx)
{
  return pDev->getEndpoint(serialPortNumber, BULK_ENDPOINT, tx?OUT:IN, 0);
}

int Sprint598UInitializer::getSerialPortCount()
{
  return 1;
}

/*virtual*/ void Sprint598UInitializer::setVidPid(uint16_t vid, uint16_t pid)
{
    if( (vid == getSerialVid() ) && ( pid == getSerialPid() ) )
    {
      m_hasSwitched = true;
      m_currentSerialIntf = 0;
      m_currentEndpoint = 0;
    }
    else
    {
      m_hasSwitched = false;
    }
}

/*virtual*/ bool Sprint598UInitializer::parseInterface(uint8_t intf_nb, uint8_t intf_class, uint8_t intf_subclass, uint8_t intf_protocol) //Must return true if the interface should be parsed
{
  if( m_hasSwitched )
  {
    DBG("Interface #%d; Class:%02x; SubClass:%02x; Protocol:%02x", intf_nb, intf_class, intf_subclass, intf_protocol);
    if( intf_class == 0xFF )
    {
      if( (m_currentSerialIntf == 0) )
      {
        m_currentSerialIntf++;
        return true;
      }
      m_currentSerialIntf++;
    }
  }
  return false;
}

/*virtual*/ bool Sprint598UInitializer::useEndpoint(uint8_t intf_nb, ENDPOINT_TYPE type, ENDPOINT_DIRECTION dir) //Must return true if the endpoint will be used
{
  if( m_hasSwitched )
  {
    DBG("USBEndpoint on Interface #%d; Type:%d; Direction:%d", intf_nb, type, dir);
    if(type == BULK_ENDPOINT)
    {
      if( (m_currentEndpoint >= 0) && (m_currentEndpoint < 2) )
      {
        m_currentEndpoint++;
        return true;
      }
      else
      {
        m_currentEndpoint++;
      }
    }
  }
  return false;
}


/*virtual*/ WAN_DONGLE_TYPE Sprint598UInitializer::getType()
{
  return WAN_DONGLE_TYPE_SPRINT598U;
}

//Sierra Wireless MC5728V (Sprint)
SprintMC5728VInitializer::SprintMC5728VInitializer(USBHost* pHost) : WANDongleInitializer(pHost)
{
  
}

uint16_t SprintMC5728VInitializer::getMSDVid()      { return 0x1199; }
uint16_t SprintMC5728VInitializer::getMSDPid()      { return 0x0000; } //No MSD mode (presumably)

uint16_t SprintMC5728VInitializer::getSerialVid()   { return 0x1199; }
uint16_t SprintMC5728VInitializer::getSerialPid()   { return 0x0028; }

bool SprintMC5728VInitializer::switchMode(USBDeviceConnected* pDev)
{
  return true;
}

USBEndpoint* SprintMC5728VInitializer::getEp(USBDeviceConnected* pDev, int serialPortNumber, bool tx)
{
  return pDev->getEndpoint(serialPortNumber, BULK_ENDPOINT, tx?OUT:IN, 0);
}

int SprintMC5728VInitializer::getSerialPortCount()
{
  return 1;
}

/*virtual*/ void SprintMC5728VInitializer::setVidPid(uint16_t vid, uint16_t pid)
{
  m_currentSerialIntf = 0;
  m_currentEndpoint = 0;
}

/*virtual*/ bool SprintMC5728VInitializer::parseInterface(uint8_t intf_nb, uint8_t intf_class, uint8_t intf_subclass, uint8_t intf_protocol) //Must return true if the interface should be parsed
{
  DBG("Interface #%d; Class:%02x; SubClass:%02x; Protocol:%02x", intf_nb, intf_class, intf_subclass, intf_protocol);
  if( intf_class == 0xFF )
  {
    if( (m_currentSerialIntf == 0) )
    {
      m_currentSerialIntf++;
      return true;
    }
    m_currentSerialIntf++;
  }
  return false;
}

/*virtual*/ bool SprintMC5728VInitializer::useEndpoint(uint8_t intf_nb, ENDPOINT_TYPE type, ENDPOINT_DIRECTION dir) //Must return true if the endpoint will be used
{
  DBG("USBEndpoint on Interface #%d; Type:%d; Direction:%d", intf_nb, type, dir);
  if(type == BULK_ENDPOINT)
  {
    if( (m_currentEndpoint >= 0) && (m_currentEndpoint < 2) )
    {
      m_currentEndpoint++;
      return true;
    }
    else
    {
      m_currentEndpoint++;
    }
  }
  return false;
}


/*virtual*/ WAN_DONGLE_TYPE SprintMC5728VInitializer::getType()
{
  return WAN_DONGLE_TYPE_SPRINTMC5728V;
}