/* mbed Microcontroller Library
 * Copyright (c) 2006-2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "toolchain.h"
#include "ble/BLE.h"

BLE ble;

static Ticker ticker;
static bool       triggerTempValueUpdate = false;
static DigitalOut alivenessLED(LED1, 1);

struct ApplicationData_t {
    uint16_t                     applicationSpecificId; /* An ID used to identify temperature value in the manufacture specific AD data field */
    int dummyValue;        /* User defined application data */
} PACKED;

void periodicCallback(void)
{
    alivenessLED = !alivenessLED;  /* Do blinky on LED1 while we're waiting for BLE events. */

    /* Note that the periodicCallback() executes in interrupt context, so it is safer to do
     * heavy-weight sensor polling from the main thread (where we should be able to block safely, if needed). */
    triggerTempValueUpdate = true;
}

void setupApplicationData(ApplicationData_t &appData)
{
    static const uint16_t APP_SPECIFIC_ID_TEST = 0xFEFE;
    appData.applicationSpecificId = APP_SPECIFIC_ID_TEST;
    appData.dummyValue        = 0;
}

void updateApplicationData(ApplicationData_t &appData)
{
    static const uint16_t APP_SPECIFIC_ID_TEST = 0xFEFE;
    appData.applicationSpecificId = APP_SPECIFIC_ID_TEST;
    appData.dummyValue=appData.dummyValue+1;
}

void startAdvertisingTemperature(void)
{
    /* Setup advertising payload */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::GENERIC_THERMOMETER);
    ApplicationData_t appData;
    setupApplicationData(appData);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::MANUFACTURER_SPECIFIC_DATA, (uint8_t *)&appData, sizeof(ApplicationData_t));

    /* Setup advertising parameters */
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_NON_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(500);

    ble.gap().startAdvertising();
}

int main(void)
{
    ticker.attach(periodicCallback, 2); /* trigger sensor polling every 2 seconds */

    ble.init();
    startAdvertisingTemperature();

    while (true) {
        if (triggerTempValueUpdate) {
            // Do blocking calls or whatever hardware-specific action is necessary to poll the sensor.
            ApplicationData_t appData;
            updateApplicationData(appData);
            ble.gap().updateAdvertisingPayload(GapAdvertisingData::MANUFACTURER_SPECIFIC_DATA, (uint8_t *)&appData, sizeof(ApplicationData_t));

            triggerTempValueUpdate = false;
        }

        ble.waitForEvent();
    }
}
