#include "mbed.h"
#include "FilterDesign.h"
#include "HIDScope.h"
#include "QEI.h"
#include "MODSERIAL.h"
#include "BiQuad4.h"
#include "BiQuad.h"
#include "FastPWM.h"
#include <vector>


MODSERIAL pc(USBTX, USBRX);

//State machine
InterruptIn stateSwitch(SW2);
InterruptIn stepSwitch(SW3);

void stateMachine();
void switchStep();
void switchState();

int noStateSteps = 4;
int stateStep = 0;
int state = 0;

void EMGState();
void demoState();

//Shared steps
void calibrateMotors();
void waiting();

//Demo steps
void playDemo();


//EMG steps
void EMGcalibration();
void moveToInitialPosition();
void moveWithEMG();

//Calibration
void moveMotorToStop(DigitalOut *M, PwmOut *E, QEI *Enc, double speed);

//play demo
void moveAlongPath(double xStart, double yStart, double xEnd, double yEnd, double speed);

//Motors
double Pi = 3.14159265359;

QEI Enc1(D12, D13, NC, 32);
QEI Enc2(D10, D11, NC, 32);

DigitalOut M1(D4);
DigitalOut M2(D7);

PwmOut E1(D5);
PwmOut E2(D6);

double gearRatio = 40/9;

double initRot1 = 0;
double initRot2 = -gearRatio*(180 - 48.5)/360;

double xCurrent;
double yCurrent;

double calcRot1(double xDes, double yDes);
double calcRot2(double xDes, double yDes);

void findDesiredLocation(double xStart, double yStart, double xEnd, double yEnd, double speed, Timer *t, vector<double> *desiredLocation);
void moveMotorContinuously(DigitalOut *M, PwmOut *E, QEI *Enc, double initRot, double dir, vector<double> *pidInfo, Timer *t, double rotDes);
void moveMotorToPoint(DigitalOut *M, PwmOut *E, QEI *Enc, double initRot, double dir, double rotDes);
void moveWithSpeed(double *xStart, double *yStart, double xSpeed, double ySpeed);
void flipx();
void flipy();

//EMG
AnalogIn emg0( A0 );
AnalogIn emg1( A3 );

HIDScope scope( 2 );

Ticker ticker_calibration;   // Ticker to send the EMG signals to screen

volatile double emg1_cal = 0.8;      // Initial value of the calibrated first emg
volatile double emg1_filtered;      //measured value of the first emg
volatile double emg2_filtered;      //measured value of the second emg
volatile double emg2_cal = 0.8;     // Initial value of the calibrated second emg

double speedy = 3;
double speedx = 3;

double xDir = 1;
double yDir = 1;

vector<double> pidInfo1 (4);
vector<double> pidInfo2 (4);

Timer tEMGMove;

InterruptIn xSwitch(D8);
InterruptIn ySwitch(D9);

void sample();

//Waiting
DigitalOut r_led(LED_RED);

int main()
{
    //set up some functions of Intterrupts
    stateSwitch.rise(switchState);
    stepSwitch.rise(switchStep);

    xSwitch.mode(PullUp);
    ySwitch.mode(PullUp);

    xSwitch.fall(flipx);
    ySwitch.fall(flipy);

    pc.baud(115200);
    //Once everything is set, this while loop runs over and over again. The stateMachine funcion finds the function that has to be executed and executes it.
    while(1) {
        stateMachine();
    }

}

void switchState()
{
    //this switches between the two states
    state++;
    state = state%2;
    
    //the number of steps has to start at 0 everytime one switches between states
    stateStep = 0;
    
    //the switch set everything up to start the different modes
    switch(state) {
        //demo mode
        case 0:
        //stop the timer used for moving with EMG
            tEMGMove.stop();
        //sets the number of steps in this mode
            noStateSteps = 4;
            break;
        //EMG mode
        case 1:
        //sets the number of steps in this mode
            noStateSteps = 7;
        //empties the vectors with info needed for PID control
            fill(pidInfo1.begin(), pidInfo1.begin()+4, 0);
            fill(pidInfo2.begin(), pidInfo2.begin()+4, 0);
        //resets and starts the timer used for moving with EMG
            tEMGMove.reset();
            tEMGMove.start();
            break;
    }
}

void switchStep()
{
    //keeps track of the step the machine is at within it's mode 
    stateStep++;
    stateStep = stateStep%noStateSteps;
}

void stateMachine()
{
    //switch between the modes
    switch (state) {
        case 0:
            demoState();
            break;
        case 1:
            EMGState();
            break;
    }
}

void demoState()
{
    //switches between the steps within the demo mode
    switch (stateStep) {
        case 0:
        //waits
            waiting();
            break;
        case 1:
        //calibrates and moves on to the next step
            calibrateMotors();
            stateStep++;
            break;
        case 2:
        //waits
            waiting();
            break;
        case 3:
        //plays the demo
            playDemo();
            break;
    }
}

void EMGState()
{
    //switches between the steps within the EMG mode
    pc.printf("EMG %d\n\r", stateStep);
    switch(stateStep) {
        case 0:
        //waits
            waiting();
            break;
        case 1:
        //calibrates the motors and moves on to the next step
            calibrateMotors();
            stateStep++;
            break;
        case 2:
        //waits
            waiting();
            break;
        case 3:
        //calibrates the EMG and moves on to the next step
            EMGcalibration();
            stateStep++;
            break;
        case 4:
        //waits 
            waiting();
            break;
        case 5:
        //moves to the starting position for EMG
            moveToInitialPosition();
            xCurrent = 0;
            yCurrent = 20;
            break;
        case 6:
        //uses EMG signals to determin the speed of the end effector
            moveWithEMG();
            break;
    }
}

void waiting()
{
    //turns on an LED and makes sure that the motors stop moving
    r_led = 0;
    E1 =0;
    E2 =0;
    r_led = 1;
}

void calibrateMotors()
{
    //Calibrates the motors by moving them until they can't go further and then resets the encoders
    moveMotorToStop(&M1, &E1, &Enc1, -0.1);
    moveMotorToStop(&M2, &E2, &Enc2, 0.2);
    moveMotorToStop(&M1, &E1, &Enc1, -0.1);
    Enc2.reset();
    Enc1.reset();
}

void playDemo()
{
    //moves the end effector along the lines of a rectangle
    moveAlongPath(10, 30, -10, 30, 3);
    moveAlongPath(-10, 30, -10, 20, 3);
    moveAlongPath(-10, 20, 10, 20, 3);
    moveAlongPath(10, 20, 10, 30, 3);
}

void EMGcalibration()
{
    Timer tijd;                                 // Timer for the duration of the calibration stage
    tijd.start();                               // Start of the timer
    ticker_calibration.attach(&sample, 0.002);  // Ticker for reading EMG-signals is attached to the function that filters the emg-signals
    do {
        if(emg1_cal < emg1_filtered) {                  // Initial calibration value is compaired to the current EMG value.
            emg1_cal = emg1_filtered ;                  // When the current EMG value is higher than the calibration value, then the calibration value becomes the current EMG value.
        }
        if(emg2_cal < emg2_filtered) {                  // The same is true for the second EMG signal
            emg2_cal = emg2_filtered ;
        }
    } while(tijd.read()<10);                            // After ten seconds the calibration of the EMG-signals stop
}


void moveToInitialPosition()
{
    //Moves the endeffector to it's initial position of (0, 20)
    
    double xStart = 0;
    double yStart = 20;

    double rot1 = calcRot1(xStart, yStart);
    double rot2 = calcRot2(xStart, yStart);

    moveMotorToPoint(&M1, &E1, &Enc1, initRot2, 1, -rot2);
    moveMotorToPoint(&M2, &E2, &Enc2, initRot1, -1, rot1);
}

void moveWithEMG()
{
    if(emg1_filtered >= 0.5*emg1_cal) {                     // When the current value of EMG1 is higher than half of the calibration value,
        speedy = 3;                                         // the speed in Y direction becomes 3. Otherwise the speed stays 0.
    } else {
        speedy = 0;
    }

    if(emg2_filtered >= 0.5*emg2_cal) {                     // When the current value of EMG2 is higher than half of the calibration value,
        speedx = 3;                                         // the speed in X direction becomes 3. Otherwise the speed stays 0.
    } else {
        speedx = 0;
    }
    
    
    //caluculates the position at wich the en effector should be given a certain speed
    xCurrent = xCurrent + xDir*speedx*0.01;
    yCurrent = yCurrent + yDir*speedy*0.01;
    
    //calculates the desired rotation of the motor and moves it there.
    double rot2 = calcRot2(xCurrent, yCurrent);
    double rot1 = calcRot1(xCurrent, yCurrent);
    moveMotorContinuously(&M1, &E1, &Enc1, initRot2, 1, &pidInfo2, &tEMGMove, -rot2);
    moveMotorContinuously(&M2, &E2, &Enc2, initRot1, -1, &pidInfo1, &tEMGMove, rot1);
    wait(0.01);
}

/*
takes as an input:
*M = a pointer to the pin controling the direction of the rotation of the motor
*E = a pointer to the pin controling the speed of the motor
*Enc = a pointer to the encoder of the motor
speed = a double to set the speed at which the motor has to rotate 
*/
void moveMotorToStop(DigitalOut *M, PwmOut *E, QEI *Enc, double speed)
{
    Timer t;

    double MotorPWM;

    // defines (and sets) the variables for the cuurent position of the motor, it's previous position and it's velocity
    double posC;
    double posP = Enc->getPulses()/(32*131.25);
    double vel = 0;

    //defines a cunter to keep track of when the motor is standing still
    int hasNotMoved = 0;

    t.start();
    do {
        //sets the duty cycle for the motor and prevents the motor of movig too fast due to the weight attached to the and effector.
        MotorPWM = speed - vel*0.7;
        
        //makes sure the motor moves in he right direction with the right speed
        if(MotorPWM > 0) {
            *M = 0;
            *E = MotorPWM;
        } else {
            *M = 1;
            *E = -MotorPWM;
        }
        wait(0.01);
        
        //reads a new value for the position of the motor
        posC = Enc->getPulses()/(32*131.25);
        //calculates the speed of the motor
        vel = (posC - posP)/t.read();
        t.reset();
        posP = posC;
        //keeps track of the consecutive number of times the motor did not move
        if(abs(vel) > 0.001) {
            hasNotMoved = 0;
        } else {
            hasNotMoved++;
        }
    } while(hasNotMoved < 6);
    //stops the motors from moving
    *E = 0;
}
/*
takes as input
xStart = a double to define the x-coordinate of the end effector at the start of the path
yStart = a double to define the y-coordinate of the end effector at the start of the path 
xEnd = a double to define the x-coordinate of the end effector at the end of the path
yEnd = a double to define the y-coordinate of the end effector at the end of the path 
speed = a double to set the speed at which the end effector should move along the path
*/

void moveAlongPath(double xStart, double yStart, double xEnd, double yEnd, double speed)
{
    //define variables for motor rotations, a timer and a vector for the desired x and y coordinate of the end effector
    double rot1;
    double rot2;

    Timer t;

    vector<double> desiredLocation;
    
    //fill the pid Info vectors with zeros
    fill(pidInfo1.begin(), pidInfo1.begin()+4, 0);
    fill(pidInfo2.begin(), pidInfo2.begin()+4, 0);

    //calculate the length of the path along which the end effectr has to move
    double pathLength = sqrt(pow((xStart - xEnd), 2.0)+pow((yStart - yEnd), 2.0));

    //Calculate the rotation of the motors at the start of the path
    rot1 = calcRot1(xStart, yStart);
    rot2 = calcRot2(xStart, yStart);

    //start the timer
    t.start();
    //move along the path for as long asit should take to do so idealy
    while(pathLength > speed * t.read()) {
        //find the desired location of the end effector
        findDesiredLocation(xStart, yStart, xEnd, yEnd, speed, &t, &desiredLocation);
        
        //find the desired rotation of the motors given the desired location of the end effector
        rot1 = calcRot1(desiredLocation.at(0), desiredLocation.at(1));
        rot2 = calcRot2(desiredLocation.at(0), desiredLocation.at(1));
        //Give the motors a certain speed given the rotation they should end up in
        moveMotorContinuously(&M1, &E1, &Enc1, initRot2, 1, &pidInfo2, &t, -rot2);
        moveMotorContinuously(&M2, &E2, &Enc2, initRot1, -1, &pidInfo1, &t, rot1);
        wait(0.01);
    }

}


/*
takes as input:
xDes = the x-coordinate desired loaction of the end effector
yDes = the y-coordinate desired loaction of the end effector
*/
double calcRot1(double xDes, double yDes)
{
    double alpha = atan(yDes/xDes);
    //makes sure the right solution for alpha was given (there are two solutions and only the positive one is usefull)
    if(alpha < 0) {
        alpha = alpha+Pi;
    }
    //returns the desired rotation of motor 1
    return gearRatio*((alpha - 0.5*(Pi - acos((pow(xDes, 2.0) + pow(yDes, 2.0) - 2*pow(20.0, 2.0))/(-2*pow(20.0, 2.0)))))/(2*Pi));
}

/*
takes as input:
xDes = the x-coordinate desired loaction of the end effector
yDes = the y-coordinate desired loaction of the end effector
*/
double calcRot2(double xDes, double yDes)
{
    double alpha = atan(yDes/xDes);
    //makes sure the right solution for alpha was given (there are two solutions and only the positive one is usefull)
    if(alpha < 0) {
        alpha = alpha+Pi;
    }
    //returns the desired rotation of motor 2
    return gearRatio*((alpha + 0.5*(Pi - acos((pow(xDes, 2.0) + pow(yDes, 2.0) - 2*pow(20.0, 2.0))/(-2*pow(20.0, 2.0)))))/(2*Pi));
}

/*
takes as input:
xStart = a double to define the x-coordinate of the end effector at the start of the path
yStart = a double to define the y-coordinate of the end effector at the start of the path 
xEnd = a double to define the x-coordinate of the end effector at the end of the path
yEnd = a double to define the y-coordinate of the end effector at the end of the path 
speed = a double to set the speed at which the end effector should move along the path
*t = a pointer to a time to keep track of the time that passed since the operation is started
*desiredLocation = a pointer to a vector of doubles in which the desired x and y coordinates can be stored 
*/
void findDesiredLocation(double xStart, double yStart, double xEnd, double yEnd, double speed, Timer *t, vector<double> *desiredLocation)
{
    //calculate the length of the path
    double pathLength = sqrt(pow((xStart - xEnd), 2.0)+pow((yStart - yEnd), 2.0));
    //calculate how far the end effector should have moved
    double traveledDistance = speed * t->read();
    
    //calculate the desired location of the end effector and store it in the vector
    double ratio = traveledDistance/pathLength;
    desiredLocation->clear();
    desiredLocation->push_back(xStart + (xEnd - xStart)*ratio);
    desiredLocation->push_back(yStart + (yEnd - yStart)*ratio);

}

/*
takes as input: 
*M = a pointer to the pin controling the direction of the rotation of the motor
*E = a pointer to the pin controling the speed of the motor
*Enc = a pointer to the encoder of the motor
initRot = the rotation of the motor after it was calibrated
dir = the direction in which the motor has to move in order to make the error smaller
*pidInfo = a pointer to a vector in which values are stored that are needed for the next time this function runs
*t = a pointer to a time to keep track of the time that passed since the operation is started
rotDes = the desired rotation of the motor
*/
void moveMotorContinuously(DigitalOut *M, PwmOut *E, QEI *Enc, double initRot, double dir, vector<double> *pidInfo, Timer *t, double rotDes)
{
    //defining and setting a variable to store the K values of the PID controler
    double Kp = 61;
    double Ki = 1;
    double Kd = 7;

    //defining and setting a variable to store the current rotation of the motor
    double rotC = Enc->getPulses()/(32*131.25) + initRot;

    //defining and setting a variable to store the current current error
    double pErrorC = rotDes - rotC;

    //defining and setting a variable to store the time that has passed since the start of the opperation
    double tieme = t->read();
    //defining and setting a variable to store delta t, using the passed time that was stored in the pidInfo vector
    double dt = tieme - pidInfo->at(2);

    //defining and setting a variable to store the integration of the error using it's previous value that was stored in the pidInfo vector
    double iError = pidInfo->at(1) + pErrorC*dt;
    //defining and setting a variable to store the derivative of the error using the previous value of the error that was stored in the pidInfo vector
    double dError = (pErrorC - pidInfo->at(0))/dt;

    //defining and setting a variable to store duty cycle of the motor and making sure it moves in the right direction
    double MotorPWM = (pErrorC*Kp + iError*Ki + dError*Kd)*dir;
    //set the PWM signal that pin E and the signal that pin M send to the motor shield
    if(MotorPWM > 0) {
        *M = 0;
        *E = MotorPWM;
    } else {
        *M = 1;
        *E = -MotorPWM;
    }
    //clear the pidInfo vector and fill it with te new values
    pidInfo->clear();
    pidInfo->push_back(pErrorC);
    pidInfo->push_back(iError);
    pidInfo->push_back(tieme);
    pidInfo->push_back(Enc->getPulses()/(32*131.25) + initRot);
}

void sample()
{
    emg1_filtered = FilterDesign(emg0.read());                  // The first emg signal is put in the filter and the filtered values are saved
    emg2_filtered = FilterDesign(emg1.read());                  // The second emg signal is put in the filter and the filtered values are saved
    
    /* Set the sampled emg values in channel 0 (the first channel) and 1 (the second channel) in the 'HIDScope' instance named 'scope' */
    scope.set(0, emg1_filtered ) ;
    scope.set(1, emg2_filtered );
    
    scope.send();                   // Finally, the values are send and can be used in other functions.
}

/*
takes as input:
*xStart = a pointer to a double that keeps track of the x-cordinate at which the end effector should start
*yStart = a pointer to a double that keeps track of the y-cordinate at which the end effector should start
xSpeed = the desired speeed in the x direction
ySpeed = the desired seed in the y direction
*/
void moveWithSpeed(double *xStart, double *yStart, double xSpeed, double ySpeed)
{
    //define a timer
    Timer t;

    //define and fill the vectors with info for the PID controler
    vector<double> pidInfo1 (4);
    vector<double> pidInfo2 (4);
    fill(pidInfo1.begin(), pidInfo1.begin()+4, 0);
    fill(pidInfo2.begin(), pidInfo2.begin()+4, 0);

    //define and set variables to store the current x and y coordinates in
    double xC = *xStart;
    double yC = *yStart;
    
    //define variables to store the desired rotations of the motors in
    double rot1;
    double rot2;

    //define variables to store previously passed time, currently passed time and the delta t in
    double tiemeP;
    double tiemeC;
    double dt;

    //start the timer
    t.start();

    //set the previously passed time
    tiemeP = t.read();
    
    //move the robot with a certain speed for in a certain direction for 1/10 of a second
    while(t.read() < 0.1) {
        //read the passes time
        tiemeC = t.read();
        //determine delta t
        dt = tiemeC - tiemeP;
        //determine the desired loaction of the end effector given a certain initial position and a speed
        xC = xC+xSpeed*dt;
        yC = yC+ySpeed*dt;
        //determine the desired rotaion of the motors
        rot1 = calcRot1(xC, yC);
        rot2 = calcRot2(xC, yC);
        //give the motors a certain speed to move with to reach their desired rotation
        moveMotorContinuously(&M1, &E1, &Enc1, initRot2, 1, &pidInfo2, &t, -rot2);
        moveMotorContinuously(&M2, &E2, &Enc2, initRot1, -1, &pidInfo1, &t, rot1);
        //set the previously passed time
        tiemeP = tiemeC;
        //give the motors some time to move
        wait(0.01);
    }
    //set the the current location of the motors as the desired location of the motors
    *xStart = xC;
    *yStart = yC;
}
/*
takes as input: 
*M = a pointer to the pin controling the direction of the rotation of the motor
*E = a pointer to the pin controling the speed of the motor
*Enc = a pointer to the encoder of the motor
initRot = the rotation of the motor after it was calibrated
dir = the direction in which the motor has to move in order to make the error smaller
rotDes = the desired rotation of the motor
*/

void moveMotorToPoint(DigitalOut *M, PwmOut *E, QEI *Enc, double initRot, double dir, double rotDes)
{
    //define and set variable to store K values
    double Kp = 2;
    double Ki = 0;
    double Kd = 0.1;

    //define (and set) variable to store errors
    double pErrorC;
    double pErrorP = 0;
    double iError = 0;
    double dError;

    //define and set a variable to store the the current ration of the motor
    double rotC = Enc->getPulses()/(32*131.25) + initRot;
    
    //define a variable to store the dury cycle of the E pin
    double MotorPWM;

    //define a timer
    Timer t;
    
    //define and set a variable to store the time that has passed
    double tieme = 0;

    //start the timer
    t.start();
    do {
        //set the previous error
        pErrorP = pErrorC;
        //read the current roation of the motor
        rotC = Enc->getPulses()/(32*131.25) + initRot;
        //determine the current error
        pErrorC = rotDes - rotC;
        //read the time that has passed and reset it
        tieme = t.read();
        t.reset();
        //determine the integration of the error
        iError = iError + pErrorC*tieme;
        //determine the derivitaive of the error
        dError = (pErrorC - pErrorP)/tieme;
        
        //determine the dutycycle of the E pin
        MotorPWM = (pErrorC*Kp + iError*Ki + dError*Kd)*dir;
        
        //Move the motor in the right direction with the right speed
        if(MotorPWM > 0) {
            *M = 0;
            *E = MotorPWM;
        } else {
            *M = 1;
            *E = -MotorPWM;
        }
        //give the motor some time to move
        wait(0.01);
    //keep doing this until the error is very small and the motor is hardly moving
    } while (abs(MotorPWM) > 0.13|| abs(dError > 0.01));; //pErrorC > 0.02 || pErrorC < -0.02 ||dError > 0.01 || dError < -0.01);
    //stop the movement of the arm
    *E = 0;
    //stop the timer
    t.stop();
}

void flipx()
{
    //flip the x direction of the end effector in EMG mode
    xDir = -xDir;

}

void flipy()
{
    //flip the y direction of the end effector in EMG mode 
    yDir = -yDir;
}