#ifndef _RECEIVER_IR_H_
#define _RECEIVER_IR_H_
 
#include <mbed.h>
 
#include "RemoteIR.h"
 
/**
 * IR receiver class.
 */
class ReceiverIR {
public:
 
    /**
     * Constructor.
     *
     * @param rxpin Pin for receive IR signal.
     */
    explicit ReceiverIR::ReceiverIR(PinName rxpin, float _speed=0.7, PinName _pwma=D6, PinName _pwmb=D5, PinName _ain0=PA_0, PinName _ain1=PA_1, PinName _bin0=PB_0, PinName _bin1=PA_4);
    /**
     * Destructor.
     */
    ~ReceiverIR();
 
    /**
     * State.
     */
    typedef enum {
        Idle,
        Receiving,
        Received
    } State;
    
    /**
     * Get state.
     *
     * @return Current state.
     */
    State getState();
    
    /**
     * Get data.
     *
     * @param format Pointer to format.
     * @param buf Buffer of a data.
     * @param bitlength Bit length of the buffer.
     *
     * @return Data bit length.
     */
    int getData(RemoteIR::Format *format, uint8_t *buf, int bitlength);
    void move(uint8_t *buf);
    
    float Speed_L, Speed_R;
    float init_sp_l, init_sp_r;
    volatile int cal_start, cal_stop;
    float kp;
    float kd;
    int ki;
    float max;
    float standard;
    
//private:
    
    typedef struct {
        RemoteIR::Format format;
        int bitcount;
        uint8_t buffer[64];
    } data_t;
    
    typedef struct {
        State state;
        int c1;
        int c2;
        int c3;
        int d1;
        int d2;
    } work_t;
    
    InterruptIn evt;    /**< Interrupt based input for input. */
    Timer timer;        /**< Timer for WDT. */
    Ticker ticker;      /**< Tciker for tick. */
    Timeout timeout;    /**< Timeout for tail. */
    
    PwmOut pwma;
    PwmOut pwmb;
    DigitalOut ain0;
    DigitalOut ain1;
    DigitalOut bin0;
    DigitalOut bin1;
 
    data_t data;
    work_t work;
    
    
    
    void init_state(void);
 
    void isr_wdt(void);
    void isr_fall(void);
    void isr_rise(void);
    void right(void);
    void left(void);
    void forward(void);
    void backward(void);
    void stop(void);
    void right_turn(void);
    void left_turn(void);
    void s_right_turn(void);
    void s_left_turn(void);    
    
    void speedup_l(void);
    void speeddown_l (void);
    void speedup_r(void);
    void speeddown_r(void);
    
    void speed_l(float sp_l);
    void speed_r(float sp_r);
    
    /**
     * ISR timeout for tail detection.
     */
    void isr_timeout(void);
    
};
 
#endif