/*******************************************************************************
Copyright © 2016, STMicroelectronics International N.V.
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
	* Redistributions of source code must retain the above copyright
	  notice, this list of conditions and the following disclaimer.
	* Redistributions in binary form must reproduce the above copyright
	  notice, this list of conditions and the following disclaimer in the
	  documentation and/or other materials provided with the distribution.
	* Neither the name of STMicroelectronics nor the
	  names of its contributors may be used to endorse or promote products
	  derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND
NON-INFRINGEMENT OF INTELLECTUAL PROPERTY RIGHTS ARE DISCLAIMED.
IN NO EVENT SHALL STMICROELECTRONICS INTERNATIONAL N.V. BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

/**
 * @file VL53L0X_def.h
 *
 * @brief Type definitions for VL53L0X API.   */

#ifndef _VL53L0X_DEF_H_
#define _VL53L0X_DEF_H_

#ifdef __cplusplus
extern "C" {
#endif

/** @defgroup VL53L0X_globaldefine_group VL53L0X Defines
 *	@brief	  VL53L0X Defines
 *	@{    */
/** PAL SPECIFICATION major version */
#define VL53L0X10_SPECIFICATION_VER_MAJOR   1
/** PAL SPECIFICATION minor version */
#define VL53L0X10_SPECIFICATION_VER_MINOR   2
/** PAL SPECIFICATION sub version */
#define VL53L0X10_SPECIFICATION_VER_SUB	   	7
/** PAL SPECIFICATION sub version */
#define VL53L0X10_SPECIFICATION_VER_REVISION 1440

/** VL53L0X PAL IMPLEMENTATION major version */
#define VL53L0X10_IMPLEMENTATION_VER_MAJOR	1
/** VL53L0X PAL IMPLEMENTATION minor version */
#define VL53L0X10_IMPLEMENTATION_VER_MINOR	0
/** VL53L0X PAL IMPLEMENTATION sub version */
#define VL53L0X10_IMPLEMENTATION_VER_SUB	9
/** VL53L0X PAL IMPLEMENTATION sub version */
#define VL53L0X10_IMPLEMENTATION_VER_REVISION	3673

/** PAL SPECIFICATION major version */
#define VL53L0X_SPECIFICATION_VER_MAJOR	 	1
/** PAL SPECIFICATION minor version */
#define VL53L0X_SPECIFICATION_VER_MINOR	 	2
/** PAL SPECIFICATION sub version */
#define VL53L0X_SPECIFICATION_VER_SUB	 	7
/** PAL SPECIFICATION sub version */
#define VL53L0X_SPECIFICATION_VER_REVISION 	1440

/** VL53L0X PAL IMPLEMENTATION major version */
#define VL53L0X_IMPLEMENTATION_VER_MAJOR	1
/** VL53L0X PAL IMPLEMENTATION minor version */
#define VL53L0X_IMPLEMENTATION_VER_MINOR	1
/** VL53L0X PAL IMPLEMENTATION sub version */
#define VL53L0X_IMPLEMENTATION_VER_SUB	  	21
/** VL53L0X PAL IMPLEMENTATION sub version */
#define VL53L0X_IMPLEMENTATION_VER_REVISION	4823

/** FixPoint1616_t is used where fractional values are needed
 * Given a floating point value f it's .16 bit point is (int)(f*(1<<16))*/
typedef uint32_t FixPoint1616_t;
#define VL53L0X_FP1616TOFP97(Value) (uint16_t)((Value>>9)&0xFFFF)
#define VL53L0X_FP97TOFP1616(Value) (FixPoint1616_t)(Value<<9)
#define VL53L0X_FP1616TOFP88(Value) (uint16_t)((Value>>8)&0xFFFF)
#define VL53L0X_FP88TOFP1616(Value) (FixPoint1616_t)(Value<<8)
#define VL53L0X_FP1616TOFP412(Value) (uint16_t)((Value>>4)&0xFFFF)
#define VL53L0X_FP412TOFP1616(Value) (FixPoint1616_t)(Value<<4)
#define VL53L0X_FP1616TOFP313(Value) (uint16_t)((Value>>3)&0xFFFF)
#define VL53L0X_FP313TOFP1616(Value) (FixPoint1616_t)(Value<<3)
#define VL53L0X_FP1616TOFP08(Value) (uint8_t)((Value>>8)&0x00FF)
#define VL53L0X_FP08TOFP1616(Value) (FixPoint1616_t)(Value<<8)
#define VL53L0X_FP1616TOFP53(Value) (uint8_t)((Value>>13)&0x00FF)
#define VL53L0X_FP53TOFP1616(Value) (FixPoint1616_t)(Value<<13)
#define VL53L0X_FP1616TOFP102(Value) (uint16_t)((Value>>14)&0x0FFF)
#define VL53L0X_FP102TOFP1616(Value) (FixPoint1616_t)(Value<<12)
#define VL53L0X_MAKEUINT16(lsb, msb) (uint16_t)((((uint16_t)msb)<<8) + (uint16_t)lsb)


/**  The device model ID  */ 
#define IDENTIFICATION_MODEL_ID      0x000

#define VL53L0X_OsDelay(...) wait_ms(2) // 2 msec delay. can also use wait(float secs)/wait_us(int)

/** Maximum buffer size to be used in i2c */
#define VL53L0X_MAX_I2C_XFER_SIZE   64 
  
#define VL53L0X_COPYSTRING(str, ...) strcpy(str, ##__VA_ARGS__)

/*  Speed of light in um per 1E-10 Seconds  */
#define VL53L0X_SPEED_OF_LIGHT_IN_AIR 		2997 

/** default device address */
#define VL53L0X_DEFAULT_ADDRESS		0x52 /* (8-bit) */

#define VL53L0X_DEFAULT_MAX_LOOP 			2000

#define VL53L0X_MAX_STRING_LENGTH 			32

/* equivalent to a range sigma of 655.35mm */
#define VL53L0X_SIGMA_ESTIMATE_MAX_VALUE    65535

#define REF_ARRAY_SPAD_0  0
#define REF_ARRAY_SPAD_5  5
#define REF_ARRAY_SPAD_10 10

static uint32_t refArrayQuadrants[4] = {REF_ARRAY_SPAD_10, REF_ARRAY_SPAD_5,
                                 REF_ARRAY_SPAD_0, REF_ARRAY_SPAD_5 };

#define  VL53L0X_STRING_DEVICE_INFO_NAME          "VL53L0X cut1.0"
#define  VL53L0X_STRING_DEVICE_INFO_NAME_TS0      "VL53L0X TS0"
#define  VL53L0X_STRING_DEVICE_INFO_NAME_TS1      "VL53L0X TS1"
#define  VL53L0X_STRING_DEVICE_INFO_NAME_TS2      "VL53L0X TS2"
#define  VL53L0X_STRING_DEVICE_INFO_NAME_ES1      "VL53L0X ES1 or later"
#define  VL53L0X_STRING_DEVICE_INFO_TYPE          "VL53L0X"

/* sensor operating modes */
typedef enum {
    range_single_shot_polling = 1,
    range_continuous_polling,
    range_continuous_interrupt,
    range_continuous_polling_low_threshold,
    range_continuous_polling_high_threshold,
    range_continuous_polling_out_of_window,
    range_continuous_interrupt_low_threshold,
    range_continuous_interrupt_high_threshold,
    range_continuous_interrupt_out_of_window,
} OperatingMode;

/** @defgroup VL53L0X_DeviceError_group Device Error
 *  @brief Device Error code
 *
 *  This enum is Device specific it should be updated in the implementation
 *  Use @a VL53L0X_GetStatusErrorString() to get the string.
 *  It is related to Status Register of the Device.
 *  @{   */
typedef uint8_t VL53L0X_DeviceError;
#define VL53L0X_DEVICEERROR_NONE                        ((VL53L0X_DeviceError) 0) /*!< 0  NoError  */
#define VL53L0X_DEVICEERROR_VCSELCONTINUITYTESTFAILURE  ((VL53L0X_DeviceError) 1)
#define VL53L0X_DEVICEERROR_VCSELWATCHDOGTESTFAILURE    ((VL53L0X_DeviceError) 2)
#define VL53L0X_DEVICEERROR_NOVHVVALUEFOUND             ((VL53L0X_DeviceError) 3)
#define VL53L0X_DEVICEERROR_MSRCNOTARGET                ((VL53L0X_DeviceError) 4)
#define VL53L0X_DEVICEERROR_SNRCHECK                    ((VL53L0X_DeviceError) 5)
#define VL53L0X_DEVICEERROR_RANGEPHASECHECK             ((VL53L0X_DeviceError) 6)
#define VL53L0X_DEVICEERROR_SIGMATHRESHOLDCHECK         ((VL53L0X_DeviceError) 7)
#define VL53L0X_DEVICEERROR_TCC                         ((VL53L0X_DeviceError) 8)
#define VL53L0X_DEVICEERROR_PHASECONSISTENCY            ((VL53L0X_DeviceError) 9)
#define VL53L0X_DEVICEERROR_MINCLIP                     ((VL53L0X_DeviceError) 10)
#define VL53L0X_DEVICEERROR_RANGECOMPLETE               ((VL53L0X_DeviceError) 11)
#define VL53L0X_DEVICEERROR_ALGOUNDERFLOW               ((VL53L0X_DeviceError) 12)
#define VL53L0X_DEVICEERROR_ALGOOVERFLOW                ((VL53L0X_DeviceError) 13)
#define VL53L0X_DEVICEERROR_RANGEIGNORETHRESHOLD        ((VL53L0X_DeviceError) 14)
/** @} end of VL53L0X_DeviceError_group */


/* PAL ERROR strings */
#define  VL53L0X_STRING_ERROR_NONE 			"No Error"
#define  VL53L0X_STRING_ERROR_CALIBRATION_WARNING "Calibration Warning Error"
#define  VL53L0X_STRING_ERROR_MIN_CLIPPED 	"Min clipped error"
#define  VL53L0X_STRING_ERROR_UNDEFINED 	"Undefined error"
#define  VL53L0X_STRING_ERROR_INVALID_PARAMS "Invalid parameters error"
#define  VL53L0X_STRING_ERROR_NOT_SUPPORTED "Not supported error"
#define  VL53L0X_STRING_ERROR_RANGE_ERROR 	"Range error"
#define  VL53L0X_STRING_ERROR_TIME_OUT 		"Time out error"
#define  VL53L0X_STRING_ERROR_MODE_NOT_SUPPORTED "Mode not supported error"
#define  VL53L0X_STRING_ERROR_BUFFER_TOO_SMALL "Buffer too small"
#define  VL53L0X_STRING_ERROR_GPIO_NOT_EXISTING "GPIO not existing"
#define  VL53L0X_STRING_ERROR_GPIO_FUNCTIONALITY_NOT_SUPPORTED "GPIO funct not supported"
#define  VL53L0X_STRING_ERROR_INTERRUPT_NOT_CLEARED "Interrupt not Cleared"
#define  VL53L0X_STRING_ERROR_CONTROL_INTERFACE "Control Interface Error"
#define  VL53L0X_STRING_ERROR_INVALID_COMMAND	"Invalid Command Error"
#define  VL53L0X_STRING_ERROR_DIVISION_BY_ZERO	"Division by zero Error"
#define  VL53L0X_STRING_ERROR_REF_SPAD_INIT		"Reference Spad Init Error"
#define  VL53L0X_STRING_UNKNOW_ERROR_CODE 		"Unknown Error Code"

/* Range Status */
#define  VL53L0X_STRING_RANGESTATUS_NONE                 "No Update"
#define  VL53L0X_STRING_RANGESTATUS_RANGEVALID           "Range Valid"
#define  VL53L0X_STRING_RANGESTATUS_SIGMA                "Sigma Fail"
#define  VL53L0X_STRING_RANGESTATUS_SIGNAL               "Signal Fail"
#define  VL53L0X_STRING_RANGESTATUS_MINRANGE             "Min Range Fail"
#define  VL53L0X_STRING_RANGESTATUS_PHASE                "Phase Fail"
#define  VL53L0X_STRING_RANGESTATUS_HW                   "Hardware Fail"

/* Range Status */
#define  VL53L0X_STRING_STATE_POWERDOWN         "POWERDOWN State"
#define  VL53L0X_STRING_STATE_WAIT_STATICINIT 	"Wait for staticinit State"
#define  VL53L0X_STRING_STATE_STANDBY           "STANDBY State"
#define  VL53L0X_STRING_STATE_IDLE                    "IDLE State"
#define  VL53L0X_STRING_STATE_RUNNING                 "RUNNING State"
#define  VL53L0X_STRING_STATE_UNKNOWN                 "UNKNOWN State"
#define  VL53L0X_STRING_STATE_ERROR                   "ERROR State"

/* Device Specific */
#define  VL53L0X_STRING_DEVICEERROR_NONE              "No Update"
#define  VL53L0X_STRING_DEVICEERROR_VCSELCONTINUITYTESTFAILURE "VCSEL Continuity Test Failure"
#define  VL53L0X_STRING_DEVICEERROR_VCSELWATCHDOGTESTFAILURE "VCSEL Watchdog Test Failure"
#define  VL53L0X_STRING_DEVICEERROR_NOVHVVALUEFOUND 	"No VHV Value found"
#define  VL53L0X_STRING_DEVICEERROR_MSRCNOTARGET 		"MSRC No Target Error"
#define  VL53L0X_STRING_DEVICEERROR_SNRCHECK 			"SNR Check Exit"
#define  VL53L0X_STRING_DEVICEERROR_RANGEPHASECHECK 	"Range Phase Check Error"
#define  VL53L0X_STRING_DEVICEERROR_SIGMATHRESHOLDCHECK "Sigma Threshold Check Error"
#define  VL53L0X_STRING_DEVICEERROR_TCC 			"TCC Error"
#define  VL53L0X_STRING_DEVICEERROR_PHASECONSISTENCY 	"Phase Consistency Error"
#define  VL53L0X_STRING_DEVICEERROR_MINCLIP 			"Min Clip Error"
#define  VL53L0X_STRING_DEVICEERROR_RANGECOMPLETE 		"Range Complete"
#define  VL53L0X_STRING_DEVICEERROR_ALGOUNDERFLOW 		"Range Algo Underflow Error"
#define  VL53L0X_STRING_DEVICEERROR_ALGOOVERFLOW 		"Range Algo Overlow Error"
#define  VL53L0X_STRING_DEVICEERROR_RANGEIGNORETHRESHOLD "Range Ignore Threshold Error"
#define  VL53L0X_STRING_DEVICEERROR_UNKNOWN 			"Unknown error code"

/* Check Enable */
#define  VL53L0X_STRING_CHECKENABLE_SIGMA_FINAL_RANGE 		"SIGMA FINAL RANGE"
#define  VL53L0X_STRING_CHECKENABLE_SIGNAL_RATE_FINAL_RANGE "SIGNAL RATE FINAL RANGE"
#define  VL53L0X_STRING_CHECKENABLE_SIGNAL_REF_CLIP 		"SIGNAL REF CLIP"
#define  VL53L0X_STRING_CHECKENABLE_RANGE_IGNORE_THRESHOLD 	"RANGE IGNORE THRESHOLD"
#define  VL53L0X_STRING_CHECKENABLE_SIGNAL_RATE_MSRC 		"SIGNAL RATE MSRC"
#define  VL53L0X_STRING_CHECKENABLE_SIGNAL_RATE_PRE_RANGE 	"SIGNAL RATE PRE RANGE"

/* Sequence Step */
#define  VL53L0X_STRING_SEQUENCESTEP_TCC                   "TCC"
#define  VL53L0X_STRING_SEQUENCESTEP_DSS                   "DSS"
#define  VL53L0X_STRING_SEQUENCESTEP_MSRC                  "MSRC"
#define  VL53L0X_STRING_SEQUENCESTEP_PRE_RANGE             "PRE RANGE"
#define  VL53L0X_STRING_SEQUENCESTEP_FINAL_RANGE           "FINAL RANGE"


/** @defgroup VL53L0X_CheckEnable_group Check Enable list
 *  @brief Check Enable code
 *
 *  Define used to specify the LimitCheckId.
 *  Use @a VL53L0X_GetLimitCheckInfo() to get the string.
 *  @{    */
#define VL53L0X_CHECKENABLE_SIGMA_FINAL_RANGE           0
#define VL53L0X_CHECKENABLE_SIGNAL_RATE_FINAL_RANGE     1
#define VL53L0X_CHECKENABLE_SIGNAL_REF_CLIP             2
#define VL53L0X_CHECKENABLE_RANGE_IGNORE_THRESHOLD      3
#define VL53L0X_CHECKENABLE_SIGNAL_RATE_MSRC            4
#define VL53L0X_CHECKENABLE_SIGNAL_RATE_PRE_RANGE       5
#define VL53L0X_CHECKENABLE_NUMBER_OF_CHECKS            6
/** @}  end of VL53L0X_CheckEnable_group */



/** @defgroup VL53L0X_DefineRegisters_group Define Registers
 *  @brief List of all the defined registers
 *  @{  */
#define VL53L0X_REG_SYSRANGE_START              	0x000
/** mask existing bit in #VL53L0X_REG_SYSRANGE_START*/
#define VL53L0X_REG_SYSRANGE_MODE_MASK          	0x0F
/** bit 0 in #VL53L0X_REG_SYSRANGE_START write 1 toggle state in
 * continuous mode and arm next shot in single shot mode */
#define VL53L0X_REG_SYSRANGE_MODE_START_STOP    	0x01
/** bit 1 write 0 in #VL53L0X_REG_SYSRANGE_START set single shot mode */
#define VL53L0X_REG_SYSRANGE_MODE_SINGLESHOT    	0x00
/** bit 1 write 1 in #VL53L0X_REG_SYSRANGE_START set back-to-back operation mode */
#define VL53L0X_REG_SYSRANGE_MODE_BACKTOBACK    	0x02
/** bit 2 write 1 in #VL53L0X_REG_SYSRANGE_START set timed operation  mode */
#define VL53L0X_REG_SYSRANGE_MODE_TIMED         	0x04
/** bit 3 write 1 in #VL53L0X_REG_SYSRANGE_START set histogram operation mode */
#define VL53L0X_REG_SYSRANGE_MODE_HISTOGRAM       			0x08
#define VL53L0X_REG_SYSTEM_THRESH_HIGH             			0x000C
#define VL53L0X_REG_SYSTEM_THRESH_LOW            			0x000E
#define VL53L0X_REG_SYSTEM_SEQUENCE_CONFIG					0x0001
#define VL53L0X_REG_SYSTEM_RANGE_CONFIG						0x0009
#define VL53L0X_REG_SYSTEM_INTERMEASUREMENT_PERIOD			0x0004
#define VL53L0X_REG_SYSTEM_INTERRUPT_CONFIG_GPIO    		0x000A
#define VL53L0X_REG_SYSTEM_INTERRUPT_GPIO_DISABLED			0x00
#define VL53L0X_REG_SYSTEM_INTERRUPT_GPIO_LEVEL_LOW			0x01
#define VL53L0X_REG_SYSTEM_INTERRUPT_GPIO_LEVEL_HIGH		0x02
#define VL53L0X_REG_SYSTEM_INTERRUPT_GPIO_OUT_OF_WINDOW		0x03
#define VL53L0X_REG_SYSTEM_INTERRUPT_GPIO_NEW_SAMPLE_READY	0x04
#define VL53L0X_REG_GPIO_HV_MUX_ACTIVE_HIGH          		0x0084
#define VL53L0X_REG_SYSTEM_INTERRUPT_CLEAR           		0x000B

/* Result registers */
#define VL53L0X_REG_RESULT_INTERRUPT_STATUS          		0x0013
#define VL53L0X_REG_RESULT_RANGE_STATUS              		0x0014
#define VL53L0X_REG_RESULT_CORE_PAGE                 		1
#define VL53L0X_REG_RESULT_CORE_AMBIENT_WINDOW_EVENTS_RTN   0x00BC
#define VL53L0X_REG_RESULT_CORE_RANGING_TOTAL_EVENTS_RTN    0x00C0
#define VL53L0X_REG_RESULT_CORE_AMBIENT_WINDOW_EVENTS_REF   0x00D0
#define VL53L0X_REG_RESULT_CORE_RANGING_TOTAL_EVENTS_REF    0x00D4
#define VL53L0X_REG_RESULT_PEAK_SIGNAL_RATE_REF             0x00B6
/* Algo register */
#define VL53L0X_REG_ALGO_PART_TO_PART_RANGE_OFFSET_MM       0x0028
#define VL53L0X_REG_I2C_SLAVE_DEVICE_ADDRESS                0x008a
/* Check Limit registers */
#define VL53L0X_REG_MSRC_CONFIG_CONTROL                     0x0060
#define VL53L0X_REG_PRE_RANGE_CONFIG_MIN_SNR               	0X0027
#define VL53L0X_REG_PRE_RANGE_CONFIG_VALID_PHASE_LOW        0x0056
#define VL53L0X_REG_PRE_RANGE_CONFIG_VALID_PHASE_HIGH       0x0057
#define VL53L0X_REG_PRE_RANGE_MIN_COUNT_RATE_RTN_LIMIT      0x0064
#define VL53L0X_REG_FINAL_RANGE_CONFIG_MIN_SNR              0X0067
#define VL53L0X_REG_FINAL_RANGE_CONFIG_VALID_PHASE_LOW      0x0047
#define VL53L0X_REG_FINAL_RANGE_CONFIG_VALID_PHASE_HIGH     0x0048
#define VL53L0X_REG_FINAL_RANGE_CONFIG_MIN_COUNT_RATE_RTN_LIMIT   0x0044
#define VL53L0X_REG_PRE_RANGE_CONFIG_SIGMA_THRESH_HI        0X0061
#define VL53L0X_REG_PRE_RANGE_CONFIG_SIGMA_THRESH_LO        0X0062

/* PRE RANGE registers */
#define VL53L0X_REG_PRE_RANGE_CONFIG_VCSEL_PERIOD           0x0050
#define VL53L0X_REG_PRE_RANGE_CONFIG_TIMEOUT_MACROP_HI      0x0051
#define VL53L0X_REG_PRE_RANGE_CONFIG_TIMEOUT_MACROP_LO      0x0052
#define VL53L0X_REG_SYSTEM_HISTOGRAM_BIN                    0x0081
#define VL53L0X_REG_HISTOGRAM_CONFIG_INITIAL_PHASE_SELECT   0x0033
#define VL53L0X_REG_HISTOGRAM_CONFIG_READOUT_CTRL           0x0055
#define VL53L0X_REG_FINAL_RANGE_CONFIG_VCSEL_PERIOD         0x0070
#define VL53L0X_REG_FINAL_RANGE_CONFIG_TIMEOUT_MACROP_HI    0x0071
#define VL53L0X_REG_FINAL_RANGE_CONFIG_TIMEOUT_MACROP_LO    0x0072
#define VL53L0X_REG_CROSSTALK_COMPENSATION_PEAK_RATE_MHz   0x0020
#define VL53L0X_REG_MSRC_CONFIG_TIMEOUT_MACROP              0x0046
#define VL53L0X_REG_SOFT_RESET_GO2_SOFT_RESET_N	            0x00bf
#define VL53L0X_REG_IDENTIFICATION_MODEL_ID                 0x00c0
#define VL53L0X_REG_IDENTIFICATION_REVISION_ID              0x00c2
#define VL53L0X_REG_OSC_CALIBRATE_VAL                       0x00f8
#define VL53L0X_REG_GLOBAL_CONFIG_VCSEL_WIDTH          		0x0032
#define VL53L0X_REG_GLOBAL_CONFIG_SPAD_ENABLES_REF_0   		0x00B0
#define VL53L0X_REG_GLOBAL_CONFIG_SPAD_ENABLES_REF_1   		0x00B1
#define VL53L0X_REG_GLOBAL_CONFIG_SPAD_ENABLES_REF_2   		0x00B2
#define VL53L0X_REG_GLOBAL_CONFIG_SPAD_ENABLES_REF_3  		0x00B3
#define VL53L0X_REG_GLOBAL_CONFIG_SPAD_ENABLES_REF_4   		0x00B4
#define VL53L0X_REG_GLOBAL_CONFIG_SPAD_ENABLES_REF_5   		0x00B5
#define VL53L0X_REG_GLOBAL_CONFIG_REF_EN_START_SELECT   	0x00B6
#define VL53L0X_REG_DYNAMIC_SPAD_NUM_REQUESTED_REF_SPAD 	0x004E /* 0x14E */
#define VL53L0X_REG_DYNAMIC_SPAD_REF_EN_START_OFFSET    	0x004F /* 0x14F */
#define VL53L0X_REG_POWER_MANAGEMENT_GO1_POWER_FORCE    	0x0080
#define VL53L0X_REG_VHV_CONFIG_PAD_SCL_SDA__EXTSUP_HV   	0x0089
#define VL53L0X_REG_ALGO_PHASECAL_LIM                   	0x0030 /* 0x130 */
#define VL53L0X_REG_ALGO_PHASECAL_CONFIG_TIMEOUT        	0x0030
/** @} VL53L0X_DefineRegisters_group */


static uint8_t DefaultTuningSettings[] = {

    /* update 02/11/2015_v36 */
    0x01, 0xFF, 0x01,    0x01, 0x00, 0x00,    0x01, 0xFF, 0x00,    0x01, 0x09, 0x00,
    0x01, 0x10, 0x00,    0x01, 0x11, 0x00,    0x01, 0x24, 0x01,    0x01, 0x25, 0xff,
    0x01, 0x75, 0x00,    0x01, 0xFF, 0x01,    0x01, 0x4e, 0x2c,    0x01, 0x48, 0x00,
    0x01, 0x30, 0x20,    0x01, 0xFF, 0x00,    0x01, 0x30, 0x09,    /* mja changed from 0x64. */
    0x01, 0x54, 0x00,    0x01, 0x31, 0x04,    0x01, 0x32, 0x03,    0x01, 0x40, 0x83,
    0x01, 0x46, 0x25,    0x01, 0x60, 0x00,    0x01, 0x27, 0x00,    0x01, 0x50, 0x06,
    0x01, 0x51, 0x00,    0x01, 0x52, 0x96,    0x01, 0x56, 0x08,    0x01, 0x57, 0x30,
    0x01, 0x61, 0x00,    0x01, 0x62, 0x00,    0x01, 0x64, 0x00,    0x01, 0x65, 0x00,
    0x01, 0x66, 0xa0,    0x01, 0xFF, 0x01,    0x01, 0x22, 0x32,    0x01, 0x47, 0x14,
    0x01, 0x49, 0xff,    0x01, 0x4a, 0x00,    0x01, 0xFF, 0x00,    0x01, 0x7a, 0x0a,
    0x01, 0x7b, 0x00,    0x01, 0x78, 0x21,    0x01, 0xFF, 0x01,    0x01, 0x23, 0x34,
    0x01, 0x42, 0x00,    0x01, 0x44, 0xff,    0x01, 0x45, 0x26,    0x01, 0x46, 0x05,
    0x01, 0x40, 0x40,    0x01, 0x0E, 0x06,    0x01, 0x20, 0x1a,    0x01, 0x43, 0x40,
    0x01, 0xFF, 0x00,    0x01, 0x34, 0x03,    0x01, 0x35, 0x44,    0x01, 0xFF, 0x01,
    0x01, 0x31, 0x04,    0x01, 0x4b, 0x09,    0x01, 0x4c, 0x05,    0x01, 0x4d, 0x04,
    0x01, 0xFF, 0x00,    0x01, 0x44, 0x00,    0x01, 0x45, 0x20,    0x01, 0x47, 0x08,
    0x01, 0x48, 0x28,    0x01, 0x67, 0x00,    0x01, 0x70, 0x04,    0x01, 0x71, 0x01,
    0x01, 0x72, 0xfe,    0x01, 0x76, 0x00,    0x01, 0x77, 0x00,    0x01, 0xFF, 0x01,
    0x01, 0x0d, 0x01,    0x01, 0xFF, 0x00,    0x01, 0x80, 0x01,    0x01, 0x01, 0xF8,
    0x01, 0xFF, 0x01,    0x01, 0x8e, 0x01,    0x01, 0x00, 0x01,    0x01, 0xFF, 0x00,
    0x01, 0x80, 0x00,    0x00, 0x00, 0x00
};

static uint8_t InterruptThresholdSettings[] = {
    /* Start of Interrupt Threshold Settings */
    0x1, 0xff, 0x00,    0x1, 0x80, 0x01,    0x1, 0xff, 0x01,    0x1, 0x00, 0x00,
    0x1, 0xff, 0x01,    0x1, 0x4f, 0x02,    0x1, 0xFF, 0x0E,    0x1, 0x00, 0x03,
    0x1, 0x01, 0x84,    0x1, 0x02, 0x0A,    0x1, 0x03, 0x03,    0x1, 0x04, 0x08,
    0x1, 0x05, 0xC8,    0x1, 0x06, 0x03,    0x1, 0x07, 0x8D,    0x1, 0x08, 0x08,
    0x1, 0x09, 0xC6,    0x1, 0x0A, 0x01,    0x1, 0x0B, 0x02,    0x1, 0x0C, 0x00,
    0x1, 0x0D, 0xD5,    0x1, 0x0E, 0x18,    0x1, 0x0F, 0x12,    0x1, 0x10, 0x01,
    0x1, 0x11, 0x82,    0x1, 0x12, 0x00,    0x1, 0x13, 0xD5,    0x1, 0x14, 0x18,
    0x1, 0x15, 0x13,    0x1, 0x16, 0x03,    0x1, 0x17, 0x86,    0x1, 0x18, 0x0A,
    0x1, 0x19, 0x09,    0x1, 0x1A, 0x08,    0x1, 0x1B, 0xC2,    0x1, 0x1C, 0x03,
    0x1, 0x1D, 0x8F,    0x1, 0x1E, 0x0A,    0x1, 0x1F, 0x06,    0x1, 0x20, 0x01,
    0x1, 0x21, 0x02,    0x1, 0x22, 0x00,    0x1, 0x23, 0xD5,    0x1, 0x24, 0x18,
    0x1, 0x25, 0x22,    0x1, 0x26, 0x01,    0x1, 0x27, 0x82,    0x1, 0x28, 0x00,
    0x1, 0x29, 0xD5,    0x1, 0x2A, 0x18,    0x1, 0x2B, 0x0B,    0x1, 0x2C, 0x28,
    0x1, 0x2D, 0x78,    0x1, 0x2E, 0x28,    0x1, 0x2F, 0x91,    0x1, 0x30, 0x00,
    0x1, 0x31, 0x0B,    0x1, 0x32, 0x00,    0x1, 0x33, 0x0B,    0x1, 0x34, 0x00,
    0x1, 0x35, 0xA1,    0x1, 0x36, 0x00,    0x1, 0x37, 0xA0,    0x1, 0x38, 0x00,
    0x1, 0x39, 0x04,    0x1, 0x3A, 0x28,    0x1, 0x3B, 0x30,    0x1, 0x3C, 0x0C,
    0x1, 0x3D, 0x04,    0x1, 0x3E, 0x0F,    0x1, 0x3F, 0x79,    0x1, 0x40, 0x28,
    0x1, 0x41, 0x1E,    0x1, 0x42, 0x2F,    0x1, 0x43, 0x87,    0x1, 0x44, 0x00,
    0x1, 0x45, 0x0B,    0x1, 0x46, 0x00,    0x1, 0x47, 0x0B,    0x1, 0x48, 0x00,
    0x1, 0x49, 0xA7,    0x1, 0x4A, 0x00,    0x1, 0x4B, 0xA6,    0x1, 0x4C, 0x00,
    0x1, 0x4D, 0x04,    0x1, 0x4E, 0x01,    0x1, 0x4F, 0x00,    0x1, 0x50, 0x00,
    0x1, 0x51, 0x80,    0x1, 0x52, 0x09,    0x1, 0x53, 0x08,    0x1, 0x54, 0x01,
    0x1, 0x55, 0x00,    0x1, 0x56, 0x0F,    0x1, 0x57, 0x79,    0x1, 0x58, 0x09,
    0x1, 0x59, 0x05,    0x1, 0x5A, 0x00,    0x1, 0x5B, 0x60,    0x1, 0x5C, 0x05,
    0x1, 0x5D, 0xD1,    0x1, 0x5E, 0x0C,    0x1, 0x5F, 0x3C,    0x1, 0x60, 0x00,
    0x1, 0x61, 0xD0,    0x1, 0x62, 0x0B,    0x1, 0x63, 0x03,    0x1, 0x64, 0x28,
    0x1, 0x65, 0x10,    0x1, 0x66, 0x2A,    0x1, 0x67, 0x39,    0x1, 0x68, 0x0B,
    0x1, 0x69, 0x02,    0x1, 0x6A, 0x28,    0x1, 0x6B, 0x10,    0x1, 0x6C, 0x2A,
    0x1, 0x6D, 0x61,    0x1, 0x6E, 0x0C,    0x1, 0x6F, 0x00,    0x1, 0x70, 0x0F,
    0x1, 0x71, 0x79,    0x1, 0x72, 0x00,    0x1, 0x73, 0x0B,    0x1, 0x74, 0x00,
    0x1, 0x75, 0x0B,    0x1, 0x76, 0x00,    0x1, 0x77, 0xA1,    0x1, 0x78, 0x00,
    0x1, 0x79, 0xA0,    0x1, 0x7A, 0x00,    0x1, 0x7B, 0x04,    0x1, 0xFF, 0x04,
    0x1, 0x79, 0x1D,    0x1, 0x7B, 0x27,    0x1, 0x96, 0x0E,    0x1, 0x97, 0xFE,
    0x1, 0x98, 0x03,    0x1, 0x99, 0xEF,    0x1, 0x9A, 0x02,    0x1, 0x9B, 0x44,
    0x1, 0x73, 0x07,    0x1, 0x70, 0x01,    0x1, 0xff, 0x01,    0x1, 0x00, 0x01,
    0x1, 0xff, 0x00,    
    0x0, 0x00, 0x00};

/** @defgroup VL53L0X_GpioFunctionality_group Gpio Functionality
 *  @brief Defines the different functionalities for the device GPIO(s)
 *  @{   */
typedef uint8_t VL53L0X_GpioFunctionality;
#define VL53L0X_GPIOFUNCTIONALITY_OFF                     \
	((VL53L0X_GpioFunctionality)  0) /*!< NO Interrupt  */
#define VL53L0X_GPIOFUNCTIONALITY_THRESHOLD_CROSSED_LOW   \
	((VL53L0X_GpioFunctionality)  1) /*!< Level Low (value < thresh_low)  */
#define VL53L0X_GPIOFUNCTIONALITY_THRESHOLD_CROSSED_HIGH   \
	((VL53L0X_GpioFunctionality)  2) /*!< Level High (value > thresh_high) */
#define VL53L0X_GPIOFUNCTIONALITY_THRESHOLD_CROSSED_OUT    \
	((VL53L0X_GpioFunctionality)  3)
/*!< Out Of Window (value < thresh_low OR value > thresh_high)  */
#define VL53L0X_GPIOFUNCTIONALITY_NEW_MEASURE_READY        \
	((VL53L0X_GpioFunctionality)  4) /*!< New Sample Ready  */
/** @} end of VL53L0X_GpioFunctionality_group */


/** @brief Defines the parameters of the Get Version Functions  */
typedef struct {
    uint32_t	 revision; /*!< revision number */
    uint8_t		 major;	   /*!< major number */
    uint8_t		 minor;	   /*!< minor number */
    uint8_t		 build;	   /*!< build number */
} VL53L0X_Version_t;

/** @defgroup VL53L0X_define_Error_group Error and Warning code returned by API
 *	The following DEFINE are used to identify the PAL ERROR
 *	@{   */
typedef int8_t VL53L0X_Error;
#define VL53L0X_ERROR_NONE				((VL53L0X_Error)	0)
#define VL53L0X_ERROR_CALIBRATION_WARNING ((VL53L0X_Error) -1)
/*!< Warning invalid calibration data may be in used
	\a	VL53L0X_InitData()
	\a VL53L0X_GetOffsetCalibrationData
	\a VL53L0X_SetOffsetCalibrationData */
#define VL53L0X_ERROR_MIN_CLIPPED		((VL53L0X_Error) -2)
/*!< Warning parameter passed was clipped to min before to be applied */
#define VL53L0X_ERROR_UNDEFINED			((VL53L0X_Error) -3)
/*!< Unqualified error */
#define VL53L0X_ERROR_INVALID_PARAMS	((VL53L0X_Error) -4)
/*!< Parameter passed is invalid or out of range */
#define VL53L0X_ERROR_NOT_SUPPORTED		((VL53L0X_Error) -5)
/*!< Function is not supported in current mode or configuration */
#define VL53L0X_ERROR_RANGE_ERROR		((VL53L0X_Error) -6)
/*!< Device report a ranging error interrupt status */
#define VL53L0X_ERROR_TIME_OUT			((VL53L0X_Error) -7)
/*!< Aborted due to time out */
#define VL53L0X_ERROR_MODE_NOT_SUPPORTED ((VL53L0X_Error) -8)
/*!< Asked mode is not supported by the device */
#define VL53L0X_ERROR_BUFFER_TOO_SMALL	((VL53L0X_Error) -9)
/*!< ... */
#define VL53L0X_ERROR_GPIO_NOT_EXISTING	((VL53L0X_Error) -10)
/*!< User tried to setup a non-existing GPIO pin */
#define VL53L0X_ERROR_GPIO_FUNCTIONALITY_NOT_SUPPORTED  ((VL53L0X_Error) -11)
/*!< unsupported GPIO functionality */
#define VL53L0X_ERROR_INTERRUPT_NOT_CLEARED	((VL53L0X_Error) -12)
/*!< Error during interrupt clear */
#define VL53L0X_ERROR_CONTROL_INTERFACE	((VL53L0X_Error) -20)
/*!< error reported from IO functions */
#define VL53L0X_ERROR_INVALID_COMMAND	((VL53L0X_Error) -30)
/*!< The command is not allowed in the current device state (power down) */
#define VL53L0X_ERROR_DIVISION_BY_ZERO	((VL53L0X_Error) -40)
/*!< In the function a division by zero occurs */
#define VL53L0X_ERROR_REF_SPAD_INIT			((VL53L0X_Error) -50)
/*!< Error during reference SPAD initialization */
/** @} VL53L0X_define_Error_group */


/** @defgroup VL53L0X_define_DeviceModes_group Defines Device modes
 *	Defines all possible modes for the device
 *	@{    */
typedef uint8_t VL53L0X_DeviceModes;
#define VL53L0X_DEVICEMODE_SINGLE_RANGING	((VL53L0X_DeviceModes)  0)
#define VL53L0X_DEVICEMODE_CONTINUOUS_RANGING	((VL53L0X_DeviceModes)  1)
#define VL53L0X_DEVICEMODE_CONTINUOUS_TIMED_RANGING ((VL53L0X_DeviceModes) 3)
#define VL53L0X_DEVICEMODE_SINGLE_ALS		((VL53L0X_DeviceModes) 10)
#define VL53L0X_DEVICEMODE_GPIO_DRIVE		((VL53L0X_DeviceModes) 20)
#define VL53L0X_DEVICEMODE_GPIO_OSC		((VL53L0X_DeviceModes) 21)
/* ... Modes to be added depending on device */
/** @} VL53L0X_define_DeviceModes_group */


/** @defgroup VL53L0X_define_PowerModes_group List of available Power Modes
 *	List of available Power Modes
 *	@{   */
typedef uint8_t VL53L0X_PowerModes;
#define VL53L0X_POWERMODE_STANDBY_LEVEL1 ((VL53L0X_PowerModes) 0)
/*!< Standby level 1 */
#define VL53L0X_POWERMODE_STANDBY_LEVEL2 ((VL53L0X_PowerModes) 1)
/*!< Standby level 2 */
#define VL53L0X_POWERMODE_IDLE_LEVEL1	((VL53L0X_PowerModes) 2)
/*!< Idle level 1 */
#define VL53L0X_POWERMODE_IDLE_LEVEL2	((VL53L0X_PowerModes) 3)
/*!< Idle level 2 */
/** @} VL53L0X_define_PowerModes_group */

/** @defgroup VL53L0X_define_State_group Defines the current status of the device
 *	Defines the current status of the device
 *	@{ */
typedef uint8_t VL53L0X_State;
#define VL53L0X_STATE_POWERDOWN		 ((VL53L0X_State)  0)
/*!< Device is in HW reset	*/
#define VL53L0X_STATE_WAIT_STATICINIT ((VL53L0X_State)  1)
/*!< Device is initialized and wait for static initialization  */
#define VL53L0X_STATE_STANDBY		 ((VL53L0X_State)  2)
/*!< Device is in Low power Standby mode   */
#define VL53L0X_STATE_IDLE			 ((VL53L0X_State)  3)
/*!< Device has been initialized and ready to do measurements  */
#define VL53L0X_STATE_RUNNING		 ((VL53L0X_State)  4)
/*!< Device is performing measurement */
#define VL53L0X_STATE_UNKNOWN		 ((VL53L0X_State)  98)
/*!< Device is in unknown state and need to be rebooted	 */
#define VL53L0X_STATE_ERROR			 ((VL53L0X_State)  99)
/*!< Device is in error state and need to be rebooted  */
/** @} VL53L0X_define_State_group */


/** @brief Defines the parameters of the Get Device Info Functions  */
typedef struct {
    char Name[VL53L0X_MAX_STRING_LENGTH];
    /*!< Name of the Device e.g. Left_Distance */
    char Type[VL53L0X_MAX_STRING_LENGTH];
    /*!< Type of the Device e.g VL53L0X */
    char ProductId[VL53L0X_MAX_STRING_LENGTH];
    /*!< Product Identifier String	*/
    uint8_t ProductType;
    /*!< Product Type, VL53L0X = 1, VL53L1 = 2 */
    uint8_t ProductRevisionMajor;
    /*!< Product revision major */
    uint8_t ProductRevisionMinor;
    /*!< Product revision minor */
} VL53L0X_DeviceInfo_t;

/** @brief Defines all parameters for the device   */
typedef struct {
    VL53L0X_DeviceModes DeviceMode;
    /*!< Defines type of measurement to be done for the next measurement */
    uint32_t MeasurementTimingBudget_us;
    /*!< Defines the allowed total time for a single measurement */
    uint32_t InterMeasurementPeriod_ms;
    /*!< Defines time between two consecutive measurements (between two
     *	measurement starts). If set to 0 means back-to-back mode */
    uint8_t XTalkCompensationEnable;
    /*!< Tells if Crosstalk compensation shall be enable or not	 */
    uint16_t XTalkCompensationRange_mm;
    /*!< CrossTalk compensation range in _mm	 */
    FixPoint1616_t XTalkCompensationRate_MHz;
    /*!< CrossTalk compensation rate in Mega counts per seconds.
     *	Expressed in 16.16 fixed point format.	*/
    int32_t RangeOffset_um;
    /*!< Range offset adjustment (mm) last programmed.	*/
    uint8_t LimitChecksEnable[VL53L0X_CHECKENABLE_NUMBER_OF_CHECKS];
    /*!< This Array store all the  6 Limit Check enables for this device. */
    uint8_t LimitChecksStatus[VL53L0X_CHECKENABLE_NUMBER_OF_CHECKS];
    /*!< This Array store all the  6 Status of the check linked to last measurement. */
    FixPoint1616_t LimitChecksValue[VL53L0X_CHECKENABLE_NUMBER_OF_CHECKS];
    /*!< This Array store all the Limit Check value for this device */
    uint8_t WrapAroundCheckEnable;
    /*!< Tells if Wrap Around Check shall be enable or not */
} VL53L0X_DeviceParameters_t;

/**
 * @struct VL53L0X_RangeData_t
 * @brief Range measurement data.  */
typedef struct {
    uint16_t Range_mm;	/*!< range distance in _mm. */
    uint16_t RangeDMax_mm; /*!< Tells what is the maximum detection distance of the device
     * in current setup and environment conditions (Filled when	applicable) */
    FixPoint1616_t SignalRateRtn_MHz; /*!< Return signal rate (MHz), as 16.16 fix point, effectively a measure of target reflectance.*/
    FixPoint1616_t AmbientRateRtn_MHz; /*!< Return ambient rate (MHz), as 16.16 fix point, effectively a measure of the ambient light.*/
    uint16_t EffectiveSpadRtnCount; /*!< Return the effective SPAD count for the return signal, as a 8.8 fix point value. */
    uint8_t RangeFractionalPart; /*!< Fractional part of range distance. Final value is a  FixPoint168 value.; only noise :( */
    uint8_t RangeStatus; /*!< Range Status for the current measurement. Value = 0 means value is valid. See \ref RangeStatusPage */
    FixPoint1616_t SigmaEstimate;		/*!< Estimated Sigma - based on ambient & VCSEL rates and signal_total_events */
} VL53L0X_RangingMeasurementData_t;


#define VL53L0X_REF_SPAD_BUFFER_SIZE 6
typedef struct {
    // merged in here all parts of "VL53L0X_DeviceSpecificParameters_t DeviceSpecificParameters;"
    FixPoint1616_t OscFrequency_MHz; 	/* Frequency used */
    uint16_t LastEncodedTimeout; 		/* last encoded Time out used for timing budget*/
    VL53L0X_GpioFunctionality Pin0GpioFunctionality; /* store the functionality of the GPIO: pin0 */
    uint32_t FinalRangeTimeout_us;/*!< Execution time of the final range*/
	uint8_t FinalRangeVcselPulsePeriod;	/*!< Vcsel pulse period (pll clocks) for the final range measurement*/
    uint32_t PreRangeTimeout_us;	/*!< Execution time of the final range*/
    uint8_t PreRangeVcselPulsePeriod;	/*!< Vcsel pulse period (pll clocks) for the pre-range measurement*/
    uint8_t ReadDataFromDeviceDone; /* Indicate if read from device has been done (==1) or not (==0) */
    uint8_t ModuleId; 				/* Module ID */
    uint8_t Revision; 				/* test Revision */
    char ProductId[VL53L0X_MAX_STRING_LENGTH]; /* Product Identifier String  */
    uint8_t ReferenceSpadCount; /* used for ref spad management */
    uint8_t ReferenceSpadType;	/* used for ref spad management */
    uint8_t RefSpadsInitialised; /* reports if ref spads are initialised. */
    uint32_t PartUIDUpper; /*!< Unique Part ID Upper */
    uint32_t PartUIDLower; /*!< Unique Part ID Lower */
    FixPoint1616_t SignalRateMeasFixed400mm; /*!< Peek Signal rate at 400 mm*/
    
    // Merged in here the VL53L0X_SpadData_t SpadData; 		/*!< Spad Data; Enables and Good-Map */
    uint8_t RefSpadEnables[VL53L0X_REF_SPAD_BUFFER_SIZE]; /*!< Reference Spad Enables */
    uint8_t RefGoodSpadMap[VL53L0X_REF_SPAD_BUFFER_SIZE]; /*!< Reference Spad Good Spad Map */
    
	// Merged in here all parts of DeviceParams_t
    int32_t	 Part2PartOffsetNVM_um; /*!< backed up NVM value */
    int32_t	 Part2PartOffsetAdjustNVM_um; /*!< backed up NVM value representing additional offset adjustment */
    uint8_t SequenceConfig; 			/*!< Internal value for the sequence config */
    uint8_t RangeFractionalEnable; 		/*!< Enable/Disable fractional part of ranging data */
    VL53L0X_State PalState;        		/*!< Current state of the PAL for this device */
    VL53L0X_PowerModes PowerMode;		/*!< Current Power Mode; Stdby1/2, Idle1/2	 */
    uint16_t SigmaEstRefArray;			/*!< Reference array sigma value in 1/100th of [mm] e.g. 100 = 1mm; loaded from tuning settings */
    uint16_t SigmaEstEffPulseWidth;		/*!< Effective Pulse width for sigma estimate in 1/100th of ns e.g. 900 = 9.0ns */
    uint16_t SigmaEstEffAmbWidth;		/*!< Effective Ambient width for sigma estimate in 1/100th of ns e.g. 500 = 5.0ns */
    uint8_t StopVariable;				/*!< StopVariable used during the stop sequence */
    uint16_t targetRefRate;				/*!< Target Ambient Rate for Ref spad management */
    FixPoint1616_t LastSignalRef_MHz;	/*!< Latest Signal ref in MHz */
    uint8_t *pTuningSettingsPointer;	/*!< Pointer for alternative Tuning Settings table */
    uint8_t UseInternalTuningSettings;	/*!< Indicate if we use	 Tuning Settings table, default = True */
    uint16_t LinearityCorrectiveGain;	/*!< Linearity Corrective Gain value in x1000 */
    uint16_t DmaxCalRange_mm; 	/*!< Dmax Calibration Range _mm */
    FixPoint1616_t DmaxCalSignalRateRtn_MHz;/*!< Dmax Calibration Signal Rate Return _MHz */
    
} VL53L0X_DevData_t;

 /*  All these macros were replaced with following regulator expressions: 
a)  Search for: \QVL53L0X_SETARRAYPARAMETERFIELD(\E(\s*)([A-Z\d_]+)[[:punct:]](\s*)([A-Z\d_]+)[[:punct:]](\s*)([A-Z\d_\*]+)\Q);\E
    Replace by: Data.CurrentParameters.\2[\4] = \6;
    to replace this:  VL53L0X_SETARRAYPARAMETERFIELD(field, index, value);  
    by this:  Data.CurrentParameters.field[index] = value; 

b)  Search for: \QVL53L0X_GETARRAYPARAMETERFIELD(\E(\s*)([A-Z\d_]+)[[:punct:]](\s*)([A-Z\d_]+)[[:punct:]](\s*)([A-Z\d_\*]+)\Q);\E
    Replace by: \6 = Data.CurrentParameters.\2[\4];
    to replace this: VL53L0X_GETARRAYPARAMETERFIELD(  LimitChecksValue, limit_check_id, temp_fix1616); 
    by this:  temp_fix1616 = Data.CurrentParameters.LimitChecksValue[limit_check_id]; 

c)  Search for: \QVL53L0X_SETPARAMETERFIELD(\E(\s*)([A-Z\d_]+)[[:punct:]](\s*)([A-Z\d_\*]+)\Q);\E
    Replace by: Data.CurrentParameters.\2 = \4;
    to replace this:  VL53L0X_SETPARAMETERFIELD(field, value);  
    by this:  Data.CurrentParameters.field = value 

d)  Search for: \QVL53L0X_GETPARAMETERFIELD(\E(\s*)([A-Z\d_]+)[[:punct:]](\s*)([A-Z\d_\*]+)\Q);\E
    Replace by: \4 = Data.CurrentParameters.\2 ;
    to replace this: VL53L0X_GETPARAMET ERFIELD(field, variable);  
    by this:  variable = Data.CurrentParameters.field; 

d)  Search for: \QPALDevDataSet(\E(\s*)([A-Z\d_]+)[[:punct:]](\s*)([A-Z\d_\*]+)\Q);\E
    Replace by: Data.\2 = \4;
    to replace this: PALDevDataSet(field, value);
    by this:  Data.field = value;

d)  Search for: \QPALDevDataGet(\E(\s*)([A-Z\d]+)\Q)\E
    Replace by: Data.\2
    to replace this: PALDevDataGet(field)
    by this:  Data.field

#define PALDevDataSet(field, value) (Data.field)=(value)
#define PALDevDataGet(field) (Data.field)
#define VL53L0X_SETPARAMETERFIELD(field, value) Data.CurrentParameters.field = value
#define VL53L0X_GETPARAMETERFIELD(field, variable) variable = Data.CurrentParameters.field
#define VL53L0X_SETARRAYPARAMETERFIELD(field, index, value) Data.CurrentParameters.field[index] = value
#define VL53L0X_GETARRAYPARAMETERFIELD(field, index, variable)	variable = Data.CurrentParameters.field[index]
#define VL53L0X_SETDEVICESPECIFICPARAMETER(field, value) Data.field = value
#define VL53L0X_GETDEVICESPECIFICPARAMETER(field) 		 Data.field
 */
 
 
/** @defgroup VL53L0X_define_InterruptPolarity_group Defines the Polarity  of the Interrupt
 *	Defines the Polarity of the Interrupt
 *	@{
 */
typedef uint8_t VL53L0X_InterruptPolarity;
#define VL53L0X_INTERRUPTPOLARITY_LOW	   ((VL53L0X_InterruptPolarity)	0)
/*!< Set active low polarity best setup for falling edge. */
#define VL53L0X_INTERRUPTPOLARITY_HIGH	   ((VL53L0X_InterruptPolarity)	1)
/*!< Set active high polarity best setup for rising edge. */

/** @} VL53L0X_define_InterruptPolarity_group */


/** @defgroup VL53L0X_define_VcselPeriod_group Vcsel Period Defines
 *	Defines the range measurement for which to access the vcsel period.
 *	@{  */
typedef uint8_t VL53L0X_VcselPeriod;
#define VL53L0X_VCSEL_PERIOD_PRE_RANGE	((VL53L0X_VcselPeriod) 0)
/*!<Identifies the pre-range vcsel period. */
#define VL53L0X_VCSEL_PERIOD_FINAL_RANGE ((VL53L0X_VcselPeriod) 1)
/*!<Identifies the final range vcsel period. */

/** @} VL53L0X_define_VcselPeriod_group */
/** @defgroup VL53L0X_define_SchedulerSequence_group Defines the steps
 * carried out by the scheduler during a range measurement.
 *	@{
 *	Defines the states of all the steps in the scheduler i.e. enabled/disabled. */
typedef struct {
    uint8_t		 TccOn;	   		/*!<Reports if Target Centre Check On  */
    uint8_t		 MsrcOn;	   	/*!<Reports if MSRC On  */
    uint8_t		 DssOn;		   	/*!<Reports if DSS On  */
    uint8_t		 PreRangeOn;   	/*!<Reports if Pre-Range On	*/
    uint8_t		 FinalRangeOn; 	/*!<Reports if Final-Range On  */
} VL53L0X_SchedulerSequenceSteps_t;
/** @} VL53L0X_define_SchedulerSequence_group */

/** @defgroup VL53L0X_define_SequenceStepId_group 
 *	Defines the the sequence steps performed during ranging..
 *	@{  */
typedef uint8_t VL53L0X_SequenceStepId;
#define	 VL53L0X_SEQUENCESTEP_TCC		  ((VL53L0X_VcselPeriod) 0) /*!<Target CentreCheck identifier. */
#define	 VL53L0X_SEQUENCESTEP_DSS		  ((VL53L0X_VcselPeriod) 1) /*!<Dynamic Spad Selection function Identifier. */
#define	 VL53L0X_SEQUENCESTEP_MSRC		  ((VL53L0X_VcselPeriod) 2) /*!<Minimum Signal Rate Check function Identifier. */
#define	 VL53L0X_SEQUENCESTEP_PRE_RANGE	  ((VL53L0X_VcselPeriod) 3) /*!<Pre-Range check Identifier. */
#define	 VL53L0X_SEQUENCESTEP_FINAL_RANGE ((VL53L0X_VcselPeriod) 4) /*!<Final Range Check Identifier. */
#define	 VL53L0X_SEQUENCESTEP_NUMBER_OF_CHECKS			 5 /*!<Number of Sequence Step Managed by the API. */
/** @} VL53L0X_define_SequenceStepId_group */


typedef enum {
	Range_Config_DEFAULT,
	Range_Config_LONG_RANGE,
	Range_Config_HIGH_ACCURACY,
	Range_Config_HIGH_SPEED
} VL53L0X_RangingConfig;
/** @} VL53L0X_define_SequenceStepId_group */

/** @} VL53L0X_globaldefine_group */


#ifdef __cplusplus
}
#endif


#endif /* _VL53L0X_DEF_H_ */
