/*
 * This VL53L1X satellite board sample application performs range measurements
 * with interrupts enabled to generate a hardware interrupt each time a new
 * measurement is ready to be read.
 *
 *  Measured ranges are output on the Serial Port, running at 115200 baud.
 *
 *  On STM32-Nucleo boards :
 *      The User Blue button stops the current measurement and entire program,
 *      releasing all resources.
 *
 *      The Black Reset button is used to restart the program.
 *
 *  *** NOTE : By default hardlinks U10, U11, U15 & U18, on the underside of
 *            the X-NUCELO-53L1A1 expansion board are not made/OFF.
 *            These links must be made to allow interrupts from the Satellite boards
 *            to be received.
 *            U11 and U18 must be made/ON to allow interrupts to be received from the
 *            INT_L & INT_R positions; or
 *            U10 and U15 must be made/ON to allow interrupts to be received from the
 *            Alternate INT_L & INT_R positions.
 *            The X_NUCLEO_53L1A1 firmware library defaults to use the INT_L/INT_R
 *            positions.
 *            INT_L is available on expansion board Arduino Connector CN5, pin 1 as D9.
 *            Alternate INT_L is on CN5 Connector pin 2 as D8.
 *            INT_R is available on expansion board Arduino Connector CN9, pin 3 as D4.
 *            Alternate INT_R is on CN9 Connector pin 5 as D2.
 *            The pinouts are shown here : https://developer.mbed.org/components/X-NUCLEO-53L1A1/
 */

#include <stdio.h>

#include "mbed.h"
#include "VL53L1X_I2C.h"
#include "VL53L1X_Class.h"
#include "LED_WS2812.h"

#define VL53L1_I2C_SDA   D14
#define VL53L1_I2C_SCL   D15

static VL53L1X *sensor = NULL;
Serial pc(SERIAL_TX, SERIAL_RX);
PwmOut servo(D6);

/* flags that handle interrupt request for sensor and user blue button*/
volatile bool int_sensor = false;
volatile bool int_stop = false;


/* ISR callback function of the sensor */
void sensor_irq(void)
{
    int_sensor = true;
    sensor->disable_interrupt_measure_detection_irq();
}

/* ISR callback function of the user blue button to switch measuring sensor. */
void measuring_stop_irq(void)
{
    int_stop = true;
}

/* Start the sensor ranging */
int start_ranging()
{
    int status = 0;
    /* start the measure on the sensor */
    if (NULL != sensor) {
        status = sensor->stop_measurement();
        if (status != 0) {
                return status;
        }

        status = sensor->start_measurement(&sensor_irq);
        if (status != 0) {
            return status;
        }
    }
    return status;
}

int range_measure(VL53L1X_DevI2C *device_i2c)
{
    int status = 0;
    uint16_t distance = 0;
    int distance_close = 0;
    
    LED_WS2812 LED(A0,12);
    LED.SetIntensity(5);
    LED.SetColor(BLACK);
    
    /* Create a xshutdown pin */
    DigitalOut xshutdown(D7);

    /* create instance of sensor class */
    sensor = new VL53L1X(device_i2c, &xshutdown, A2);

    sensor->vl53l1_off();
    /* initialise sensor */
    sensor->init_sensor(0x52);

    if (status) {
        delete sensor;
        sensor= NULL;
        printf("Sensor centre not present\n\r");
    }

    status = sensor->vl53l1x_set_inter_measurement_in_ms(1000);    
    status = sensor->vl53l1x_set_distance_threshold(50,
                  50, 0, 0);
    status = start_ranging();
    if (status != 0) {
        printf("Failed to start ranging!\r\n");
        return status;
    }
    if (NULL != sensor) {
        printf("Entering loop mode\r\n");
        /* Main ranging interrupt loop */
        while (true) {
            if (int_sensor) {
                
                servo.period_ms(20);
                
                servo.pulsewidth_us(1150);
                wait_ms(100);    
                servo.pulsewidth_us(1100);
                wait_ms(100);    
                servo.pulsewidth_us(1050);
                wait_ms(100);    
                servo.pulsewidth_us(1000);
                wait_ms(100);    
                servo.pulsewidth_us(950);
                wait_ms(100);    
                servo.pulsewidth_us(900);
                wait_ms(100);    

                servo.pulsewidth_us(850);
                wait_ms(100);    
                servo.pulsewidth_us(800);
                wait_ms(100);
                servo.pulsewidth_us(750);
                wait_ms(100);
                servo.pulsewidth_us(700);
//                wait_ms(100);
//                servo.pulsewidth_us(650);
//                wait_ms(100);
//                servo.pulsewidth_us(600);
//                wait_ms(100);   
//                servo.pulsewidth_us(550);
//                wait_ms(100);
//                servo.pulsewidth_us(500);
 
                wait_ms(500);
                servo.pulsewidth_us(1150);
                
                int_sensor = false;
                status = sensor->handle_irq(&distance);
                printf("distance: %d\r\n", distance);         
            }

        }
    }

    return status;

}


/*=================================== Main ==================================
=============================================================================*/
int main()
{
#if TARGET_STM  // we are cross compiling for an STM32-Nucleo    
    InterruptIn stop_button(USER_BUTTON);
    stop_button.rise(&measuring_stop_irq);
#endif
#if TARGET_Freescale // we are cross-compiling for NXP FRDM boards.
    InterruptIn stop_button(SW2);
    stop_button.rise(&measuring_stop_irq);
#endif
    VL53L1X_DevI2C *device_i2c = new VL53L1X_DevI2C(VL53L1_I2C_SDA, VL53L1_I2C_SCL);
    range_measure(device_i2c);  // start continuous measures
}
