/**
 * Steuert die 4 Phaserunner
 */

#ifndef PA_REGLER_H
#define PA_REGLER_H

#include "mbed.h"
#include "Phaserunner.h"
#include "Encoder.h"
#include "PID.h"


#define ERGO 0
#define MOFA 1
#define PEDELEC 2

class Regler{
private:
    //Alle Phaserunner
    Phaserunner& vorderrad;
    Phaserunner& hinterrad;
    Phaserunner& linkspedal;
    Phaserunner& rechtspedal;
    
    // Analog Outputs für Generatoren am Pedalen
    AnalogOut gegenmomentLinks;
    AnalogOut gegenmomentRechts;
    
    // PID Regler
    PID&          pid;
    
    //Encoder
    Encoder&     encoder;

    //Drehmomentverhältnis
    uint8_t torqueRatio;

    //Verhältnis des Gegenmomentes
    uint8_t recupRatio;
    
    //Zeitliche Berechnung der Werte
    void ReglerCalculate();
    
    // Ticker
    Ticker ticker;
    
    // Ellipse Parameters
    float a_,b_,beta_;
    
    // Regler Parameters Pedale
    float Kp_,Ki_,Ts_,setPoint_;
    
    // Betriebsmodus (0->Ergo, 1->Mofa, 2->Pedelec) 
    uint8_t Modus_;
    
    // Ergometer Stufe "(von 1 bis 10)"
    float ErgoStufe_;
    
    // Pedelec Values
    float PedStufe_, PedFactor_;
    
    // AnalogOut Pins für gegenmomentsteuerung
    static const PinName GEG_LINKS_PIN = PA_4; 
    static const PinName GEG_RECHTS_PIN = PA_5; 
    
    const static float PI = 3.14159265;


public:
    /**
     * Erstellt einen Regler für alle Phaserunner
     * @param vorderrad
     * @param hinterrad
     * @param linkspedal
     * @param rechtspedal
     * @param encoder
     */
    Regler(Phaserunner& vorderrad, Phaserunner& hinterrad, Phaserunner& linkspedal, Phaserunner& rechtspedal, Encoder& encoder);

    /**
     * Regelt die Phaserunner auf eine bestimmte Geschwindigkeit.
     * @param speed Geschwindigkeit in km/h
     */
    void setSpeed(uint8_t speed);

    /**
     * Setzt das Drehmomentverhältnis zwischen Vorderrad und Hinterrad.
     * @param torqueRatio: Drehmoment des Hinterrades in %. Das Drehmoment auf dem Vorderrad ist dann 100% - torqueRatio.
     */
    void setTorqueProportion(uint8_t torqueRatio);

    /**
     * Setzt das Verhältnis des Gegenmomentes zwischen Vorderrad und Hinterrad.
     * @param torqueRatio: Gegenmoment des Hinterrades in %. Das Gegenmoment auf dem Vorderrad ist dann 100% - torqueRatio.
     */
    void setRecuperationProportion(uint8_t recupRatio);

    /**
     * Setzt das Drehmoment auf beide Phaserunner.
     * @param torque Drehmoment in Prozent
     */
    void setTorqueMotors(uint8_t torque);

    /**
     * Setzt das Drehmoment auf beide Phaserunner.
     * @param torque Drehmoment in Prozent
     */
    void setTorquePedals(uint8_t torque);

    /**
     * Setzt das Nullpunkt der Pedalen.
     * 
     */
    void setZero();
    
    /**
     * Setzt die Rekuperation der Motoren
     * @param recuperation
     */
    void setRecuperationMotors(uint8_t recuperation);

    /**
     * Setzt die Rekuperation der Pedale
     * @param recuperation
     */
    void setRecuperationPedals(uint8_t recuperation);
    
    /**
     * Setzt die Ellipse Parameters
     * @param a
     * @param b
     * @param beta
     */
    void setEllipseParameters(float a, float b, float beta);
    
    /**
     * Setzt die PI Regler Parameters
     * @param Kp
     * @param Ki
     * @param Ts in s
     */
    void setReglerParameters(float Kp, float Ki, float Ts);
    
    /**
     * Setzt die PI Regler Setpoint
     * @param SetPoint
     */
    void setReglerSetPoint(float Setpoint);
    
    /**
     * Setzt die Ergometer schwirichkeit Stufe
     * @param ErgoStufe
     */
    void setErgoStufe(float ErgoStufe);
    
    /**
     * Setzt die Pedelec schwirichkeit Stufe
     * @param ErgoStufe
     */
    void setPedStufe(float PedStufe);
    
    /**
     * Setzt die Pedelec Factor
     * @param PedFactor
     */
    void setPedFactor(float PedFactor);
    
    /**
     * Setzt die Modus
     * @param Modus
     */
    void setModus(float Modus);
};

#endif //PA_REGLER_H

