/**
 * Handelt die Kommunikation (Lesen und Schreiben) mit den Phaserunnern
 */

#ifndef PHASERUNNER_H
#define PHASERUNNER_H

#include "mbed.h"
#include "rtos.h"

class Phaserunner {
private:
    //Private constants
    static const uint8_t BUFLEN = 255;

    //Adressen
    static const unsigned short REMOTE_THROTTLE_VOLTAGE =       495;
    static const unsigned short REMOTE_ANALOG_BREAK_VOLTAGE =   497;
    static const unsigned short MOTOR_CURRENT =                 262;
    static const unsigned short LAST_FAULT =                    269;
    static const unsigned short EBRAKESOURCE =                  249;

    //Modbus-Protokoll
    static const uint8_t SCHREIBANTWORT =   0x10;
    static const uint8_t LESEANTWORT =      0x03;

    //Pedale oder Motor
    const uint8_t PHASERUNNERTYPE;

    //Sonstiges
    static const uint16_t WRITE_PERIOD =    3;      //Minimaler Zeitintervall zwischen zwei Messages an den Phaserunner [ms]
    static const float TICKER_PERIOD =      0.02f;  //Zeitintervall für das Schreiben auf den Phaserunner [s]

    AnalogOut* analogOut;   //Wenn der Phaserunner Pedale kontrolliert, hat er einen analogOut

    //Thread thread; //Wird für die Funktion writeToPhaserunner() gebraucht

    Ticker ticker;      //Wird gebraucht um nicht zu häufig zu schreiben
    uint8_t timer;       //Zeit, die vergehen muss, bis wieder geschrieben werden kann [ms]

    //Verbindung zum Phaserunner
    RawSerial& connection;

    //Buffer in das gelesen wird.
    uint8_t writeBuffer[BUFLEN];

    //Buffer aus dem geschrieben wird.
    uint8_t read_buffer[BUFLEN];

    //State valiables
    float frequency;
    float voltage;
    float current;
    uint16_t error;
    uint16_t newTorque;
    uint16_t newRecuperation;

    /**
     * Wartet auf ein Byte. Wenn ein Datenstring gelesen wird, werden die Daten mithilfe einer Zustandsmaschine abgespeichert.
     */
    void Rx_interrupt();

    /**
     * @brief Bereitet buf darauf vor, versendet zu werden.
     * @param buf:      Byte-Array
     * @param addr:     Adresse des Registers, das beschrieben wird
     * @param value:    Wert der in das Register geschrieben wird
     */
    static int WriteRegister(uint8_t* buf, unsigned short addr, unsigned short value);

    /**
     * @brief Sendet einen Schreibbefehl auf einen Phaserunner
     * @param writeBuf:         Byte-Array in das geschrieben wird.
     * @param registerAddress:  Adresse des Registers aus dem gelesen wird.
     */
    static int readRegister(uint8_t *buf, uint16_t registerAddress);

    /**
     * @brief Sendet ein Byte-Array zum Phaserunner
     * @param adress:   Adresse des Zielregisters
     * @param value:    Wert der in das Register geschriebern werden soll.
     * @return          Anzahl gesendeter Bytes
     */
    int sendBuffer(unsigned short adress, unsigned short value);

    /**
     * @brief   Sendet den writeBuffer
     * @param:  Länge des zu sendenden Arrays
     * @return: Anzahl gesendeter Bytes
     */
    int sendBuffer(int length);

    /**
     * @brief:          Liest ein Register aus.
     * @param adress:   Adresse des Registers aus dem gelesen werden soll.
     * @return:         Anzahl gelesener Bytes
     */
    int readBuffer(uint16_t adress);

    /**
     * @brief:  Schreibt ein Drehmoment auf den Phaserunner.
     * @param:  torque Drehmoment in percent
     */
    void writeTorque(uint8_t torque);

    /**
     * @brief Schreibt einen Analogwert mit dem das Drehmoment kontrolliert wird.
     * @param torque
     */
    void analogTorque(uint8_t torque);

    /**
     * @brief:  Schreibt den Rekuperationswert auf den Phaserunner.
     * @param:  recuperation Rekupertation in Prozent
     */
    void writeRecuperation(uint8_t recuperation);

    /**
     * @brief:  Prüfziffer berechnen
     * @param:  msgByte Byte-Array
     * @param:  length length of Byte-Array
     */
    static uint16_t getCRC(uint8_t* msgByte, uint8_t length);

    /**
     * @brief: Sendet Befehle dem Phaserunner sobald es möglich ist.
     */
    //void writeToPhaserunner();

    /**
     * @brief:  Sendet einen Schreibbefehl für die Drehgewschwindigkeit, die Stromstärke und die Spannung
     */
    void readRPM();

    /**
     * @brief:  Liest das Fehlerregister aus
     * @return  Byte
     */
    uint16_t readFaults();

    /**
     * Reduziert den timer (Wird mit dem Ticker aufgerufen).
     */
    void reduce_timer();

public:
    static const uint8_t MAX_TORQUE_GAIN    = 2;    //Maximaler Sprung für Drehmoment
    static const uint8_t MIN_RECUPERATION   = 10;   //Schwellwert für die Aktivierung des Daumengriffbetätigers
    static const uint8_t MIN_HANDGRIFF      = 5;    //Schwellwert für die Aktivierung des Handgriffbetätigers

    static const uint8_t PEDALS             = 0;    //Phaserunner Type Pedal
    static const uint8_t MOTORS             = 1;    //Phaserunner Type Motor

    //int sendBuffer(unsigned short adress, unsigned short value);
    //int readBuffer(uint16_t adress);
    /**
     * @brief Initialisiert ein Phaserunner-Objekt
     * @param connection: Serielle Schnittstelle zum Phaserunner
     */
    Phaserunner(RawSerial& connection);

    /**
     * Konstruktor für Pedale
     * @param connection
     * @param analogOut
     */
    Phaserunner(RawSerial& connection, AnalogOut* analogOut);

    /**
     * @brief:  Schreibt ein Drehmoment ins Phaserunnerobjekt, das dann geschrieben wird.
     * @param:  torque Drehmoment in Prozent
     */
    void setTorque(uint8_t torque);

    /**
     * @brief:  Schreibt einen Rekuperationswert ins Phaserunnerobjekt, dass dann geschrieben wird.
     * @param:  recuperation Rekuperationswert in Prozent
     */
    void setRecuperation(uint8_t recuperation);

    /**
     * @return frequenz
     */
    float getFrequency();

    /**
     * @return strom
     */
    float getCurrent();

    /**
     * @return spannung
     */
    float getVoltage();

    /**
     * @return Ebrike Source
     */
    int getRegister(int address);

    //Delete
    uint8_t badBuf[256];
    uint8_t bufPointer;
    uint8_t read;
    DigitalOut led;

    uint16_t getRecup();
};

#endif
