#include "mbed.h"
#include "rtos.h"
#include "LPC17xx.h"
#include "crc16.h"
#include "VAM.h"

Serial pc(USBTX, USBRX);

DigitalOut led1(LED1); // Error Led
DigitalOut led2(LED2); // Still Alive Answer
DigitalOut led3(LED3); // Short Message
DigitalOut led4(LED4); // Long Message
DigitalOut dataOut(p11);
InterruptIn dataIn(p12);
InterruptIn btn1(p15);
InterruptIn btn2(p14);


Thread sender;
Thread receiver;


bool detectingPreambule = false;
bool dataInState = false;

uint32_t elapsed;
uint32_t dataToggleT[7];
int dataInToggleCount = 0;

uint32_t elapsedMin, elapsedMax;

bool reading = false;
bool bitReceived = false;
bool clkUp = false;
bool shortMessageRequested = false;
bool longMessageRequested = false;
bool stillAliveRequested = false;
int stillAliveCounter = 10;

Queue<void, 128> bitsReceived;

#define SIGNAL_CLK_TOGGLE 0x01
#define SIGNAL_BIT_READY  0x04
#define SIGNAL_PREAMBULE_TIME_READY 0x08

void timer0_init(void)
{
    LPC_SC->PCLKSEL0 |= 1<<2;        //pclk = cclk timer0
    LPC_SC->PCONP |= 1<<1;           //timer0 power on
    LPC_TIM0->EMR = 0;               //EMC0 = 00 (DoNothing)
    LPC_TIM0->TCR = 1;               //enable Timer1
}

extern "C" void TIMER2_IRQHandler (void)
{
    if((LPC_TIM2->IR & 0x01) == 0x01) { // if MR0 interrupt, proceed
        LPC_TIM2->IR |= 1 << 0;         // Clear MR0 interrupt flag
        clkUp = !clkUp;
        sender.signal_set(SIGNAL_CLK_TOGGLE);
    }
}

void timer2_init(void)
{
    LPC_SC->PCLKSEL1 |=1<<12;       //pclk = cclk timer2
    LPC_SC->PCONP |=1<<22;          //timer2 power on
    LPC_TIM2->MR0 = 96000;         //10 msec
    LPC_TIM2->MCR = 3;              //interrupt and reset control
    //3 = Interrupt & reset timer2 on match
    //1 = Interrupt only, no reset of timer0
    LPC_TIM2->EMR =3<<4;             //EMC0 = 11 (Toogle)
    NVIC_EnableIRQ(TIMER2_IRQn);    //enable timer2 interrupt
    LPC_TIM2->TCR = 1;              //enable Timer2
}

void DataInRise(){
    elapsed = LPC_TIM0->TC;
    LPC_TIM0->TC = 0;
    
    dataInState = true;
    
    dataToggleT[dataInToggleCount] = elapsed;
    dataInToggleCount = (dataInToggleCount + 1) % 7;
    if (elapsed < elapsedMin){
        LPC_TIM0->TC = elapsed + LPC_TIM0->TC;
    } else if (elapsed <= elapsedMax){
        bitsReceived.put((void*)0);
    }
}

void DataInFall(){
    elapsed = LPC_TIM0->TC;
    LPC_TIM0->TC = 0;
    
    dataInState = false;
    
    dataToggleT[dataInToggleCount] = elapsed;
    dataInToggleCount = (dataInToggleCount + 1) % 7;
    if (elapsed < elapsedMin){
        LPC_TIM0->TC = elapsed + LPC_TIM0->TC;
    } else if (elapsed <= elapsedMax){
        bitsReceived.put((void*)1);
    }
}

void ShortMessageInt(){
    shortMessageRequested = true;
}

void LongMessageInt(){
    longMessageRequested = true;
}

// *************************
//         Sender
// *************************

void SetManchester(bool data)
{
    if (data) {
        dataOut = 1;
    } else {
        dataOut = 0;
    }
}

void SendManchester(bool data)
{
    if (data) {
        dataOut = 0;
    } else {
        dataOut = 1;
    }
}

void SendByte(char data)
{
    for (int i =0; i < 8; i++) {
        bool bit = (data>>(7-i))&0x1;
        Thread::signal_wait(SIGNAL_CLK_TOGGLE);
        SetManchester(bit);
        Thread::signal_wait(SIGNAL_CLK_TOGGLE);
        SendManchester(bit);
    }
}

void SendFrame(char* msg, int length, char type)
{
    crc_t crc = initCrc();
    
    //Attente d'un front descendent avant de commencer la transmission
    do{
        Thread::signal_wait(SIGNAL_CLK_TOGGLE);
    } while(clkUp);
    
    SendByte(PREAMBULE);
    SendByte(START);
    
    SendByte(type); //type & flag
    addByteToCrc(crc, type);
    
    SendByte(length);
    addByteToCrc(crc, length);
    
    for (int i = 0; i < length; i++){
        SendByte(msg[i]);
        addByteToCrc(crc, msg[i]);
    }
    
    uint16_t diggest = getCrc(crc);
    SendByte(diggest >> 8);
    SendByte(diggest & 0xff); // CRC
    
    SendByte(END);
}

void SendMessage(char* msg, int length){
    char buffer[MAX_LENGTH];
    if (length >= MAX_LENGTH){
        int i = 0; 
        buffer[0] = 0;
        while (i < length){
            buffer[0] = (length - i) > (MAX_LENGTH - 1) ? buffer[0] + 1 : 0xFF;
            int subLength = (length - i) > (MAX_LENGTH - 1) ? MAX_LENGTH - 1: length - i;
            memcpy(&buffer[1],&msg[i],subLength);
            SendFrame(buffer, subLength + 1, TYPE_DATA|FLAG_QUESTION);
            i += MAX_LENGTH - 1;
        }            
    }
    else {
        buffer[0] = FRAGMENTATION_ALONE;
        memcpy(&buffer[1],msg,length);
        SendFrame(buffer, length + 1, TYPE_DATA|FLAG_QUESTION);
    }
}

const char * SHORT_MSG = "Patate";
const char * LONG_MSG = "Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.";

void SendShortMessage(){
    char msg[6];
    memcpy(msg, SHORT_MSG, 6);
    SendMessage(msg,6);
}

void SendLongMessage(){
    char msg[123];
    memcpy(msg, LONG_MSG, 123);
    SendMessage(msg, 123);
}

void SenderMain(){
    while(true){
        --stillAliveCounter;
        if(stillAliveRequested){
            SendFrame(0, 0, TYPE_STILL_ALIVE | FLAG_ANSWER);
            stillAliveRequested = false;
        } else if (stillAliveCounter < 5) {
            SendFrame(0, 0, TYPE_STILL_ALIVE | FLAG_QUESTION);
        } else if(shortMessageRequested){
            SendShortMessage();
            shortMessageRequested = false;
        } else if(longMessageRequested){
            SendLongMessage();
            longMessageRequested = false;
        }
    }
}

// *************************
//         Receiver
// *************************

int ReadBit(){
    return (int)bitsReceived.get().value.p;
}

void WaitForPreambule(){
    
    uint8_t buffer = 0x00;
    
    elapsedMin = 0;
    elapsedMax = 960000000;
    
    while(true){
        do{
            buffer = (buffer << 1) | ReadBit();
        }while (buffer != PREAMBULE);
        
        uint32_t delta = dataToggleT[0];
        uint32_t tempMin = (uint32_t) ((float)delta * 0.9f);
        uint32_t tempMax = (uint32_t) ((float)delta * 1.1f);
        bool success = !dataInState;
        for (int i = 1; i < 7 && success; ++i){
            if (dataToggleT[i] > tempMax || dataToggleT[i] < tempMin){
                success = false;
            }
        }
        
        if (success){
            elapsedMin = tempMin;
            elapsedMax = tempMax;
            break;
        }
    }
}

char ReadByte()
{
    uint8_t buffer = 0x00;
    for (int i =0; i < 8; i++) {
        //Thread::signal_wait(SIGNAL_BIT_READY);
        buffer = (buffer << 1) | ReadBit();
    }
    return buffer;
}

bool readFrame (char* msg)
{
    char data;
    crc_t crc = initCrc();
    
    WaitForPreambule();
    data = ReadByte();
    if (data != START){
        //pc.printf("Bad Start Byte %02X\r\n", data);
        return false;
    }
    
    msg[0] = ReadByte(); // type & flag
    addByteToCrc(crc, msg[0]);
    
    msg[1] = ReadByte();
    addByteToCrc(crc, msg[1]);
    
    int length = msg[1];
    for (int i = 0; i < length; i++){
        msg[2+i] = ReadByte();
        addByteToCrc(crc, msg[2+i]);
    }
    
    uint16_t myDiggest = getCrc(crc);
    uint16_t hisDiggest = ReadByte();
    hisDiggest = (hisDiggest << 8) | ReadByte();
    
    if(myDiggest != hisDiggest){
        //pc.printf("Bad Diggest %04X, %04X\r\n", myDiggest, hisDiggest);
        return false;
    }
    
    data = ReadByte();
    if (data != END){
        //pc.printf("Bad End Byte %02X\r\n", data);
        return false;
    }
    return true;
}

void HandleStillAliveFrame(char * frame, bool question){
    if(question){
        stillAliveRequested = true;
    } else {
        stillAliveCounter = 10;
        //pc.printf("The other device is still alive!!!\r\n");
        led2 = !led2;
    }
}

void HandleDataFrame(char * frame){
    char frag = frame[FRAGMENTATION_IDX];
    
    if (frag == 0x00){
        led3 = !led3;
    } else if (frag != 0xFF){
        led4 = 1;
    } else {
        led4 = 0;
    }
    
    //char msg[MAX_LENGTH];
    //msg[frame[LENGTH_IDX] - 1] = 0;
    //memcpy(msg, &frame[FRAGMENTATION_IDX + 1], frame[LENGTH_IDX] - 1);
    //pc.printf("Message %d: %s \r\n",frag,msg);
}

void HandleFrame(char * frame){
    bool question = (frame[TYPE_FLAG_IDX] & FLAG_MASK) == FLAG_QUESTION;
    switch (frame[TYPE_FLAG_IDX] & TYPE_MASK){
    case TYPE_STILL_ALIVE:
        HandleStillAliveFrame(frame, question);
        break;
    case TYPE_DATA:
        HandleDataFrame(frame);
        break;
    default:
        //pc.printf("Unknown Frame Type : %02X\r\n", frame[TYPE_FLAG_IDX]);
        break;
    }
}

void ReceiverMain(){
    char frame[80];
    while(true){
        if(!readFrame(frame)){
            //pc.printf("Error while receiving frame\r\n");
            led1 = 1;
            continue;
        }
        led1 = 0;
        HandleFrame(frame);
    }
}

int main() {
    
    dataOut = 0;
    
    timer0_init();
    
    LPC_PINCON->PINSEL0 |=3<<12;        //P0.6 = MAT2.0

    timer2_init();
    
    dataIn.rise(DataInRise);
    dataIn.fall(DataInFall);
    
    btn1.rise(&ShortMessageInt);
    btn2.rise(&LongMessageInt);
    
    sender.start(SenderMain);
    receiver.start(ReceiverMain);
    
    Thread::signal_wait(0x01);
}
