/*
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __GROVE_COLOUR_SENSOR_HPP__
#define __GROVE_COLOUR_SENSOR_HPP__

#include "mbed.h"

/**
 * This module is based on the color sensor TCS3414CS with digital output over I2C.
 * http://www.seeedstudio.com/wiki/index.php?title=Twig_-_I2C_Color_Sensor_v0.9b
 *
 * example use:
 *
 *     GroveColourSensor colorSensor(I2C_SDA0, I2C_SCL0);
 *
 *     colorSensor.powerUp();
 *     unsigned colour;
 *     for (colour = GroveColourSensor::GREEN; colour < GroveColourSensor::NUM_COLORS; colour++) {
 *         uint16_t colourValue = colorSensor.readColour(colour);
 *         ...
 *     }
 *     colorSensor.powerDown();
 */

class GroveColourSensor {
public:
    enum Colour_t {
        GREEN = 0,
        RED,
        BLUE,
        CLEAR,
        NUM_COLORS
    };

    GroveColourSensor(PinName sda, PinName scl) : i2c(sda, scl) {
        /* empty*/
    }

    void powerUp(void) {
        static const char powerUpCommand[] = {0x80, 0x03};
        /* turn on the color sensor */
        if (i2c.write((SEVEN_BIT_ADDRESS << 1), powerUpCommand, sizeof(powerUpCommand)) != 0) {
            error("failed to power up the sensor");
        }
    }

    void powerDown(void) {
        static const char powerDownCommand[] = {0x80, 0x00};
        /* turn on the color sensor */
        if (i2c.write((SEVEN_BIT_ADDRESS << 1), powerDownCommand, sizeof(powerDownCommand)) != 0) {
            error("failed to power down the sensor");
        }
    }

    uint16_t readColour(Colour_t colour) {
        char readColorRegistersCommand = 0xb0 + (2 * static_cast<unsigned>(colour));
        i2c.write((SEVEN_BIT_ADDRESS << 1), &readColorRegistersCommand, 1 /* size */);

        uint16_t colourValue;
        i2c.read((SEVEN_BIT_ADDRESS << 1), reinterpret_cast<char *>(&colourValue), sizeof(uint16_t));
        return colourValue;
    }

    uint16_t readColour(unsigned colour) {
        if (colour >= NUM_COLORS) {
            return 0;
        }

        return readColour(static_cast<Colour_t>(colour));
    }

private:
    static const uint8_t SEVEN_BIT_ADDRESS = 0x39;

    I2C i2c;
};
#endif /* #ifndef __GROVE_COLOUR_SENSOR_HPP__ */
