/* CSRLocation class for mbed Microcontroller
 * Copyright 2014 CSR plc
 */

#include "mbed.h"
#include "GPSProvider.h"

#define PINMAP_GPIO_BTN  D5
#define PINMAP_GPIO_TEST D10
#define LOC_LED1         D7
#define LOC_LED2         D6

#define APP_DBG_PORT_BAUD       115200
#define CSR_APP_LOG_INFO(...)   sSerialDebug.printf(__VA_ARGS__)

/* appliation commands */
typedef enum AppCmd {
    APP_CMD_IDLE,           // No special command
    APP_CMD_HELP,           // Show the supported commands
    APP_CMD_START,          // Start location
    APP_CMD_STOP,           // Stop location
    APP_CMD_PM_FULL,        // Set full power mode
    APP_CMD_PM_PTF,         // Set low power PTF mode
    APP_CMD_PTF_GETPOS,     // Get location immediately in low power PTF mode
    APP_CMD_RESET,          // Debug command, pull reset pin high level
} eAppCmd;

static void _AppShowCmd(void);
static void _AppBtnPushed(void);
static void _ConsoleRxHandler(void);
static void _AppCmdProcess(char *pCmd);

static int         sAppCmd  = APP_CMD_IDLE;

static DigitalOut  sLedLocOn(LOC_LED1);
static DigitalOut  sLedPosReport(LOC_LED2);
static InterruptIn sBtn(PINMAP_GPIO_BTN);
static DigitalOut  sPinTest(PINMAP_GPIO_TEST);

Serial             sSerialDebug(USBTX, USBRX);

void
locationHandler(const GPSProvider::LocationUpdateParams_t *params)
{
    if (params->valid) {
        /* application specific handling of location data; */
    }
}

int main(void)
{
    sLedLocOn     = 0;
    sLedPosReport = 0;
    sPinTest      = 1;
    sBtn.mode(PullUp);
    sBtn.fall(&_AppBtnPushed);

    /* initialize the debug serial port */
    sSerialDebug.baud(APP_DBG_PORT_BAUD);
    sSerialDebug.attach(&_ConsoleRxHandler);

    GPSProvider gps;
    gps.setPowerMode(GPSProvider::POWER_FULL);
    gps.reset();
    gps.onLocationUpdate(locationHandler);
    CSR_APP_LOG_INFO("Success to new csrLocation.\r\n");

    _AppShowCmd();

    while (true) {
        switch (sAppCmd) {
            case APP_CMD_HELP:
                sAppCmd = APP_CMD_IDLE;
                _AppShowCmd();
                break;
            case APP_CMD_IDLE:
                gps.process();
                sleep();
                break;
            case APP_CMD_START:
                sAppCmd = APP_CMD_IDLE;
                CSR_APP_LOG_INFO("start location.\r\n");
                gps.start();
                sLedLocOn = 1;
                break;
            case APP_CMD_STOP:
                sAppCmd = APP_CMD_IDLE;
                CSR_APP_LOG_INFO("stop location.\r\n");
                gps.stop();
                sLedLocOn = 0;
                break;
            case APP_CMD_RESET:
                sAppCmd   = APP_CMD_IDLE;
                gps.reset();
                CSR_APP_LOG_INFO("reset on.\r\n");
                break;
            case APP_CMD_PTF_GETPOS:
                CSR_APP_LOG_INFO("lpm get pos.\r\n");
                sAppCmd = APP_CMD_IDLE;
                gps.lpmGetImmediateLocation();
                break;
            case APP_CMD_PM_FULL:
                sAppCmd  = APP_CMD_IDLE;
                gps.setPowerMode(GPSProvider::POWER_FULL);
                CSR_APP_LOG_INFO("fpm set.\r\n");
                break;
            case APP_CMD_PM_PTF:
                sAppCmd  = APP_CMD_IDLE;
                gps.setPowerMode(GPSProvider::POWER_LOW);
                CSR_APP_LOG_INFO("lpm ptf set.\r\n");
                break;
        }
    }
}

static void _AppShowCmd(void)
{
    CSR_APP_LOG_INFO("Location commands:\r\n");
    CSR_APP_LOG_INFO("    help     - help to show supported commands\r\n");
    CSR_APP_LOG_INFO("    start    - begin location\r\n");
    CSR_APP_LOG_INFO("    stop     - end location\r\n");
    CSR_APP_LOG_INFO("    fpm      - full power mode\r\n");
    CSR_APP_LOG_INFO("    ptf      - ptf low power mode\r\n");
    CSR_APP_LOG_INFO("    getpos   - get location immediately in low power ptf mode\r\n");
}

static void _AppBtnPushed(void)
{
    sAppCmd = APP_CMD_PTF_GETPOS;
    //	sLedLocOn = !sLedLocOn;
}

static void _ConsoleRxHandler(void)
{
    static char cmd[32] = {0};
    char        ch;

    ch = sSerialDebug.getc();
    sSerialDebug.putc(ch);
    if (ch == '\r') {
        sSerialDebug.putc('\n');
        if (strlen(cmd) > 0) {
            _AppCmdProcess(cmd);
            memset(cmd, 0, sizeof(cmd));
        }
    } else {
        cmd[strlen(cmd)] = ch;
    }
}

static void _AppCmdProcess(char *pCmd)
{
    if (strcmp(pCmd, "help") == 0) {
        sAppCmd = APP_CMD_HELP;
    } else if (strcmp(pCmd, "start") == 0) {
        sAppCmd = APP_CMD_START;
    } else if (strcmp(pCmd, "stop") == 0) {
        sAppCmd = APP_CMD_STOP;
    } else if (strcmp(pCmd, "fpm") == 0) {
        sAppCmd = APP_CMD_PM_FULL;
    } else if (strcmp(pCmd, "ptf") == 0) {
        sAppCmd = APP_CMD_PM_PTF;
    } else if (strcmp(pCmd, "getpos") == 0) {
        sAppCmd = APP_CMD_PTF_GETPOS;
    } else if (strcmp(pCmd, "reset") == 0) {
        sAppCmd = APP_CMD_RESET;
    } else {
        CSR_APP_LOG_INFO("\r\nUnknown command %s\r\n", pCmd);
    }

    CSR_APP_LOG_INFO("\r\n");
}
