/*
Copyright (c) 2014 Aurobo Pty Ltd

=== A test program for the auPower integrated power supply board ===

The auPower power fusion board is a highly integrated complete power 
    solution designed specifically for micro-UAV / RC applications.

The board's i2c interface need to be connected to the PTE0 and PTE1 ports 
    of the Freescale device

For more information, please visit our web-site:
*****************************
*   http://aupilot.com.au   *
*****************************

License:

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "mbed.h"

// device i2c address
#define AUPOWER_I2C_ADDRESS     0xE6

// registers
#define AUPOWER_REG_CTL1        0x0A
#define AUPOWER_REG_CURRENT     0x00
#define AUPOWER_REG_VMAIN       0x02
#define AUPOWER_REG_VSPARE      0x06
#define AUPOWER_REG_TEMP        0x08

#define AUPOWER_LOW_SENS_MODE   0x01  // Low sensitivity mode (amplifier gain = 4x)
#define AUPOWER_HIGH_SENS_MODE  0x02  // High sensitivity mode (amplifier gain = 8x)
#define AUPOWER_RUN             0x07  // Start all sensors in fast-read mode, 
                                      //  sequentially updating every 10ms.

/* 
   There are two current sensing modes - low and high sensitivity.
   In the low sens. mode, the board can measure baypass current 
   up to 220A, while in the high sensitivity mode, the max current 
   is limited by 110A 
*/ 

//#define AUPOWER_HIGH_SENS
#undef AUPOWER_HIGH_SENS

DigitalOut myled(LED1);
Serial pc(USBTX, USBRX);
I2C i2c(PTE0, PTE1); 

// protos
int aupower_init(void);
int aupower_read_sensors(uint8_t reg, float* fValue);
int aupower_read_temperature(float* fValue);

int main()
{
    float temperature;
    float voltageMainBattery;
    float voltageSpareBattery;
    float bypassCurrent;
    
#ifdef AUPOWER_HIGH_SENS
    pc.printf("\n\r\n\rTest program for auPower board, high sensitivity mode\n\r");
#else
    pc.printf("\n\r\n\rTest program for auPower board, low sensitivity mode\n\r");
#endif

    if (aupower_init() != 0 ) {
        pc.printf("\n\rInit Error!");
        for(;;);    // halt
    }

    while(1) {
        myled = 0;
        wait(0.1);
        myled = 1;

        // read temperature
        if (aupower_read_temperature(&temperature) != 0 )
            pc.printf("\n\rError reading temperature");
            
        if (aupower_read_sensors(AUPOWER_REG_VMAIN, &voltageMainBattery) != 0 )
            pc.printf("\n\rError reading Main Battery Voltage");

        if (aupower_read_sensors(AUPOWER_REG_VSPARE, &voltageSpareBattery) != 0 )
            pc.printf("\n\rError reading Spare Battery Voltage");

        if (aupower_read_sensors(AUPOWER_REG_CURRENT, &bypassCurrent) != 0 )
            pc.printf("\n\rError reading Bypass Current");
            

        pc.printf("\n\rTemper (deg C): %.2f, Main Bat (V): %.2f, Spare Bat (V): %.2f, Current (mA): %.0f", 
                temperature, 
                voltageMainBattery, 
                voltageSpareBattery,
                bypassCurrent); 
        wait(2.0);

    }
}

// reads 2 bytes from a register of an i2c device
// return: rd[2]  data read from register
// return: 0 if OK, -1 otherwise
int i2c_reg16_read(int i2c_address, char reg, char* rd)
{
    if (i2c.write(i2c_address, &reg, 1, true) == 0)  // "true" means don't send stop condition
        if (i2c.read(i2c_address, rd, 2) == 0)
            return 0;
    return -1;
}

// writes 1 byte to a register on a i2c device
// return: 0 if OK, -1 otherwise
int i2c_reg8_write(int i2c_address, char reg, const char wd)
{
    char data[2];
    data[0] = reg;
    data[1] = wd;
    if (i2c.write(i2c_address, &data[0], 2, false) == 0)
        return 0;
    return -1;
}

// start the monitor unit (continuously reading all sensors)
int aupower_init(void)
{
    // we can use the initial value of the temperature as a whoami - must be 0x80 on power-on reset before
    // initialisation, but could be arbitary after init
    
    /*
    char data[2];
    if (i2c_reg16_read(AUPOWER_I2C_ADDRESS, AUPOWER_REG_TEMP, data) !=0 )
        return -1;
    if (data[0] != 0x80)
        return -2;  // wrong monitor unit
    */
    
    // initialise the measurement unit
    
#ifdef AUPOWER_HIGH_SENS
    // high sensitivity, 110A max
    if (i2c_reg8_write(AUPOWER_I2C_ADDRESS,AUPOWER_REG_CTL1,AUPOWER_HIGH_SENS_MODE) != 0)
        return -1;
#else 
    // OR low sensitivity, 220A max
    if (i2c_reg8_write(AUPOWER_I2C_ADDRESS,AUPOWER_REG_CTL1,AUPOWER_LOW_SENS_MODE) != 0)
        return -1;
#endif

    // start ADC fast-read mode
    return i2c_reg8_write(AUPOWER_I2C_ADDRESS,AUPOWER_REG_CTL1,AUPOWER_RUN);
}

// read the current/voltage sensors and convert ADC units to si
int aupower_read_sensors(uint8_t sensor, float* fValue)
{
    int16_t tmp1, tmp2;
    char data[2];
    float tmp3;

    if (i2c_reg16_read(AUPOWER_I2C_ADDRESS, sensor, data) !=0 )
        return -1;

    tmp1 = (int16_t)data[0]<<4;
    tmp2 = (int16_t)(data[1]&0xf0)>>4;
    tmp3 = (float)(tmp1+tmp2);
    
    if (sensor == AUPOWER_REG_CURRENT)
#ifdef AUPOWER_HIGH_SENS
        *fValue = (tmp3-38)*24.69;
#else
        *fValue = (tmp3-18)*49.37;
#endif
    if (sensor == AUPOWER_REG_VMAIN)
        *fValue = tmp3*0.01427;
    if (sensor == AUPOWER_REG_VSPARE)
        *fValue = tmp3*0.005404;
        
    return 0;
}

// read the temperature sensor and convert to degrees C
int aupower_read_temperature(float* fValue)
{
    int16_t tmp1, tmp2;
    char data[2];

    if (i2c_reg16_read(AUPOWER_I2C_ADDRESS, AUPOWER_REG_TEMP, data) !=0 )
        return -1;

    tmp1 = (int16_t)data[0]<<1;
    tmp2 = (int16_t)(data[1]&0x80)>>7;
    *fValue = (tmp1+tmp2) * 0.48;

    return 0;
}

