/**
 * @file timer.c
 * @brief mbed-os implementation of the network interface needed for AWS.
 */
#include <stdlib.h>     /* atoi */
#include "network_interface.h"
#include "EthernetInterface.h"

#include "aws_iot_config.h"
#include "aws_iot_log.h"

#include "mbedtls/net_sockets.h"

//=====================================================================================================================
//
// Uses Avnet Sheild (AT&T wireless LTE)
//
//=====================================================================================================================
#ifdef USING_AVNET_SHIELD
#include "WNCInterface.h"
#include "WNCTCPSocketConnection.h"

// Expose serial for WNC boot
extern MODSERIAL pc;
extern char iccidName[21];

// Network socket
WNCTCPSocketConnection* _tcpsocket;

WNCInterface eth_iface;

int net_modem_boot()
{
    INFO("Booting WNC modem...");
    int rc = -1;
    string str;
    //WNCInterface eth_iface;
    
    INFO("...Using Avnet Shield and AT&T wireless LTE");
    rc = eth_iface.init(NULL, &pc);    
    if( !rc ) 
    {
        ERROR("DHCP failed.");
        return 1;
    }               
    INFO("WNC Module %s initialized (%02X).", rc?"IS":"IS NOT", rc);
    
  
    rc = eth_iface.connect();
    if(rc !=0 ) 
    {
        ERROR("Connection failed.");
        return 1;
    }               

    INFO("...IP Address: %s ", eth_iface.getIPAddress());
    
        
    // Get SIM card number (ICCID)
    eth_iface.getICCID(&str);
    strcpy(iccidName, str.c_str());
    INFO("...ICCID: %s", iccidName);
    
    //get object
    //INFO("Reading Certificate");
    //eth_iface.getObject();
    
    return 0;
}

int GetSignalStrength(int16_t *dbm)
{
    INFO("Reading Signal Strength...");
    int16_t rc = 0;
    string str;
  
    rc = eth_iface.getSignalStrength();
    if(rc ==99 ) 
    {
        ERROR("Get Signal Strength Failed.");
        return 1;
    }               

    INFO("...Singal Strength: %d", rc);

    *dbm = rc;
        
    return 0;
}

int GetUpdateStatus(unsigned char *cStatus)
{
    INFO("Reading Update Status...");
    bool rc = 0;
    string str;
  
    rc = eth_iface.getUpdateStatus(cStatus);
    if(rc == false) 
    {
        ERROR("Get Update Status Failed.");
        return 1;
    }               

    INFO("...Update Status: %02X", *cStatus);

    return 0;
}

int GetAllObjects()
{
    INFO("Reading All Objects...");
    int16_t rc = 0;
    string str;
  
    rc = eth_iface.getAllObjects();
    if(rc == 0 ) 
    {
        ERROR("Read All Objects Failed.");
        return 1;
    }               

        
    return 0;
}


void mbedtls_net_init( mbedtls_net_context *ctx )
{
    DEBUG("...mbedtls_net_init()");
                 
    _tcpsocket = new WNCTCPSocketConnection;
    
    return;
}

int mbedtls_net_connect( mbedtls_net_context *ctx, const char *host, const char *port, int proto )
{
    DEBUG("...mbedtls_net_connect");
    int ret = -1;
    
    /* Connect to the server */
    INFO("Connecting with %s\r\n", host);
    ret = _tcpsocket->connect(host, atoi(port));
     
    return ret;
}

int mbedtls_net_recv_timeout( void *ctx, unsigned char *buf, size_t len, uint32_t timeout )
{
    DEBUG("...mbedtls_net_recv_timeout len: %d, timeout: %d", len, timeout);   
    return (int)_tcpsocket->receive((char*)buf, len);
}

int mbedtls_net_send( void *ctx, const unsigned char *buf, size_t len )
{
    DEBUG("...mbedtls_net_send");
    return _tcpsocket->send((char*)buf, len);
}

int mbedtls_net_set_block( mbedtls_net_context *ctx )
{
    DEBUG("...mbedtls_net_set_block");
    _tcpsocket->set_blocking (false,1500);
    return 0;
}

void mbedtls_net_free( mbedtls_net_context *ctx )
{
    DEBUG("...mbedtls_net_free");
    return; 
}
#endif

//=====================================================================================================================
//
// Uses FRDM-K64F wired ethernet
//
//=====================================================================================================================
#ifdef USING_FRDM_K64F_LWIP
// Network Socket
TCPSocket* _tcpsocket;

int net_modem_boot()
{
    // Do nothing
    return 0;  
}

void mbedtls_net_init( mbedtls_net_context *ctx )
{
    DEBUG("...mbedtls_net_init()");  
    EthernetInterface eth_iface;
    
    eth_iface.connect();
    INFO("...Using FRDM-K64F Ethernet LWIP");
    const char *ip_addr = eth_iface.get_ip_address();
    if (ip_addr) {
        INFO("...Client IP Address is %s", ip_addr);
    } else {
        INFO("...No Client IP Address");
    }  
    _tcpsocket = new TCPSocket(&eth_iface);
    
    return;
}

int mbedtls_net_connect( mbedtls_net_context *ctx, const char *host, const char *port, int proto )
{
    DEBUG("...mbedtls_net_connect");
    int ret;
    
    INFO("...Connecting with %s", host);
    ret = _tcpsocket->connect(host, atoi(port));
    if (ret != NSAPI_ERROR_OK) {
        ERROR("Failed to connect");
        return ret;
    }
    
    INFO("...Connected to Amazon!");   
    return ret;
}

int mbedtls_net_recv_timeout( void *ctx, unsigned char *buf, size_t len, uint32_t timeout )
{
    DEBUG("...mbedtls_net_recv_timeout len: %d, timeout: %d", len, timeout);
    int recv = -1;
    
    _tcpsocket->set_timeout(timeout);
    recv = _tcpsocket->recv(buf, len);
        
    if(NSAPI_ERROR_WOULD_BLOCK == recv ||
       recv == 0){     
        DEBUG("...NSAPI_ERROR_WOULD_BLOCK");
        return 0;
    }else if(recv < 0){
        ERROR("...RECV FAIL");
        return -1;
    }else{
        DEBUG("...RECV OK: %d, %d", len, recv);
        return recv;
    }
}

int mbedtls_net_send( void *ctx, const unsigned char *buf, size_t len )
{
    DEBUG("...mbedtls_net_send");  
    int size = -1;

    size = _tcpsocket->send(buf, len);

    if(NSAPI_ERROR_WOULD_BLOCK == size){
        DEBUG("...SEND OK, len = %d", len);
        return len;
    }else if(size < 0){
        ERROR("...SEND FAIL");
        return -1;
    }else{
        DEBUG("...SEND OK, size = %d", size);
        return size;
    }
}

int mbedtls_net_set_block( mbedtls_net_context *ctx )
{
    DEBUG("...mbedtls_net_set_block");
    _tcpsocket->set_blocking(false);
    return 0;
}

void mbedtls_net_free( mbedtls_net_context *ctx )
{
    DEBUG("...mbedtls_net_free");   
    return; 
}
#endif