#ifndef MPU9250_H
#define MPU9250_H

#include "mbed.h"
#include "math.h"
#include "MPU9250-common.h"


// Set initial input parameters
enum Ascale {
	AFS_2G = 0,
	AFS_4G = 1,
	AFS_8G = 2,
	AFS_16G = 3
};

enum Gscale {
	GFS_250DPS = 0,
	GFS_500DPS = 1,
	GFS_1000DPS = 2,
	GFS_2000DPS = 3
};

enum Mscale {
	MFS_14BITS = 0, 	// 0.6 mG per LSB
	MFS_16BITS			// 0.15 mG per LSB
};



class MPU9250 {
public:

// The sufficientMeasurements boolean is used to indicate that the sensor has
// gathered enough measurements to do future sensor measurements reliably.
// After roughly 10 seconds of full speed measuring, it should be accurate
// enough (with the filter updates), so this can then be set to true.
bool sufficientMeasurements;

I2C *i2c;
float beta, zeta;
float deltat;			// integration interval for both filter schemes
float q[4];				// vector to hold quaternion
float eInt[3];			// vector to hold integral error for Mahony method

const uint8_t Ascale;	// AFS_2G, AFS_4G, AFS_8G, AFS_16G
const uint8_t Gscale;	// GFS_250DPS, GFS_500DPS, GFS_1000DPS, GFS_2000DPS
const uint8_t Mscale;	// MFS_14BITS or MFS_16BITS, 14-bit or 16-bit magnetometer resolution
const uint8_t Mmode;	// Either 8 Hz 0x02) or 100 Hz (0x06) magnetometer data ODR

float aRes, gRes, mRes;	// Resolution scales for accelerometer, gyro, magnetometer

float accelCalibration[3];
float accelBias[3];

float gyroCalibration[3];
float gyroBias[3];

float magCalibration[3];
float magBias[3];


MPU9250(I2C *i2cConnection) : i2c(i2cConnection), Ascale(AFS_8G),
							Gscale(GFS_1000DPS), Mscale(MFS_16BITS), Mmode(0x06) {
	// Gyroscope measurement error in rads/s (start at 60 deg/s),
	// then reduce after ~10 s to 3
	float GyroMeasError = PI * (60.0f / 180.0f);
	beta = sqrt(3.0f / 4.0f) * GyroMeasError;

	// Gyroscope measurement drift in rad/s/s (start at 0.0 deg/s/s)
	float GyroMeasDrift = PI * (1.0f / 180.0f);

	// Other free parameter zeta in the Madgwick scheme usually set to a small or zero value
	zeta = sqrt(3.0f / 4.0f) * GyroMeasDrift;

	// Integration interval for both filter schemes
	deltat = 0.0f;

	q[0] = 1.0f;
	q[1] = q[2] = q[3] = 0.0f;

	eInt[0] = eInt[1] = eInt[2] = 0.0f;

	// Get resolution scales
	aRes = getAres();
	gRes = getGres();
	mRes = getMres();

	// Set initial calibration stuff all to 0.0f
	accelCalibration[0] = accelCalibration[1] = accelCalibration[2] = 0.0f;
	accelBias[0] = accelBias[1] = accelBias[2] = 0.0f;
	gyroCalibration[0] = gyroCalibration[1] = gyroCalibration[2] = 0.0f;
	gyroBias[0] = gyroBias[1] = gyroBias[2] = 0.0f;
	magCalibration[0] = magCalibration[1] = magCalibration[2] = 0.0f;

	// User environmental xyz-axes correction in milliGauss, should be automatically calculated
	magBias[0] = 470.0f;
	magBias[1] = 120.0f;
	magBias[2] = 125.0f;

	// We haven't done enough measurements to get accurate values yet!
	sufficientMeasurements = false;
}

void writeByte(uint8_t address, uint8_t subAddress, uint8_t data) {
	char data_write[2];
	data_write[0] = subAddress;
	data_write[1] = data;
	i2c->write(address, data_write, 2, 0);
}

char readByte(uint8_t address, uint8_t subAddress) {
	// `data` will store the register data
	char data[1];
	char data_write[1];
	data_write[0] = subAddress;
	i2c->write(address, data_write, 1, 1); // no stop
	i2c->read(address, data, 1, 0);
	return data[0];
}

void readBytes(uint8_t address, uint8_t subAddress, uint8_t count, uint8_t *dest) {
	char data[14];
	char data_write[1];
	data_write[0] = subAddress;
	i2c->write(address, data_write, 1, 1); // no stop
	i2c->read(address, data, count, 0);

	for (int ii = 0; ii < count; ii++) {
		dest[ii] = data[ii];
	}
}

float getMres() {
	switch (Mscale) {
		// Possible magnetometer scales (and their register bit settings) are:
		// 14 bit resolution (0) and 16 bit resolution (1)
		case MFS_14BITS:
			return 10.0 * 4219.0 / 8190.0; // Proper scale to return milliGauss
		case MFS_16BITS:
			return 10.0 * 4219.0 / 32760.0; // Proper scale to return milliGauss
	}

	return -1.0f;
}


float getGres() {
	switch (Gscale) {
		// Possible gyro scales (and their register bit settings) are:
		// 250 DPS (00), 500 DPS (01), 1000 DPS (10), and 2000 DPS	(11).
		// Here's a bit of an algorith to calculate DPS/(ADC tick) based on that 2-bit value:
		case GFS_250DPS:
			return 250.0/32768.0;
		case GFS_500DPS:
			return 500.0/32768.0;
		case GFS_1000DPS:
			return 1000.0/32768.0;
		case GFS_2000DPS:
			return 2000.0/32768.0;
	}

	return -1.0f;
}


float getAres() {
	switch (Ascale) {
		// Possible accelerometer scales (and their register bit settings) are:
		// 2 Gs (00), 4 Gs (01), 8 Gs (10), and 16 Gs	(11).
		// Here's a bit of an algorith to calculate DPS/(ADC tick) based on that 2-bit value:
		case AFS_2G:
			return 2.0 / 32768.0;
		case AFS_4G:
			return 4.0 / 32768.0;
		case AFS_8G:
			return 8.0 / 32768.0;
		case AFS_16G:
			return 16.0 / 32768.0;
	}

	return -1.0f;
}

uint8_t hasNewData() {
	return readByte(MPU9250_ADDRESS, INT_STATUS) & 0x01;
}

uint8_t getWhoAmI() {
	return readByte(MPU9250_ADDRESS, WHO_AM_I_MPU9250);
}


void readAccelData(float *ax, float *ay, float *az) {
	// x/y/z accel register data stored here
	uint8_t rawData[6];

	// Read the six raw data registers into data array
	readBytes(MPU9250_ADDRESS, ACCEL_XOUT_H, 6, &rawData[0]);

	// Turn the MSB and LSB into a signed 16-bit value
	int16_t axTemp = (int16_t) (((int16_t) rawData[0] << 8) | rawData[1]);
	int16_t ayTemp = (int16_t) (((int16_t) rawData[2] << 8) | rawData[3]);
	int16_t azTemp = (int16_t) (((int16_t) rawData[4] << 8) | rawData[5]);

	// "Return" ax, ay and az in actual g's, depending on resolution
	*ax = (float) axTemp * aRes - accelBias[0];
	*ay = (float) ayTemp * aRes - accelBias[1];
	*az = (float) azTemp * aRes - accelBias[2];
}

void readGyroData(float *gx, float *gy, float *gz) {
	// x/y/z gyro register data stored here
	uint8_t rawData[6];

	// Read the six raw data registers sequentially into data array
	readBytes(MPU9250_ADDRESS, GYRO_XOUT_H, 6, &rawData[0]);

	// Turn the MSB and LSB into a signed 16-bit value
	int16_t gxTemp = (int16_t) (((int16_t) rawData[0] << 8) | rawData[1]);
	int16_t gyTemp = (int16_t) (((int16_t) rawData[2] << 8) | rawData[3]);
	int16_t gzTemp = (int16_t) (((int16_t) rawData[4] << 8) | rawData[5]);

	// "Return" gx, gy and gz in actual deg/s, depending on scale
	*gx = (float) gxTemp * gRes - gyroBias[0];
	*gy = (float) gyTemp * gRes - gyroBias[1];
	*gz = (float) gzTemp * gRes - gyroBias[2];
}

void readMagData(float *mx, float *my, float *mz) {
	// x/y/z gyro register data, ST2 register stored here, must read ST2 at end of data acquisition
	uint8_t rawData[7];

	// Wait for magnetometer data ready bit to be set
	if (readByte(AK8963_ADDRESS, AK8963_ST1) & 0x01) {
		// Read the six raw data and ST2 registers sequentially into data array
		readBytes(AK8963_ADDRESS, AK8963_XOUT_L, 7, &rawData[0]);
		uint8_t c = rawData[6]; // End data read by reading ST2 register

		// Check if magnetic sensor overflow set, if not then report data
		if (!(c & 0x08)) {
			// Turn the MSB and LSB into a signed 16-bit value
			// Data stored as little Endian
			int16_t mxTemp = (int16_t) (((int16_t) rawData[1] << 8) | rawData[0]);
			int16_t myTemp = (int16_t) (((int16_t) rawData[3] << 8) | rawData[2]);
			int16_t mzTemp = (int16_t) (((int16_t) rawData[5] << 8) | rawData[4]);

			// Calculate the magnetometer values in milliGauss. Include factory
			// calibration per data sheet and user environmental corrections
			// "Return" gx, gy and gz in actual deg/s, depending on scale
			*mx = (float) mxTemp * mRes * magCalibration[0] - magBias[0];
			*my = (float) myTemp * mRes * magCalibration[1] - magBias[1];
			*mz = (float) mzTemp * mRes * magCalibration[2] - magBias[2];
		}
	}
}

int16_t readTempData() {
	// x/y/z gyro register data stored here
	uint8_t rawData[2];

	// Read the two raw data registers sequentially into data array
	readBytes(MPU9250_ADDRESS, TEMP_OUT_H, 2, &rawData[0]);

	// Turn the MSB and LSB into a 16-bit value
	return (int16_t) (((int16_t) rawData[0]) << 8 | rawData[1]);
}

float getTemperature() {
	int16_t tempData = readTempData();
	return ((float) tempData) / 333.87f + 21.0f;	// In Celsius
}


void getYawPitchRoll(float *yaw, float *pitch, float *roll, float declination) {
	// Define output variables from updated quaternion---these are
	// Tait-Bryan angles, commonly used in aircraft orientation. In
	// this coordinate system, the positive z-axis is down toward Earth.
	// Yaw is the angle between Sensor x-axis and Earth magnetic North
	// (or true North if corrected for local declination, looking down
	// on the sensor positive yaw is counterclockwise.
	// Pitch is angle between sensor x-axis and Earth ground plane,
	// toward the Earth is positive, up toward the sky is negative.
	// Roll is angle between sensor y-axis and Earth ground plane,
	// y-axis up is positive roll. These arise from the definition of
	// the homogeneous rotation matrix constructed from quaternions.
	// Tait-Bryan angles as well as Euler angles are non-commutative;
	// that is, the get the correct orientation the rotations must be
	// applied in the correct order which for this configuration is
	// yaw, pitch, and then roll.
	// For more see
	// http://en.wikipedia.org/wiki/Conversion_between_quaternions_and_Euler_angles
	// which has additional links.
	*yaw = atan2(2.0f * (q[1] * q[2] + q[0] * q[3]),
		q[0] * q[0] + q[1] * q[1] - q[2] * q[2] - q[3] * q[3]);
	*pitch = -asin(2.0f * (q[1] * q[3] - q[0] * q[2]));
	*roll = atan2(2.0f * (q[0] * q[1] + q[2] * q[3]),
		q[0] * q[0] - q[1] * q[1] - q[2] * q[2] + q[3] * q[3]);

	// Quaternion q is in deg, so convert it to radian
	*pitch = RAD2DEG(*pitch);
	*yaw = RAD2DEG(*yaw)- declination;
	*roll = RAD2DEG(*roll);
}


void resetMPU9250() {
	// Write a one to bit 7 reset bit; toggle reset device
	writeByte(MPU9250_ADDRESS, PWR_MGMT_1, 0x80);
	wait(0.1);
}

void initAK8963() {
	// First extract the factory calibration for each magnetometer axis
	// x/y/z gyro calibration data stored here
	uint8_t rawData[3];

	// Power down magnetometer
	writeByte(AK8963_ADDRESS, AK8963_CNTL, 0x00);
	wait(0.01);

	// Enter Fuse ROM access mode
	writeByte(AK8963_ADDRESS, AK8963_CNTL, 0x0F);
	wait(0.01);

	// Read the x-, y-, and z-axis calibration values
	readBytes(AK8963_ADDRESS, AK8963_ASAX, 3, &rawData[0]);

	// Return x-axis sensitivity adjustment values, etc.
	magCalibration[0] = (float)(rawData[0] - 128) / 256.0f + 1.0f;
	magCalibration[1] = (float)(rawData[1] - 128) / 256.0f + 1.0f;
	magCalibration[2] = (float)(rawData[2] - 128) / 256.0f + 1.0f;

	// Power down magnetometer
	writeByte(AK8963_ADDRESS, AK8963_CNTL, 0x00);
	wait(0.01);

	// Configure the magnetometer for continuous read and highest resolution
	// set Mscale bit 4 to 1 (0) to enable 16 (14) bit resolution in CNTL register,
	// and enable continuous mode data acquisition Mmode (bits [3:0]),
	// 0010 for 8 Hz and 0110 for 100 Hz sample rates
	// Set magnetometer data resolution and sample ODR
	writeByte(AK8963_ADDRESS, AK8963_CNTL, Mscale << 4 | Mmode);
	wait(0.01);
}


void initMPU9250() {
	// Initialize MPU9250 device
	// Wake up device
	// Clear sleep mode bit (6), enable all sensors
	writeByte(MPU9250_ADDRESS, PWR_MGMT_1, 0x00);

	// Delay 100 ms for PLL to get established on x-axis gyro
	// Should check for PLL ready interrupt
	wait(0.1);

	// Get stable time source
	// Set clock source to be PLL with x-axis gyroscope reference, bits 2:0 = 001
	writeByte(MPU9250_ADDRESS, PWR_MGMT_1, 0x01);

	// Configure Gyro and Accelerometer
	// Disable FSYNC and set accelerometer and gyro bandwidth to 44 and 42 Hz, respectively;
	// DLPF_CFG = bits 2:0 = 010; this sets the sample rate at 1 kHz for both
	// Maximum delay is 4.9 ms which is just over a 200 Hz maximum rate
	writeByte(MPU9250_ADDRESS, CONFIG, 0x03);

	// Set sample rate = gyroscope output rate/(1 + SMPLRT_DIV)
	// Use a 200 Hz rate; the same rate set in CONFIG above
	writeByte(MPU9250_ADDRESS, SMPLRT_DIV, 0x04);

	// Set gyroscope full scale range
	// Range selects FS_SEL and AFS_SEL are 0 - 3, so 2-bit values are
	// left-shifted into positions 4:3
	uint8_t c =	readByte(MPU9250_ADDRESS, GYRO_CONFIG);

	// Clear self-test bits [7:5]
	writeByte(MPU9250_ADDRESS, GYRO_CONFIG, c & ~0xE0);

	// Clear AFS bits [4:3]
	writeByte(MPU9250_ADDRESS, GYRO_CONFIG, c & ~0x18);

	// Set full scale range for the gyro
	writeByte(MPU9250_ADDRESS, GYRO_CONFIG, c | Gscale << 3);

	// Set accelerometer configuration
	c =	readByte(MPU9250_ADDRESS, ACCEL_CONFIG);

	// Clear self-test bits [7:5]
	writeByte(MPU9250_ADDRESS, ACCEL_CONFIG, c & ~0xE0);

	// Clear AFS bits [4:3]
	writeByte(MPU9250_ADDRESS, ACCEL_CONFIG, c & ~0x18);

	// Set full scale range for the accelerometer
	writeByte(MPU9250_ADDRESS, ACCEL_CONFIG, c | Ascale << 3);

	// Set accelerometer sample rate configuration
	// It is possible to get a 4 kHz sample rate from the accelerometer by choosing
	// 1 for accel_fchoice_b bit [3]; in this case the bandwidth is 1.13 kHz
	c = readByte(MPU9250_ADDRESS, ACCEL_CONFIG2);

	// Clear accel_fchoice_b (bit 3) and A_DLPFG (bits [2:0])
	writeByte(MPU9250_ADDRESS, ACCEL_CONFIG2, c & ~0x0F);

	// Set accelerometer rate to 1 kHz and bandwidth to 41 Hz
	writeByte(MPU9250_ADDRESS, ACCEL_CONFIG2, c | 0x03);

	// The accelerometer, gyro, and thermometer are set to 1 kHz sample rates,
	// but all these rates are further reduced by a factor of 5 to 200 Hz
	// because of the SMPLRT_DIV setting

	// Configure Interrupts and Bypass Enable
	// Set interrupt pin active high, push-pull, and clear on read of INT_STATUS,
	// enable I2C_BYPASS_EN so additional chips can join the I2C bus and all
	// can be controlled by the Arduino as master
	writeByte(MPU9250_ADDRESS, INT_PIN_CFG, 0x22);

	// Enable data ready (bit 0) interrupt
	writeByte(MPU9250_ADDRESS, INT_ENABLE, 0x01);
}

// Function which accumulates gyro and accelerometer data after device
// initialization. It calculates the average of the at-rest readings and then
// loads the resulting offsets into accelerometer and gyro bias registers.
void calibrateMPU9250() {
	// Data array to hold accelerometer and gyro x, y, z, data
	uint8_t data[12];
	uint16_t ii, packet_count, fifo_count;
	int32_t gyro_bias[3] = {0, 0, 0}, accel_bias[3] = {0, 0, 0};

	// Reset device, reset all registers, clear gyro and accelerometer bias registers
	// Write a one to bit 7 reset bit; toggle reset device
	writeByte(MPU9250_ADDRESS, PWR_MGMT_1, 0x80);
	wait(0.1);

	// Get stable time source
	// Set clock source to be PLL with x-axis gyroscope reference, bits 2:0 = 001
	writeByte(MPU9250_ADDRESS, PWR_MGMT_1, 0x01);
	writeByte(MPU9250_ADDRESS, PWR_MGMT_2, 0x00);
	wait(0.2);

	// Configure device for bias calculation
	// Disable all interrupts
	writeByte(MPU9250_ADDRESS, INT_ENABLE, 0x00);

	// Disable FIFO
	writeByte(MPU9250_ADDRESS, FIFO_EN, 0x00);

	// Turn on internal clock source
	writeByte(MPU9250_ADDRESS, PWR_MGMT_1, 0x00);

	 // Disable I2C master
	writeByte(MPU9250_ADDRESS, I2C_MST_CTRL, 0x00);

	// Disable FIFO and I2C master modes
	writeByte(MPU9250_ADDRESS, USER_CTRL, 0x00);

	// Reset FIFO and DMP
	writeByte(MPU9250_ADDRESS, USER_CTRL, 0x0C);
	wait(0.015);

	// Configure MPU9250 gyro and accelerometer for bias calculation
	// Set low-pass filter to 188 Hz
	writeByte(MPU9250_ADDRESS, CONFIG, 0x01);

	// Set sample rate to 1 kHz
	writeByte(MPU9250_ADDRESS, SMPLRT_DIV, 0x00);

	// Set gyro full-scale to 250 degrees per second, maximum sensitivity
	writeByte(MPU9250_ADDRESS, GYRO_CONFIG, 0x00);

	// Set accelerometer full-scale to 2 g, maximum sensitivity
	writeByte(MPU9250_ADDRESS, ACCEL_CONFIG, 0x00);

	// = 131 LSB/degrees/sec
	uint16_t gyrosensitivity = 131;

	// = 16384 LSB/g
	uint16_t accelsensitivity = 16384;

	// Configure FIFO to capture accelerometer and gyro data for bias calculation
	// Enable FIFO
	writeByte(MPU9250_ADDRESS, USER_CTRL, 0x40);

	// Enable gyro and accelerometer sensors for FIFO (max size 512 bytes in MPU-9250)
	writeByte(MPU9250_ADDRESS, FIFO_EN, 0x78);

	// accumulate 40 samples in 80 milliseconds = 480 bytes
	wait(0.04);

	// At end of sample accumulation, turn off FIFO sensor read
	// Disable gyro and accelerometer sensors for FIFO
	writeByte(MPU9250_ADDRESS, FIFO_EN, 0x00);

	// read FIFO sample count
	readBytes(MPU9250_ADDRESS, FIFO_COUNTH, 2, &data[0]);
	fifo_count = ((uint16_t) data[0] << 8) | data[1];

	// How many sets of full gyro and accelerometer data for averaging
	packet_count = fifo_count/12;

	for (ii = 0; ii < packet_count; ii++) {
		int16_t accel_temp[3] = {0, 0, 0}, gyro_temp[3] = {0, 0, 0};

		// read data for averaging
		readBytes(MPU9250_ADDRESS, FIFO_R_W, 12, &data[0]);

		// Form signed 16-bit integer for each sample in FIFO
		accel_temp[0] = (int16_t) (((int16_t)data[0] << 8) | data[1]);
		accel_temp[1] = (int16_t) (((int16_t)data[2] << 8) | data[3]);
		accel_temp[2] = (int16_t) (((int16_t)data[4] << 8) | data[5]);
		gyro_temp[0] = (int16_t) (((int16_t)data[6] << 8) | data[7]);
		gyro_temp[1] = (int16_t) (((int16_t)data[8] << 8) | data[9]);
		gyro_temp[2] = (int16_t) (((int16_t)data[10] << 8) | data[11]);

		// Sum individual signed 16-bit biases to get accumulated signed 32-bit biases
		accel_bias[0] += (int32_t) accel_temp[0];
		accel_bias[1] += (int32_t) accel_temp[1];
		accel_bias[2] += (int32_t) accel_temp[2];
		gyro_bias[0] += (int32_t) gyro_temp[0];
		gyro_bias[1] += (int32_t) gyro_temp[1];
		gyro_bias[2] += (int32_t) gyro_temp[2];
	}

	// Normalize sums to get average count biases
	accel_bias[0] /= (int32_t) packet_count;
	accel_bias[1] /= (int32_t) packet_count;
	accel_bias[2] /= (int32_t) packet_count;
	gyro_bias[0]	/= (int32_t) packet_count;
	gyro_bias[1]	/= (int32_t) packet_count;
	gyro_bias[2]	/= (int32_t) packet_count;

	// Remove gravity from the z-axis accelerometer bias calculation
	if (accel_bias[2] > 0L) {
		accel_bias[2] -= (int32_t) accelsensitivity;
	} else {
		accel_bias[2] += (int32_t) accelsensitivity;
	}

	// Construct the gyro biases for push to the hardware gyro bias registers,
	// which are reset to zero upon device startup. Divide by 4 to get
	// 32.9 LSB per deg/s to conform to expected bias input format
	data[0] = (-gyro_bias[0]/4	>> 8) & 0xFF;

	// Biases are additive, so change sign on calculated average gyro biases
	data[1] = (-gyro_bias[0]/4) & 0xFF;
	data[2] = (-gyro_bias[1]/4 >> 8) & 0xFF;
	data[3] = (-gyro_bias[1]/4) & 0xFF;
	data[4] = (-gyro_bias[2]/4 >> 8) & 0xFF;
	data[5] = (-gyro_bias[2]/4) & 0xFF;

	// Push gyro biases to hardware registers
	/*
	writeByte(MPU9250_ADDRESS, XG_OFFSET_H, data[0]);
	writeByte(MPU9250_ADDRESS, XG_OFFSET_L, data[1]);
	writeByte(MPU9250_ADDRESS, YG_OFFSET_H, data[2]);
	writeByte(MPU9250_ADDRESS, YG_OFFSET_L, data[3]);
	writeByte(MPU9250_ADDRESS, ZG_OFFSET_H, data[4]);
	writeByte(MPU9250_ADDRESS, ZG_OFFSET_L, data[5]);
	*/
	// Construct gyro bias in deg/s for later manual subtraction
	gyroBias[0] = (float) gyro_bias[0]/(float) gyrosensitivity;
	gyroBias[1] = (float) gyro_bias[1]/(float) gyrosensitivity;
	gyroBias[2] = (float) gyro_bias[2]/(float) gyrosensitivity;

	// Construct the accelerometer biases for push to the hardware
	// accelerometer bias registers. These registers contain factory trim values
	// which must be added to the calculated accelerometer biases; on boot up
	// these registers will hold non-zero values. In addition, bit 0 of the
	// lower byte must be preserved since it is used for temperature compensation
	// calculations. Accelerometer bias registers expect bias input as
	// 2048 LSB per g, so that the accelerometer biases calculated above must
	// be divided by 8.
	// A place to hold the factory accelerometer trim biases
	int32_t accel_bias_reg[3] = {0, 0, 0};

	// Read factory accelerometer trim values
	readBytes(MPU9250_ADDRESS, XA_OFFSET_H, 2, &data[0]);
	accel_bias_reg[0] = (int16_t) ((int16_t)data[0] << 8) | data[1];
	readBytes(MPU9250_ADDRESS, YA_OFFSET_H, 2, &data[0]);
	accel_bias_reg[1] = (int16_t) ((int16_t)data[0] << 8) | data[1];
	readBytes(MPU9250_ADDRESS, ZA_OFFSET_H, 2, &data[0]);
	accel_bias_reg[2] = (int16_t) ((int16_t)data[0] << 8) | data[1];

	// Define mask for temperature compensation bit 0 of lower byte of
	// accelerometer bias registers
	uint32_t mask = 1uL;

	// Define array to hold mask bit for each accelerometer bias axis
	uint8_t mask_bit[3] = {0, 0, 0};

	for (ii = 0; ii < 3; ii++) {
		// If temperature compensation bit is set, record that fact in mask_bit
		if (accel_bias_reg[ii] & mask) {
			mask_bit[ii] = 0x01;
		}
	}

	// Construct total accelerometer bias, including calculated average
	// accelerometer bias from above. Subtract calculated averaged
	// accelerometer bias scaled to 2048 LSB/g (16 g full scale)
	accel_bias_reg[0] -= (accel_bias[0]/8);
	accel_bias_reg[1] -= (accel_bias[1]/8);
	accel_bias_reg[2] -= (accel_bias[2]/8);

	data[0] = (accel_bias_reg[0] >> 8) & 0xFF;
	data[1] = (accel_bias_reg[0]) & 0xFF;

	// Preserve temperature compensation bit when writing back to accelerometer bias registers
	data[1] = data[1] | mask_bit[0];
	data[2] = (accel_bias_reg[1] >> 8) & 0xFF;
	data[3] = (accel_bias_reg[1]) & 0xFF;

	// Preserve temperature compensation bit when writing back to accelerometer bias registers
	data[3] = data[3] | mask_bit[1];
	data[4] = (accel_bias_reg[2] >> 8) & 0xFF;
	data[5] = (accel_bias_reg[2]) & 0xFF;

	// Preserve temperature compensation bit when writing back to accelerometer bias registers
	data[5] = data[5] | mask_bit[2];

	// Apparently this is not working for the acceleration biases in the MPU-9250
	// Are we handling the temperature correction bit properly?
	// Push accelerometer biases to hardware registers
	/*
	writeByte(MPU9250_ADDRESS, XA_OFFSET_H, data[0]);
	writeByte(MPU9250_ADDRESS, XA_OFFSET_L, data[1]);
	writeByte(MPU9250_ADDRESS, YA_OFFSET_H, data[2]);
	writeByte(MPU9250_ADDRESS, YA_OFFSET_L, data[3]);
	writeByte(MPU9250_ADDRESS, ZA_OFFSET_H, data[4]);
	writeByte(MPU9250_ADDRESS, ZA_OFFSET_L, data[5]);
	*/

	// Output scaled accelerometer biases for manual subtraction in the main program
	accelBias[0] = (float)accel_bias[0] / (float)accelsensitivity;
	accelBias[1] = (float)accel_bias[1] / (float)accelsensitivity;
	accelBias[2] = (float)accel_bias[2] / (float)accelsensitivity;
}


// Accelerometer and gyroscope self test; check calibration wrt factory settings
void MPU9250SelfTest(float *destination) {
	// Should return percent deviation from factory trim values,
	// +/- 14 or less deviation is a pass {
	uint8_t rawData[6] = {0, 0, 0, 0, 0, 0};
	uint8_t selfTest[6];
	int16_t gAvg[3], aAvg[3], aSTAvg[3], gSTAvg[3];
	float factoryTrim[6];
	uint8_t FS = 0;

	// Set gyro sample rate to 1 kHz
	writeByte(MPU9250_ADDRESS, SMPLRT_DIV, 0x00);

	// Set gyro sample rate to 1 kHz and DLPF to 92 Hz
	writeByte(MPU9250_ADDRESS, CONFIG, 0x02);

	// Set full scale range for the gyro to 250 dps
	writeByte(MPU9250_ADDRESS, GYRO_CONFIG, 1<<FS);

	// Set accelerometer rate to 1 kHz and bandwidth to 92 Hz
	writeByte(MPU9250_ADDRESS, ACCEL_CONFIG2, 0x02);

	// Set full scale range for the accelerometer to 2 g
	writeByte(MPU9250_ADDRESS, ACCEL_CONFIG, 1<<FS);

	// Get average current values of gyro and acclerometer
	for (int ii = 0; ii < 200; ii++) {
		// Read the six raw data registers into data array
		readBytes(MPU9250_ADDRESS, ACCEL_XOUT_H, 6, &rawData[0]);

		// Turn the MSB and LSB into a signed 16-bit value
		aAvg[0] += (int16_t)(((int16_t)rawData[0] << 8) | rawData[1]);
		aAvg[1] += (int16_t)(((int16_t)rawData[2] << 8) | rawData[3]);
		aAvg[2] += (int16_t)(((int16_t)rawData[4] << 8) | rawData[5]);

		// Read the six raw data registers sequentially into data array
		readBytes(MPU9250_ADDRESS, GYRO_XOUT_H, 6, &rawData[0]);

		// Turn the MSB and LSB into a signed 16-bit value
		gAvg[0] += (int16_t)(((int16_t)rawData[0] << 8) | rawData[1]);
		gAvg[1] += (int16_t)(((int16_t)rawData[2] << 8) | rawData[3]);
		gAvg[2] += (int16_t)(((int16_t)rawData[4] << 8) | rawData[5]);
	}

	// Get average of 200 values and store as average current readings
	for (int ii = 0; ii < 3; ii++) {
		aAvg[ii] /= 200;
		gAvg[ii] /= 200;
	}

	// Configure the accelerometer for self-test
	// Enable self test on all three axes and set accelerometer range to +/- 2 g
	writeByte(MPU9250_ADDRESS, ACCEL_CONFIG, 0xE0);

	// Enable self test on all three axes and set gyro range to +/- 250 degrees/s
	writeByte(MPU9250_ADDRESS, GYRO_CONFIG, 0xE0);

	// Delay a while to let the device stabilize
	wait(0.1);

	// Get average self-test values of gyro and acclerometer
	for (int ii = 0; ii < 200; ii++) {
		// Read the six raw data registers into data array
		readBytes(MPU9250_ADDRESS, ACCEL_XOUT_H, 6, &rawData[0]);

		// Turn the MSB and LSB into a signed 16-bit value
		aSTAvg[0] += (int16_t)(((int16_t)rawData[0] << 8) | rawData[1]);
		aSTAvg[1] += (int16_t)(((int16_t)rawData[2] << 8) | rawData[3]);
		aSTAvg[2] += (int16_t)(((int16_t)rawData[4] << 8) | rawData[5]);

		// Read the six raw data registers sequentially into data array
		readBytes(MPU9250_ADDRESS, GYRO_XOUT_H, 6, &rawData[0]);

		// Turn the MSB and LSB into a signed 16-bit value
		gSTAvg[0] += (int16_t)(((int16_t)rawData[0] << 8) | rawData[1]);
		gSTAvg[1] += (int16_t)(((int16_t)rawData[2] << 8) | rawData[3]);
		gSTAvg[2] += (int16_t)(((int16_t)rawData[4] << 8) | rawData[5]);
	}

	// Get average of 200 values and store as average self-test readings
	for (int ii = 0; ii < 3; ii++) {
		aSTAvg[ii] /= 200;
		gSTAvg[ii] /= 200;
	}

	// Configure the gyro and accelerometer for normal operation
	writeByte(MPU9250_ADDRESS, ACCEL_CONFIG, 0x00);
	writeByte(MPU9250_ADDRESS, GYRO_CONFIG, 0x00);

	// Delay a while to let the device stabilize
	wait(0.1);

	// Retrieve accelerometer and gyro factory Self-Test Code from USR_Reg
	selfTest[0] = readByte(MPU9250_ADDRESS, SELF_TEST_X_ACCEL);
	selfTest[1] = readByte(MPU9250_ADDRESS, SELF_TEST_Y_ACCEL);
	selfTest[2] = readByte(MPU9250_ADDRESS, SELF_TEST_Z_ACCEL);
	selfTest[3] = readByte(MPU9250_ADDRESS, SELF_TEST_X_GYRO);
	selfTest[4] = readByte(MPU9250_ADDRESS, SELF_TEST_Y_GYRO);
	selfTest[5] = readByte(MPU9250_ADDRESS, SELF_TEST_Z_GYRO);

	// Retrieve factory self-test value from self-test code reads
	// FT[Xa] factory trim calculation, and FT[Ya], FT[Xg], etc
	factoryTrim[0] = (float) (2620 / 1<<FS) * (pow(1.01, ((float) selfTest[0] - 1.0)));
	factoryTrim[1] = (float) (2620 / 1<<FS) * (pow(1.01, ((float) selfTest[1] - 1.0)));
	factoryTrim[2] = (float) (2620 / 1<<FS) * (pow(1.01, ((float) selfTest[2] - 1.0)));
	factoryTrim[3] = (float) (2620 / 1<<FS) * (pow(1.01, ((float) selfTest[3] - 1.0)));
	factoryTrim[4] = (float) (2620 / 1<<FS) * (pow(1.01, ((float) selfTest[4] - 1.0)));
	factoryTrim[5] = (float) (2620 / 1<<FS) * (pow(1.01, ((float) selfTest[5] - 1.0)));

	// Report results as a ratio of (STR - FT)/FT; the change from Factory Trim
	// of the Self-Test Response. To get percent, must multiply by 100
	for (int i = 0; i < 3; i++) {
		// Report percent differences
		destination[i] = 100.0 * ((float) (aSTAvg[i] - aAvg[i])) / factoryTrim[i];

		// Report percent differences
		destination[i + 3] = 100.0 * ((float) (gSTAvg[i] - gAvg[i])) / factoryTrim[i + 3];
	}
}



// Implementation of Sebastian Madgwick's "...efficient orientation filter
// for... inertial/magnetic sensor arrays"
// (see http://www.x-io.co.uk/category/open-source/ for examples and more details)
// which fuses acceleration, rotation rate, and magnetic moments to produce a
// quaternion-based estimate of absolute
// device orientation -- which can be converted to yaw, pitch, and roll.
// Useful for stabilizing quadcopters, etc. The performance of the orientation
// filter is at least as good as conventional Kalman-based filtering algorithms
// but is much less computationally intensive---it can be performed on
// a 3.3 V Pro Mini operating at 8 MHz!
void MadgwickQuaternionUpdate(float ax, float ay, float az, float gx, float gy,
							float gz, float mx, float my, float mz) {
	// Short name local variable for readability
	float q1 = q[0], q2 = q[1], q3 = q[2], q4 = q[3];
	float norm;
	float hx, hy, _2bx, _2bz;
	float s1, s2, s3, s4;
	float qDot1, qDot2, qDot3, qDot4;

	// Auxiliary variables to avoid repeated arithmetic
	float _2q1mx;
	float _2q1my;
	float _2q1mz;
	float _2q2mx;
	float _4bx;
	float _4bz;
	float _2q1 = 2.0f * q1;
	float _2q2 = 2.0f * q2;
	float _2q3 = 2.0f * q3;
	float _2q4 = 2.0f * q4;
	float _2q1q3 = 2.0f * q1 * q3;
	float _2q3q4 = 2.0f * q3 * q4;
	float q1q1 = q1 * q1;
	float q1q2 = q1 * q2;
	float q1q3 = q1 * q3;
	float q1q4 = q1 * q4;
	float q2q2 = q2 * q2;
	float q2q3 = q2 * q3;
	float q2q4 = q2 * q4;
	float q3q3 = q3 * q3;
	float q3q4 = q3 * q4;
	float q4q4 = q4 * q4;

	// Normalise accelerometer measurement
	norm = sqrt(ax * ax + ay * ay + az * az);

	// Handle NaN
	if (norm == 0.0f) {
		return;
	}

	norm = 1.0f / norm;
	ax *= norm;
	ay *= norm;
	az *= norm;

	// Normalise magnetometer measurement
	norm = sqrt(mx * mx + my * my + mz * mz);

	// Handle NaN
	if (norm == 0.0f) {
		return;
	}

	norm = 1.0f/norm;
	mx *= norm;
	my *= norm;
	mz *= norm;

	// Reference direction of Earth's magnetic field
	_2q1mx = 2.0f * q1 * mx;
	_2q1my = 2.0f * q1 * my;
	_2q1mz = 2.0f * q1 * mz;
	_2q2mx = 2.0f * q2 * mx;
	hx = mx * q1q1 - _2q1my * q4 + _2q1mz * q3 + mx * q2q2 + _2q2 * my * q3
		+ _2q2 * mz * q4 - mx * q3q3 - mx * q4q4;
	hy = _2q1mx * q4 + my * q1q1 - _2q1mz * q2 + _2q2mx * q3 - my * q2q2
		+ my * q3q3 + _2q3 * mz * q4 - my * q4q4;
	_2bx = sqrt(hx * hx + hy * hy);
	_2bz = -_2q1mx * q3 + _2q1my * q2 + mz * q1q1 + _2q2mx * q4 - mz * q2q2
		+ _2q3 * my * q4 - mz * q3q3 + mz * q4q4;
	_4bx = 2.0f * _2bx;
	_4bz = 2.0f * _2bz;

	// Gradient descent algorithm corrective step
	s1 = -_2q3 * (2.0f * q2q4 - _2q1q3 - ax) + _2q2 * (2.0f * q1q2 + _2q3q4 - ay)
		- _2bz * q3 * (_2bx * (0.5f - q3q3 - q4q4) + _2bz * (q2q4 - q1q3) - mx)
		+ (-_2bx * q4 + _2bz * q2) * (_2bx * (q2q3 - q1q4) + _2bz * (q1q2 + q3q4) - my)
		+ _2bx * q3 * (_2bx * (q1q3 + q2q4) + _2bz * (0.5f - q2q2 - q3q3) - mz);
	s2 = _2q4 * (2.0f * q2q4 - _2q1q3 - ax) + _2q1 * (2.0f * q1q2 + _2q3q4 - ay)
		- 4.0f * q2 * (1.0f - 2.0f * q2q2 - 2.0f * q3q3 - az)
		+ _2bz * q4 * (_2bx * (0.5f - q3q3 - q4q4) + _2bz * (q2q4 - q1q3) - mx)
		+ (_2bx * q3 + _2bz * q1) * (_2bx * (q2q3 - q1q4) + _2bz * (q1q2 + q3q4) - my)
		+ (_2bx * q4 - _4bz * q2) * (_2bx * (q1q3 + q2q4)
		+ _2bz * (0.5f - q2q2 - q3q3) - mz);
	s3 = -_2q1 * (2.0f * q2q4 - _2q1q3 - ax) + _2q4 * (2.0f * q1q2 + _2q3q4 - ay)
		- 4.0f * q3 * (1.0f - 2.0f * q2q2 - 2.0f * q3q3 - az)
		+ (-_4bx * q3 - _2bz * q1) * (_2bx * (0.5f - q3q3 - q4q4)
		+ _2bz * (q2q4 - q1q3) - mx) + (_2bx * q2 + _2bz * q4) * (_2bx * (q2q3 - q1q4)
		+ _2bz * (q1q2 + q3q4) - my) + (_2bx * q1 - _4bz * q3) * (_2bx * (q1q3 + q2q4)
		+ _2bz * (0.5f - q2q2 - q3q3) - mz);
	s4 = _2q2 * (2.0f * q2q4 - _2q1q3 - ax) + _2q3 * (2.0f * q1q2
		+ _2q3q4 - ay) + (-_4bx * q4 + _2bz * q2) * (_2bx * (0.5f - q3q3 - q4q4)
		+ _2bz * (q2q4 - q1q3) - mx) + (-_2bx * q1 + _2bz * q3) * (_2bx * (q2q3 - q1q4)
		+ _2bz * (q1q2 + q3q4) - my) + _2bx * q2 * (_2bx * (q1q3 + q2q4)
		+ _2bz * (0.5f - q2q2 - q3q3) - mz);

	// Normalise step magnitude
	norm = sqrt(s1 * s1 + s2 * s2 + s3 * s3 + s4 * s4);
	norm = 1.0f / norm;
	s1 *= norm;
	s2 *= norm;
	s3 *= norm;
	s4 *= norm;

	// Compute rate of change of quaternion
	qDot1 = 0.5f * (-q2 * gx - q3 * gy - q4 * gz) - beta * s1;
	qDot2 = 0.5f * (q1 * gx + q3 * gz - q4 * gy) - beta * s2;
	qDot3 = 0.5f * (q1 * gy - q2 * gz + q4 * gx) - beta * s3;
	qDot4 = 0.5f * (q1 * gz + q2 * gy - q3 * gx) - beta * s4;

	// Integrate to yield quaternion
	q1 += qDot1 * deltat;
	q2 += qDot2 * deltat;
	q3 += qDot3 * deltat;
	q4 += qDot4 * deltat;

	// Normalise quaternion
	norm = sqrt(q1 * q1 + q2 * q2 + q3 * q3 + q4 * q4);
	norm = 1.0f / norm;
	q[0] = q1 * norm;
	q[1] = q2 * norm;
	q[2] = q3 * norm;
	q[3] = q4 * norm;
}


// Similar to Madgwick scheme but uses proportional and integral filtering on
// the error between estimated reference vectors and measured ones.
void MahonyQuaternionUpdate(float ax, float ay, float az, float gx, float gy,
							float gz, float mx, float my, float mz) {
	// short name local variable for readability
	float q1 = q[0], q2 = q[1], q3 = q[2], q4 = q[3];
	float norm;
	float hx, hy, bx, bz;
	float vx, vy, vz, wx, wy, wz;
	float ex, ey, ez;
	float pa, pb, pc;

	// Auxiliary variables to avoid repeated arithmetic
	float q1q1 = q1 * q1;
	float q1q2 = q1 * q2;
	float q1q3 = q1 * q3;
	float q1q4 = q1 * q4;
	float q2q2 = q2 * q2;
	float q2q3 = q2 * q3;
	float q2q4 = q2 * q4;
	float q3q3 = q3 * q3;
	float q3q4 = q3 * q4;
	float q4q4 = q4 * q4;

	// Normalise accelerometer measurement
	norm = sqrt(ax * ax + ay * ay + az * az);

	// Handle NaN
	if (norm == 0.0f) {
		return;
	}

	// Use reciprocal for division
	norm = 1.0f / norm;
	ax *= norm;
	ay *= norm;
	az *= norm;

	// Normalise magnetometer measurement
	norm = sqrt(mx * mx + my * my + mz * mz);

	// Handle NaN
	if (norm == 0.0f) {
		return;
	}

	// Use reciprocal for division
	norm = 1.0f / norm;
	mx *= norm;
	my *= norm;
	mz *= norm;

	// Reference direction of Earth's magnetic field
	hx = 2.0f * mx * (0.5f - q3q3 - q4q4) + 2.0f * my * (q2q3 - q1q4)
		+ 2.0f * mz * (q2q4 + q1q3);
	hy = 2.0f * mx * (q2q3 + q1q4) + 2.0f * my * (0.5f - q2q2 - q4q4)
		+ 2.0f * mz * (q3q4 - q1q2);
	bx = sqrt((hx * hx) + (hy * hy));
	bz = 2.0f * mx * (q2q4 - q1q3) + 2.0f * my * (q3q4 + q1q2)
		+ 2.0f * mz * (0.5f - q2q2 - q3q3);

	// Estimated direction of gravity and magnetic field
	vx = 2.0f * (q2q4 - q1q3);
	vy = 2.0f * (q1q2 + q3q4);
	vz = q1q1 - q2q2 - q3q3 + q4q4;
	wx = 2.0f * bx * (0.5f - q3q3 - q4q4) + 2.0f * bz * (q2q4 - q1q3);
	wy = 2.0f * bx * (q2q3 - q1q4) + 2.0f * bz * (q1q2 + q3q4);
	wz = 2.0f * bx * (q1q3 + q2q4) + 2.0f * bz * (0.5f - q2q2 - q3q3);

	// Error is cross product between estimated direction and measured direction of gravity
	ex = (ay * vz - az * vy) + (my * wz - mz * wy);
	ey = (az * vx - ax * vz) + (mz * wx - mx * wz);
	ez = (ax * vy - ay * vx) + (mx * wy - my * wx);

	if (Ki > 0.0f) {
		// Accumulate integral error
		eInt[0] += ex;
		eInt[1] += ey;
		eInt[2] += ez;
	} else {
		// Prevent integral wind up
		eInt[0] = 0.0f;
		eInt[1] = 0.0f;
		eInt[2] = 0.0f;
	}

	// Apply feedback terms
	gx = gx + Kp * ex + Ki * eInt[0];
	gy = gy + Kp * ey + Ki * eInt[1];
	gz = gz + Kp * ez + Ki * eInt[2];

	// Integrate rate of change of quaternion
	pa = q2;
	pb = q3;
	pc = q4;
	q1 = q1 + (-q2 * gx - q3 * gy - q4 * gz) * (0.5f * deltat);
	q2 = pa + (q1 * gx + pb * gz - pc * gy) * (0.5f * deltat);
	q3 = pb + (q1 * gy - pa * gz + pc * gx) * (0.5f * deltat);
	q4 = pc + (q1 * gz + pa * gy - pb * gx) * (0.5f * deltat);

	// Normalise quaternion
	norm = sqrt(q1 * q1 + q2 * q2 + q3 * q3 + q4 * q4);
	norm = 1.0f / norm;
	q[0] = q1 * norm;
	q[1] = q2 * norm;
	q[2] = q3 * norm;
	q[3] = q4 * norm;
}

};

#endif
