#include "Target_Planer.h"

Target_Planer::Target_Planer(double *Pathx, double *Pathy, uint16_t N, double R)
{
    for(uint16_t i = 0; i < N; i++) {
        Path_.x.push_back(Pathx[i]);
        Path_.y.push_back(Pathy[i]);
    }
    Path_.N = N;

    RobotPos_.x = -999.999;
    RobotPos_.y = -999.999;

    TargetPos_.x = -999.999;
    TargetPos_.y = -999.999;
    
    TargetAng_ = -999.999;

    R_ = R;
    del_ = 0.001;    // in meter
    i_min_ = 65535;  // biggest number for uint16_t = 2^16-1 = 65535
}

Target_Planer::~Target_Planer() {}

void Target_Planer::initialize(double RobotPosx, double RobotPosy)
{
    RobotPos_.x = RobotPosx;
    RobotPos_.y = RobotPosy;
    i_min_ = findClosestPointOnPath(Path_, RobotPos_);
}

void Target_Planer::update(double RobotPosx, double RobotPosy)
{
    // update robot position
    RobotPos_.x = RobotPosx;
    RobotPos_.y = RobotPosy;

    /* project on path */

    // check if next point is closer and update if yes
    double d_min = calcDistSquared(Path_, RobotPos_, i_min_);
    if(i_min_ < Path_.N-1) {
        double d_ip1 = calcDistSquared(Path_, RobotPos_, i_min_+1);
        if(d_ip1 <= d_min) {
            d_min = d_ip1;
            i_min_ = i_min_ + 1;
        }
    }
    // since root function is monotonic calculate square root after evaluation
    d_min = sqrt(d_min);

    // already treat case where no projection lies on a path segment:
    // if it is a corner and no closer valid projection then this already is the
    // projection point of the robot to the path (corner point)
    double sx = Path_.x[i_min_];
    double sy = Path_.y[i_min_];
    double dl1 = d_min;
    uint16_t i_next = i_min_;
    double a_i;

    if(i_min_ == 0) {
        projectOnPath(Path_, RobotPos_, 1, &a_i, &sx, &sy, &dl1);
        if(a_i < 0) {
            i_next = 0; // robot must be before first path segment
        } else {
            i_next = 1; // robot projection is on first path segment
        }
    } else if(i_min_ == Path_.N-1) {
        projectOnPath(Path_, RobotPos_, Path_.N-2, &a_i, &sx, &sy, &dl1);
        i_next = Path_.N-1;
    } else {
        // check if projection is valid and which projection point is the
        // closest -> actual projection on path
        for(uint16_t i = i_min_-1; i <= i_min_; i++) {
            double sx_i, sy_i, dl1_i;
            double dp_i_norm = calcNormPathsegment_i(Path_, i);
            projectOnPath(Path_, RobotPos_, i, &a_i, &sx_i, &sy_i, &dl1_i);
            // true if robot is on path segment
            if(a_i > 0 - del_ && a_i < dp_i_norm + del_) {
                // true if distance to projection point is shorter
                if(dl1_i < dl1) {
                    sx = sx_i;
                    sy = sy_i;
                    dl1 = dl1_i;
                    i_next = i + 1;
                }
            }
        }
    }
    
    /* based on this projection find target point along path (intersection with radius R) */
    
    if(dl1 >= R_ - del_) {
        TargetPos_.x = sx;
        TargetPos_.y = sy;
    } else {
        double i_ahead = i_next;
        double px_i = sx;
        double py_i = sy;
        double px_ip1 = Path_.x[i_ahead];
        double py_ip1 = Path_.y[i_ahead];
        bool isvalid;
        double x_intsec_valid, y_intsec_valid;
        while(1) {
            findIntersectionWithPath(px_i, py_i, px_ip1, py_ip1, RobotPos_.x, RobotPos_.y, &isvalid, &x_intsec_valid, &y_intsec_valid);
            if(isvalid) {
                TargetPos_.x = x_intsec_valid;
                TargetPos_.y = y_intsec_valid;
                break;
            } else {
                i_ahead = i_ahead + 1;
                if(i_ahead > Path_.N-1) { // TargetPos has to be the endpoint
                    i_ahead = Path_.N-1;
                    TargetPos_.x = Path_.x[i_ahead];
                    TargetPos_.y = Path_.y[i_ahead];
                    break;
                }
                px_i   = Path_.x[i_ahead-1];
                py_i   = Path_.y[i_ahead-1];
                px_ip1 = Path_.x[i_ahead];
                py_ip1 = Path_.y[i_ahead];
            }
        }
    }
    
    // restore R for special cases before first point or after last point
    double dTargetPosx = TargetPos_.x - RobotPos_.x;
    double dTargetPosy = TargetPos_.y - RobotPos_.y;
    double dTargetPos_norm = sqrt(dTargetPosx*dTargetPosx + dTargetPosy*dTargetPosy);
    if(dTargetPos_norm > R_ + del_) {
        TargetPos_.x = RobotPos_.x + R_*dTargetPosx/dTargetPos_norm;
        TargetPos_.y = RobotPos_.y + R_*dTargetPosy/dTargetPos_norm;
    }

    // calculate target angle
    TargetAng_ = atan2(TargetPos_.y - RobotPos_.y, TargetPos_.x - RobotPos_.x);
    
}

void Target_Planer::readTargetPos(double *TargetPosx, double *TargetPosy)
{
    *TargetPosx = TargetPos_.x;
    *TargetPosy = TargetPos_.y;
}

void Target_Planer::readTargetAng(double *TargetAng)
{
    *TargetAng = TargetAng_;
}

void Target_Planer::readTarget(double *TargetPosx, double *TargetPosy, double *TargetAng)
{
    *TargetPosx = TargetPos_.x;
    *TargetPosy = TargetPos_.y;
    *TargetAng = TargetAng_;
}

double Target_Planer::readTargetPosx()
{
    return TargetPos_.x;
}
    
double Target_Planer::readTargetPosy()
{
    return TargetPos_.y;
}

double Target_Planer::readTargetAng()
{
    return TargetAng_;
}

void Target_Planer::updateAndReturnTarget(double RobotPosx, double RobotPosy, double *TargetPosx, double *TargetPosy, double *TargetAng)
{
    update(RobotPosx, RobotPosy);
    *TargetPosx = TargetPos_.x;
    *TargetPosy = TargetPos_.y;
    *TargetAng  = TargetAng_;
}

uint16_t Target_Planer::findClosestPointOnPath(PATH Path, ROBOTPOS RobotPos)
{
    double   d_min = 1000000;
    uint16_t i_min = 0;
    for(uint16_t i = 0; i < Path.N; i++) {
        double dx = Path.x[i] - RobotPos.x;
        double dy = Path.y[i] - RobotPos.y;
        // since root function is monotonic no need to evaluate root itself
        double d = dx*dx + dy*dy;
        if(i == 0) {
            d_min = d;
            i_min = 0;
        } else {
            if(d <= d_min) {
                d_min = d;
                i_min = i;
            }
        }
    }
    return i_min;
}

double Target_Planer::calcDistSquared(PATH Path, ROBOTPOS RobotPos, uint16_t i)
{
    double dx = Path.x[i] - RobotPos.x;
    double dy = Path.y[i] - RobotPos.y;
    return dx*dx + dy*dy;
}

double Target_Planer::calcNormPathsegment_i(PATH Path, uint16_t i)
{
    double dx = Path.x[i+1] - Path.x[i];
    double dy = Path.y[i+1] - Path.y[i];
    return sqrt(dx*dx + dy*dy);
}

void Target_Planer::projectOnPath(PATH Path, ROBOTPOS RobotPos, uint16_t i, double *a_i, double *sx_i, double *sy_i, double *dl1_i)
{
    // delta pathsegment
    double dpx_i = Path.x[i+1] - Path.x[i];
    double dpy_i = Path.y[i+1] - Path.y[i];

    // norm delta pathsegment
    double s = 1/sqrt(dpx_i*dpx_i + dpy_i*dpy_i);
    double dpx_in = s*dpx_i;
    double dpy_in = s*dpy_i;

    // a_i:   scalar length of robot position projected on pathsegment
    // sx_i:  x-value of projection point s
    // sy_i:  y-value of projection point s
    // dl1_i: scalar length of robot position to projection point s (length of tangent)
    *a_i = (RobotPos.x - Path.x[i])*dpx_in + (RobotPos.y - Path.y[i])*dpy_in;
    *sx_i = Path.x[i] + *a_i * dpx_in;
    *sy_i = Path.y[i] + *a_i * dpy_in;
    double dx = *sx_i - RobotPos.x;
    double dy = *sy_i - RobotPos.y;
    *dl1_i = sqrt(dx*dx + dy*dy);
}

void Target_Planer::findIntersectionWithPath(double px_i, double py_i, double px_ip1, double py_ip1, double rx, double ry, bool *isvalid, double *x_intsec_valid, double *y_intsec_valid)
{
    // transform data on pathsegment so that origin is in waypoint_0 and x points towards waypoint_1 
    double wp1x, wp1y, x_act, y_act, s_phi, c_phi;
    transformDataOnPath(px_i, py_i, px_ip1, py_ip1, rx, ry, &wp1x, &wp1y, &x_act, &y_act, &s_phi, &c_phi);
    
    // coefficients for intersection calculus
    double radic = R_*R_ - wp1y*wp1y + 2.0*wp1y*y_act - y_act*y_act;;
    
    // intersection candidates
    double sqrt_radic;
    uint8_t possible_solutions;
    if(radic > 0.0) {
        // 2 intersections
        sqrt_radic = sqrt(radic);
        possible_solutions = 2;
    } else {
        // 1 contact point or tangent point
        sqrt_radic = 0.0;
        possible_solutions = 1;
    }
    
    // check if the solution is valid with bounded box
    double dist_to_wp1_min = 1000000; // big number
    double sign_radicand = 1.0;
    double x_intsec_valid_temp = -999.999;
    double y_intsec_valid_temp = -999.999;
    bool   isvalid_temp = false;
    for(uint8_t i = 0; i < possible_solutions-1; i++) {
        double lowb_x, uppb_x, lowb_y, uppb_y;
        if(0.0 < wp1x) {
            lowb_x = 0.0;
            uppb_x = wp1x;
        } else {
            lowb_x = wp1x;
            uppb_x = 0.0;
        }
        if(0.0 < wp1y) {
            lowb_y = 0.0;
            uppb_y = wp1y;
        } else {
            lowb_y = wp1y;
            uppb_y = 0.0;
        }
        double x_intsec_cand = x_act + sign_radicand*sqrt_radic; // intersection candidate x
        double y_intsec_cand = wp1y;                             // intersection candidate y
        if(lowb_x - del_ < x_intsec_cand && x_intsec_cand < uppb_x + del_) {
            if(lowb_y - del_ < y_intsec_cand && y_intsec_cand < uppb_y + del_) {
                if((x_intsec_cand - x_act)*(x_intsec_cand - x_act) + (y_intsec_cand - y_act)*(y_intsec_cand - y_act) <= (R_ + del_)*(R_ + del_)) {
                    // valid if it shorter than R, del treats numerical uncertainty
                    isvalid_temp = true;
                    double dist_to_wp1 = sqrt((wp1x - x_intsec_cand)*(wp1x - x_intsec_cand) + (wp1y - y_intsec_cand)*(wp1y - y_intsec_cand));
                    if(dist_to_wp1 <= dist_to_wp1_min) {
                        // if there are 2 solutions we have to guarantee that we take the closer one ahead (we creep along the path)
                        dist_to_wp1_min = dist_to_wp1;
                        x_intsec_valid_temp = x_intsec_cand;
                        y_intsec_valid_temp = y_intsec_cand;
                    }
                }
            }
        }
        sign_radicand = -1.0;
    }
    
    // inverse transformation of solution if valid
    if(isvalid_temp) {
        inverseTransformDataOnPath(px_i, py_i, s_phi, c_phi, &x_intsec_valid_temp, &y_intsec_valid_temp);
    }
    
    // copy variables
    *isvalid = isvalid_temp;
    *x_intsec_valid = x_intsec_valid_temp;
    *y_intsec_valid = y_intsec_valid_temp;
    
}

void Target_Planer::transformDataOnPath(double px_i, double py_i, double px_ip1, double py_ip1, double rx, double ry, double *wp1x, double *wp1y, double *x_act, double *y_act, double *s_phi, double *c_phi)
{
    double phi = atan2(py_ip1 - py_i, px_ip1 - px_i);
    double sin_phi = sin(phi);
    double cos_phi = cos(phi);
    
    *wp1x  =  cos_phi*(px_ip1 - px_i) + sin_phi*(py_ip1 - py_i);
    *wp1y  = -sin_phi*(px_ip1 - px_i) + cos_phi*(py_ip1 - py_i);
    *x_act =  cos_phi*(rx - px_i) + sin_phi*(ry - py_i);
    *y_act = -sin_phi*(rx - px_i) + cos_phi*(ry - py_i);  
    *s_phi = sin_phi;
    *c_phi = cos_phi;
}

void Target_Planer::inverseTransformDataOnPath(double px_i, double py_i, double s_phi, double c_phi, double *x_intsec_valid_temp, double *y_intsec_valid_temp)
{
    double x_temp = *x_intsec_valid_temp;
    double y_temp = *y_intsec_valid_temp;
    *x_intsec_valid_temp =  c_phi*x_temp - s_phi*y_temp + px_i;
    *y_intsec_valid_temp =  s_phi*x_temp + c_phi*y_temp + py_i;
}

/* debug */

double Target_Planer::returnPathxAtIndex(int i)
{
    return Path_.x[i];
}

double Target_Planer::returnPathyAtIndex(int i)
{
    return Path_.y[i];
}

uint16_t Target_Planer::returnPathLength()
{
    return Path_.N;
}

uint16_t Target_Planer::returnClosestPointOnPath(double RobotPosx, double RobotPosy)
{
    RobotPos_.x = RobotPosx;
    RobotPos_.y = RobotPosy;
    return findClosestPointOnPath(Path_, RobotPos_);
}

uint16_t Target_Planer::return_i_min_()
{
    return i_min_;
}


