/*
Copyright (c) 2012, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

/*  MS5611 Modifications
Aerodyne Labs
Matthew Nelson - 2014
This library was originally the MS5607 library.  It has been modified to 
to be used the MS5611 Pressure sensor
------------
IndNav: Made adaptions, even though, the original works well, problem:
the original has a delay of 8ms from writing to reading (see line 73 in MS5611I2c
Solution: seperate write and read in 2 processes (hene's hint):
    - write at end of data_read thread, without waiting
    - at beginning of next call, read the data
since there are 2 seperate reading/writings (one for pressure(altitude), one for Temperature -and we need both!)
toggle the 2 processes, this results, that only every 2nd pressure/altitude value is a new value!

altb 20.2.2020
*/

#ifndef MS5611_BASE_H
#define MS5611_BASE_H

class MS5611Base {
public:
    void printCoefficients() {
        printf("%d, %d, %d, %d, %d, %d \r\n", c1, c2, c3, c4, c5, c6);
    }

    int getRawPressure() {
        return readADC(ADC_D1 | OSR_4096);  // fuer 1024: gibt das 4
    }
    int getRawPressure_read() {
        return readADC_read(ADC_D1 | OSR_4096);  // fuer 1024: gibt das 4
    }
    int getRawTemperature() {
        return readADC(ADC_D2 | OSR_4096);  // fuer 1024: gibt das 20
    }
    int getRawTemperature_read() {
        return readADC_read(ADC_D2 | OSR_4096);  // fuer 1024: gibt das 20
    }
 
    float getTemperature() {
        int dT = getRawTemperature() - (c5 << 8);
        int temp = 2000 + ((dT * c6) >> 23);
        
        // 2nd order temperature compensation
        if (temp < 2000) {
            int t2 = (int64_t) dT * dT >> 31;
            temp -= t2;
        }

        return float(temp) / 100;
    }

    float getPressure() {
        int dT = getRawTemperature() - (c5 << 8);
        int temp = 2000 + ((dT * c6) >> 23);
        int64_t off = ((int64_t) c2 << 16) + ((int64_t) dT * c4 >> 7);
        int64_t sens = ((int64_t) c1 << 15) + ((int64_t) dT * c3 >> 8);

        // 2nd order temperature compensation
        if (temp < 2000) {
            int64_t off2 = (int64_t) 5 * (temp - 2000) * (temp - 2000) >> 1;
            int64_t sens2 = (int64_t) 5 * (temp - 2000) * (temp - 2000) >> 2;
            if (temp < -1500) {
                off2 += (int64_t) 7 * (temp + 1500) * (temp + 1500);
                sens2 += (int64_t) 11 * (temp + 1500) * (temp + 1500) >> 1;
            }
            off -= off2;
            sens -= sens2;
        }

        return float((((int64_t) getRawPressure() * sens >> 21) - off) >> 15);
    }
    float getPressure(int dT) {
//        int dT = getRawTemperature() - (c5 << 8);
        int temp = 2000 + ((dT * c6) >> 23);
        int64_t off = ((int64_t) c2 << 16) + ((int64_t) dT * c4 >> 7);
        int64_t sens = ((int64_t) c1 << 15) + ((int64_t) dT * c3 >> 8);

        // 2nd order temperature compensation
        if (temp < 2000) {
            int64_t off2 = (int64_t) 5 * (temp - 2000) * (temp - 2000) >> 1;
            int64_t sens2 = (int64_t) 5 * (temp - 2000) * (temp - 2000) >> 2;
            if (temp < -1500) {
                off2 += (int64_t) 7 * (temp + 1500) * (temp + 1500);
                sens2 += (int64_t) 11 * (temp + 1500) * (temp + 1500) >> 1;
            }
            off -= off2;
            sens -= sens2;
        }

        return float((((int64_t) getRawPressure() * sens >> 21) - off) >> 15);
    }

    float getAltitude(int presssure = 0) {
        return toAltitude(presssure ? presssure : (int) getPressure());
    }

    float getAltitude_toggle_Temp_read(int presssure = 0) {
        if(temp_toggle)
            {
                dT_keep = getRawTemperature_read() - (c5 << 8);
                temp_toggle = false;
            }
        else
            {
                temp_toggle = true;
                old_alt = toAltitude(presssure ? presssure : (int) getPressure(dT_keep));
            }
        return old_alt;  
    }

    int getAltitude_toggle_Temp_write(int presssure = 0) {
        if(temp_toggle)
            {
               return readADC_write(ADC_D2 | OSR_4096);
            }
        else
            {
                return readADC_write(ADC_D1 | OSR_4096);
            }
    }
    int dT_keep;
protected:
    int32_t c1, c2, c3, c4, c5, c6;
    bool temp_toggle;
    float old_alt;      // get altitude only every 2nd time.
    enum {
        RESET     = 0x1E,
        ADC_READ  = 0x00,
        ADC_CONV  = 0x40,
        ADC_D1    = 0x00,
        ADC_D2    = 0x10,
        OSR_256   = 0x00,
        OSR_512   = 0x02,
        OSR_1024  = 0x04,
        OSR_2048  = 0x06,
        OSR_4096  = 0x08,
        PROM_READ = 0xA0
    };

    virtual void writeCommand(int command, int ms = 0) = 0;
    virtual int readPROM(int address) = 0;
    virtual int readADC(int command) = 0;
    virtual int readADC_write(int command) = 0;
    virtual int readADC_read(int command) = 0;

    void init() {
        writeCommand(RESET, 3);
        c1 = readPROM(1);
        c2 = readPROM(2);
        c3 = readPROM(3);
        c4 = readPROM(4);
        c5 = readPROM(5);
        c6 = readPROM(6);
        dT_keep = 2000;
        temp_toggle = true;
        old_alt = 0.0;
    }

    float toAltitude(int pressure) {
        // Ref. 29124-AltimeterAppNote1.pdf
        const float R = 287.052; // specific gas constant R*/M0
        const float g = 9.80665; // standard gravity 
        const float t_grad = 0.0065; // gradient of temperature
        const float t0 = 273.15 + 15; // temperature at 0 altitude
        const float p0 = 101325; // pressure at 0 altitude

        return t0 / t_grad * (1 - exp((t_grad * R / g) * log(pressure / p0)));
    }
};

#endif