#ifndef TFMINI_I2C_H
#define TFMINI_I2C_H

# include <cstdint>
# include "mbed.h"

# define TFMINI_ADDRESS_DEFAULT         0x10

// data register address
#define TFMINI_READ_DATA_REG_H          0x01
#define TFMINI_READ_DATA_REG_L          0x02

// general parameter register address
#define TFMINI_RNG_GEAR_H               0x00
#define TFMINI_RNG_GEAR_L               0x50

#define TFMINI_DET_PAT_H                0x00
#define TFMINI_DET_PAT_L                0x51

#define TFMINI_SET_RNG_LIM_H            0x00
#define TFMINI_SET_RNG_LIM_L            0x55

#define TFMINI_RNG_OUT_LIM_THR_H        0x00
#define TFMINI_RNG_OUT_LIM_THR_L        0x56

#define TFMINI_STREN_THR_LOW_H          0x00
#define TFMINI_STREN_THR_LOW_L          0x58

#define TFMINI_STREN_THR_UPP_H          0x00
#define TFMINI_STREN_THR_UPP_L          0x5A

#define TFMINI_OUT_VAL_SGN_STREN_H      0x00
#define TFMINI_OUT_VAL_SGN_STREN_L      0x5C

#define TFMINI_UNIT_DIST_H              0x00
#define TFMINI_UNIT_DIST_L              0x66

// special parameter register address
#define TFMINI_SLAVE_ADDR_H             0x00
#define TFMINI_SLAVE_ADDR_L             0x26

#define TFMINI_TRIG_MOD_H               0x00
#define TFMINI_TRIG_MOD_L               0x27

#define TFMINI_EXT_COM_TRIG_REG_H       0x01
#define TFMINI_EXT_COM_TRIG_REG_L       0x00

#define TFMINI_RESET_H                  0x00
#define TFMINI_RESET_L                  0x70

enum ranging_gear_mode_t {
    SHORTE = 0x00,
    MIDDLE = 0x03,
    LONGE = 0x07,
};

enum detection_pattern_t {
    AUTO_ = 0x00,
    FIX = 0x01,
};

enum setting_range_limit_t {
    ENABLED = 0x00,
    DISABLED = 0x01,
};

enum unit_distance_t {
    MILLIMETER = 0x00,
    CENTIMETER = 0x01,
};

enum trigger_mode_t {
    INTERNAL = 0x00,
    EXTERNAL = 0x01,
};


class TFMini_i2c
{
public:
    //  constructor
    TFMini_i2c(I2C * i2c, const uint8_t & addr = TFMINI_ADDRESS_DEFAULT);

    // deconstructor
    virtual ~TFMini_i2c();

    operator float();

    // Methods

public:

    void initialize();

    //Reading Data from TFmini-I2C
    void readRegister( const char &reg_h, const char &reg_l, char * bufIn, const uint8_t & len );
    void readMeasurement( uint16_t * dist, uint16_t * stren, uint8_t * mode );

    // General Parameter Configuration
    void setRangingGearMode( detection_pattern_t mode, const ranging_gear_mode_t dist = SHORTE);

    void setRangeLimit( setting_range_limit_t mode, const uint16_t & threshold = 0x2EE0 );

    void setLowerLimitOfSignalStrengthThreshold( uint16_t & threshold );

    void setUpperLimitOfSignalStrengthThreshold( uint16_t & threshold );

    void setOutputValueOfSignalStrengthThresholdAtHighestPoint(  uint16_t & val );

    void setDistanceUnit( unit_distance_t unit );


    // Special Parameter Configuration
    void setSlaveAddress(const uint8_t &addr );

    void setTriggerMode( trigger_mode_t mode );

    void externalCommandTrigger( void );

    void reset( void );


private:

    I2C * i2c_;
    uint8_t addr_ ;

    uint8_t waitTime_ms_;

    bool isInitialized_;

    void setSlaveAddrActive( void );

    void write8( const char &reg_h, const char &reg_l, const uint8_t &val );
    void write16( const char &reg_h, const char &reg_l, const uint16_t &val );

    void read( const char &reg_h, const char &reg_l, char * bufIn,
               const uint8_t &len = 0 );


};

#endif