#include "TFMini_i2c.h"

// constructor
TFMini_i2c::TFMini_i2c(I2C * i2c, const uint8_t& addr):i2c_(i2c), addr_( addr )
{
    waitTime_ms_ = 10;
    isInitialized_ = false;
    if( !isInitialized_ )
        initialize();
}

// deconstructor
TFMini_i2c::~TFMini_i2c() {}

TFMini_i2c::operator float()
{
    uint16_t dist(0), stren(0);
    uint8_t mod(0);
    char distUnit[1] = {0};

    readRegister(TFMINI_UNIT_DIST_H, TFMINI_UNIT_DIST_L, distUnit, 1);
    readMeasurement( &dist, &stren, &mod );

    switch(static_cast<uint8_t>(distUnit[0]) ) {
        case(0x00):
            return(dist/1000.0f);
        case(0x01):
            return(dist/100.0f);
        default:
            return(0xffffffff);
    }
}
// Initialization
// Write here the default configuration registers
//

void TFMini_i2c::initialize()
{
    isInitialized_ = false;

    reset();
    setRangingGearMode(FIX,MIDDLE);
    setDistanceUnit(MILLIMETER);

    isInitialized_ = true;

    return;
}
// Reading Data from TFmini-I2C

void TFMini_i2c::readRegister( const char &reg_h, const char &reg_l, char * bufIn, const uint8_t &len )
{
    read( reg_h, reg_l, bufIn, len );
}

void TFMini_i2c::readMeasurement( uint16_t *dist, uint16_t *stren, uint8_t *mode )
{
    char dataIn[7] = {0};

    read( TFMINI_READ_DATA_REG_H, TFMINI_READ_DATA_REG_L, dataIn, 7 );

    *dist = static_cast<uint16_t>( dataIn[2] | ( dataIn[3] << 8 ) );
    *stren = static_cast<uint16_t>( dataIn[4] | ( dataIn[5] << 8 ) );
    *mode = static_cast<uint8_t>( dataIn[6] );

}


// General Parameter Configuration

void TFMini_i2c::setRangingGearMode( detection_pattern_t mode, const ranging_gear_mode_t dist )
{
    uint16_t regVal( mode << 8 | dist );

    write16( TFMINI_RNG_GEAR_H, TFMINI_RNG_GEAR_L, regVal );
    wait_ms( waitTime_ms_ );
}


void TFMini_i2c::setRangeLimit( setting_range_limit_t mode, const uint16_t & threshold )
{

    write16( TFMINI_SET_RNG_LIM_H, TFMINI_SET_RNG_LIM_L, (uint16_t) mode );
    wait_ms( waitTime_ms_ );
    write16( TFMINI_RNG_OUT_LIM_THR_H, TFMINI_RNG_OUT_LIM_THR_L, threshold );
    wait_ms( waitTime_ms_ );
}


void TFMini_i2c::setLowerLimitOfSignalStrengthThreshold( uint16_t & threshold)
{
    write16(TFMINI_STREN_THR_LOW_H, TFMINI_STREN_THR_LOW_L, threshold);
    wait_ms( waitTime_ms_ );
}


void TFMini_i2c::setUpperLimitOfSignalStrengthThreshold( uint16_t & threshold)
{
    write16(TFMINI_STREN_THR_UPP_H, TFMINI_STREN_THR_UPP_L, threshold);
    wait_ms( waitTime_ms_ );
}


void TFMini_i2c::setOutputValueOfSignalStrengthThresholdAtHighestPoint( uint16_t & val)
{
    write16(TFMINI_OUT_VAL_SGN_STREN_H, TFMINI_OUT_VAL_SGN_STREN_L, val);
    wait_ms( waitTime_ms_ );
}


void TFMini_i2c::setDistanceUnit( unit_distance_t unit )
{
    write8(TFMINI_UNIT_DIST_H, TFMINI_UNIT_DIST_L, unit);
    wait_ms( waitTime_ms_ );
}


// Special Parameter Configuration

////////////////////////////////////////////////////////////////////////////////
//
// Set a new i2c-address between 0x10 - 0x78
// After setting a new address the power of the TFMini must be shut of.
// This function is maintenance. It is not possible to change the address
// in process yet!!!
//
////////////////////////////////////////////////////////////////////////////////

void TFMini_i2c::setSlaveAddress( const uint8_t &addr )
{
    char addrW( addr << 1 | 0 ), len( 0x02 );
    char bufW[] = { addrW, TFMINI_SLAVE_ADDR_H, TFMINI_SLAVE_ADDR_L, len, addr, 0x00 };

    write8( TFMINI_SLAVE_ADDR_H, TFMINI_SLAVE_ADDR_L, addr );
    wait_ms( waitTime_ms_ );
    setSlaveAddrActive();
    wait_ms( waitTime_ms_ );
    addr_ = addr;
}

////////////////////////////////////////////////////////////////////////////////
//
// There are two trigger modes for the measurement of TFmini-I²C. The default
// one is the internaltimer trigger by the timer with one measurement per 10ms.
// The user may modify this mode into the external command trigger mode which
// allows the TFmini-I²C to start the distance measurement by an external
// trigger instruction. Please note that the trigger frequency of TFmini-I²C
// should in no way be more than 100Hz as the maximum.
//
////////////////////////////////////////////////////////////////////////////////

void TFMini_i2c::setTriggerMode( trigger_mode_t mode )
{
    write8( TFMINI_TRIG_MOD_H, TFMINI_TRIG_MOD_L, mode );
    wait_ms( waitTime_ms_ );
}

////////////////////////////////////////////////////////////////////////////////
//
// Use this Method to trigger Measurement. Works only if trigger mode
// is EXTERNAL.
//
////////////////////////////////////////////////////////////////////////////////

void TFMini_i2c::externalCommandTrigger( void )
{
    write8( TFMINI_EXT_COM_TRIG_REG_H, TFMINI_EXT_COM_TRIG_REG_L, 0x01 );
    wait_ms( waitTime_ms_ );
}

////////////////////////////////////////////////////////////////////////////////
//
// Reset of default configuration. By sending such instruction, all adjustable
// configurations will bereset back to the default configurations (excluding
// slave address and trigger mode). Please use it withcaution.
//
////////////////////////////////////////////////////////////////////////////////

void TFMini_i2c::reset( void )
{
    write8( TFMINI_RESET_H, TFMINI_RESET_L, 0x02 );
    wait_ms( waitTime_ms_ );
}




// private
//
void TFMini_i2c::setSlaveAddrActive( void )
{
    char bufW[] = { 0x06 };
    i2c_ -> write( addr_ << 1, bufW, 1, false );
    wait_ms( waitTime_ms_ );
}

////////////////////////////////////////////////////////////////////////////////
//
// Write one byte at given register address. Though it is only possible to write
// two bytes at once, the neighbouring register will be read. The two bytes are
// concatenated. Then two bytes are written.
//
////////////////////////////////////////////////////////////////////////////////

void TFMini_i2c::write8( const char & reg_h, const char & reg_l, const uint8_t & val )
{
    // read the configuration at register address + 1 to not overwrite these settings
    //
    char bufR[2] = { 0 };
    readRegister( reg_h, reg_l, bufR, 2 );
    bufR[0] = val;

    // write the two byte value
    //
    uint16_t val2bytes( bufR[1] << 8 | bufR[0] );

    write16( reg_h, reg_l, val2bytes );
}

////////////////////////////////////////////////////////////////////////////////
//
//  Write 2 bytes at given register address
//
////////////////////////////////////////////////////////////////////////////////

void TFMini_i2c::write16(const char &reg_h, const char &reg_l, const uint16_t &val ) 
{

    char addrW( addr_ << 1 | 0 ), data0( val & 0xff ),
         data1( ( val & ( 0xff << 8 ) ) >> 8 ), len( 0x02 );

    char bufW[] = { addrW, reg_h, reg_l, len, data0, data1 };

    i2c_ -> start( );
    i2c_ -> write( bufW[0] );
    i2c_ -> write( bufW[1] );
    i2c_ -> write( bufW[2] );
    i2c_ -> write( bufW[3] );

    i2c_ -> start( );
    i2c_ -> write( bufW[0] );
    i2c_ -> write( bufW[4] );
    i2c_ -> write( bufW[5] );
    i2c_ -> stop( );
}

////////////////////////////////////////////////////////////////////////////////
//
// Read a define number of bytes (len) at the a given register address. The
// incoming bytes are stored at the address of bufIn.
//
////////////////////////////////////////////////////////////////////////////////

void TFMini_i2c::read( const char & reg_h, const char & reg_l, char * bufIn,
                   const uint8_t & len )
{
    char bufOut[] = { reg_h, reg_l, (char)len };

    i2c_ -> write( addr_ << 1, bufOut, 3, true );
    i2c_ -> read( addr_ << 1, bufIn, len, false );
}