/* Copyright (c) 2012 Nick Ryder, University of Oxford
 * nick.ryder@physics.ox.ac.uk
 *
 *  MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "mbed.h"
#include "RangeFinder.h"
#include "IIR_filter.h"

/*  Notes pmic 21.06.2019:
  - RangeFinder class extended with offset.
  - Static measurements have shown that the actual scaling is approx. 5782 (instead of 5800).
  - Avg. offset relative to top side of  the board (pointing towards meas. direction) is 0.02 m
  - With 7000 mus (7 ms) timout time the max. measured distance is approx. 1.14 m. Theoreticaly
    it is ( 0.007 s * 343.2 m/s )/2 = 1.201 m. If understood rigth the sensor does 8 measurements
    and takes the average. 1.201 - 1.14 m = 0.0612 m := 357 mus -> 357/8 = 44.6 ms / measurement
  - Min. distance is approx. 0.04 m (relative to top side board)
  - Using 50 Hz (Ts = 20 ms) the max. delay is 7 ms (@ max. dist. 1.14 m), there for 13 ms remain for other tasks
    (in real time implementation probably also possible to use 100 Hz)
  - read_m() returns -1 if object is not within certain range
  - sensor opening angle is approx. 15 deg
*/

/*  Notes pmic 24.06.2019:
  - With 17500 mus (17.5 ms = 50/20*7000) timout time the max. measured distance is approx. 2.67 m. Theoreticaly
    it is ( 0.0175 s * 343.2 m/s )/2 = 3.003 m.
  - Using 20 Hz (Ts = 50 ms) the max. delay is 17.5 ms (@ max. dist. 2.67 m), there for 32.5 ms remain for other tasks
*/

// PB_3 ist ist der 4te von unten rechts, aussen GND innen S16

/** Create a RangeFinder object
* @param pin Digital I/O pin the range finder is connected to.
* @param pulsetime Time of pulse to send to the rangefinder to trigger a measurement, in microseconds.
* @param scale Scaling of the range finder's output pulse from microseconds to meters.
* @param offset Offset of the range finder's output pulse to adjust absolut reference.
* @param timeout Time to wait for a pulse from the range finder before giving up.
*        y = x/scale + offset
*/
// RangeFinder rangefinder(PB_3, 10, 5782.0, 0.02, 7000); // 50 Hz parametrization
RangeFinder rangefinder(PB_3, 10, 5782.0, 0.02, 17500); // 20 Hz parametrization
float d = 0.0f;

Serial pc(SERIAL_TX, SERIAL_RX);  // serial connection via USB - programmer
InterruptIn Button(USER_BUTTON);  // User Button
Ticker  LoopTimer;                // interrupt for control loop
Timer t;                          // timer to analyse Button

int   k;
bool  doRun = false;
// float Ts = 0.02f;                 // sample time of main loop, 50 Hz
float Ts = 0.05f;                 // sample time of main loop, 20 Hz

IIR_filter pt1(0.2f, Ts, 1.0f);
float df = 0.0f;

/*
// Sensor 1
float usGain = 0.0001728f;
float usOffset = 0.0215916f;
// Sensor 2
float usGain = 0.0001731f;
float usOffset = 0.0186031f;
*/

// user defined functions
void updateLoop(void);   // loop (via interrupt)
void pressed(void);      // user Button pressed
void released(void);     // user Button released

// main program and control loop
// -----------------------------------------------------------------------------
int main()
{
    pc.baud(2000000);                  // for serial comm.
    LoopTimer.attach(&updateLoop, Ts); // attach loop to timer interrupt
    Button.fall(&pressed);             // attach key pressed function
    Button.rise(&released);            // attach key pressed function
    k = 0;
    pt1.reset(0.0f);
}

// original
/*
if (d == -1.0) {
    printf("Timeout Error.\n");
} else if (d > 5.0) {
    // Seeed's sensor has a maximum range of 4m, it returns
    // something like 7m if the ultrasound pulse isn't reflected.
    printf("No object within detection range.\n");
} else {
    printf("Distance = %f m.\n", d);
}
*/

// the updateLoop starts as soon as you pressed the blue botton
void updateLoop(void)
{
    // d = usGain*rangefinder() + usOffset;
    d = rangefinder();
    df = pt1(d);
    if(doRun) {
        /*
        // use this section to do dynamical measurements
        if(doRun && k++ < 4000) {
            pc.printf("%10i %10.9e\r\n", k, d);
        }
        */
        ///*
        // use this section to do static measurements
        if(doRun && k++%25 == 0) {
            pc.printf("%10i %10.6e\r\n", k, df);
        }
        //*/
    }
}

// buttonhandling
// -----------------------------------------------------------------------------
// start timer as soon as button is pressed
void pressed()
{
    t.start();
}

// evaluating statemachine
void released()
{
    // toggle state over boolean
    if(doRun) {
        k = 0;
        pt1.reset(0.0f);
    }
    doRun = !doRun;
    t.stop();
    t.reset();
}

