#include "NFC.h"

DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut led4(LED4);

//Create an NFC object with the given sda and scl lines
NFC::NFC(PinName sda, PinName scl, PinName NFCIRQ, char NFCAddress):bus(sda, scl), address(NFCAddress), interrupt(NFCIRQ)
{
    // configure board to read RFID tags
    SAMConfig();
}

//Read an NFC tag into an NFCTag object
bool NFC::readNFCTag(NFCTag* tag)
{
    char success;
    char uid[] = { 0, 0, 0, 0, 0, 0, 0 };  // Buffer to store the returned UID
    char uidLength;                        // Length of the UID (4 or 7 bytes depending on ISO14443A card type)
    
    // Wait for an ISO14443A type cards (Mifare, etc.).  When one is found
    // 'uid' will be populated with the UID, and uidLength will indicate
    // if the uid is 4 bytes (Mifare Classic) or 7 bytes (Mifare Ultralight)
    success = readPassiveTargetID(0x00, uid, &uidLength, 100);
    
    
    if (success)
    {
        
        if (uidLength == 4)
        {
            // Now we need to try to authenticate it for read/write access
            // Try with the factory default KeyA: 0xFF 0xFF 0xFF 0xFF 0xFF 0xFF
            char keya[6] = { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };
            
            // Start with block 4 (the first block of sector 1) since sector 0
            // contains the manufacturer data and it's probably better just
            // to leave it alone unless you know what you're doing
            success = mifareclassic_AuthenticateBlock(uid, uidLength, 4, 0, keya);
            
            if (success)
            {
                char data[16];
                
                // If you want to write something to block 4 to test with, uncomment
                // the following line and this text should be read back in a minute
                success = mifareclassic_ReadDataBlock (4, data);
                if(success)
                {
                    memcpy((*tag).itemString, (const char*)data, sizeof data);
                }
            }
        }
        else
        {
            success = 0;
        }
    }
    return success;
}

//Write a tag from an NFCTag object
bool NFC::writeNFCTag(NFCTag* tag)
{
    char success;
    char uid[] = { 0, 0, 0, 0, 0, 0, 0 };  // Buffer to store the returned UID
    char uidLength;                        // Length of the UID (4 or 7 bytes depending on ISO14443A card type)
    
    // Wait for an ISO14443A type cards (Mifare, etc.).  When one is found
    // 'uid' will be populated with the UID, and uidLength will indicate
    // if the uid is 4 bytes (Mifare Classic) or 7 bytes (Mifare Ultralight)
    success = readPassiveTargetID(0x00, uid, &uidLength, 100);
    
    
    if (success)
    {
        
        if (uidLength == 4)
        {
            // Now we need to try to authenticate it for read/write access
            // Try with the factory default KeyA: 0xFF 0xFF 0xFF 0xFF 0xFF 0xFF
            char keya[6] = { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };
            
            // Start with block 4 (the first block of sector 1) since sector 0
            // contains the manufacturer data and it's probably better just
            // to leave it alone unless you know what you're doing
            success = mifareclassic_AuthenticateBlock(uid, uidLength, 4, 0, keya);
            
            if (success)
            {
                char data[16];
                
                // If you want to write something to block 4 to test with, uncomment
                // the following line and this text should be read back in a minute
                memcpy(data, (const char*)(*tag).itemString, sizeof data);
                
                
                // Try to read the contents of block 4
                success = mifareclassic_WriteDataBlock(4, data);
            }
        }
    }
    return success;
}

void NFC::progStatus(char status, int waitMs)
{
    led1 = status & 0x01;
    led2 = (status >> 1) & 0x01;
    led3 = (status >> 2) & 0x01;
    led4 = (status >> 3) & 0x01;
    
    wait_ms(waitMs);
}

//Checks if the PN532 is busy
char NFC::checkBusy(void)
{
    return interrupt;
}

char NFC::readAck(void)
{
    
    
    char rb[8] =       {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
    char pn532ack[8] = {0x01, 0x00, 0x00, 0xFF, 0x00, 0xFF, 0x00, 0x00};
    
    bus.read(address, rb, 8);
    
    char same = 1;
    
    for(int i = 0; i < 7; ++i)
    {
        if(rb[i] != pn532ack[i])
        {
            same = 0;
        }
    }
    
    return same;
}

void NFC::wireReadData(char* buff, char n)
{
    char inBuffer[n+2];
    
    wait_ms(2);//Maybe not needed
    
    bus.read(address, inBuffer, n+2);
    
    for(int i =0; i < n; ++i)
    {
        buff[i] = inBuffer[i+1];
    }
    
}

char NFC::sendCommand(char* cmd, char cmdlen)
{
    
    //command buffer
    char cb[64];
    //checksum, duh
    char checksum;
    
    //If the command is too long for the buffer, return error
    if(cmdlen > (64-8))
    {
        return (unsigned char)-1;
    }
    
    //Initialize the NFC
    cb[0] = 0x00; //PN532_PREAMBLE
    cb[1] = 0x00; //PN532_STARTCODE1
    cb[2] = 0xFF; //PN532_STARTCODE2
    //Add to the checksum
    checksum = cb[0] + cb[1] + cb[2];
    
    
    //Send the length of the command and bitflip of command(both plus one, for some reason, don't ask me)
    cb[3] = (cmdlen+1);
    cb[4] = (char)(~(cmdlen+1) + 1); //May cause some issues without char cast, look out for this
    
    
    //Send the direction of communication
    cb[5] = 0xD4; //PN532_HOSTTOPN532
    //Add to the checksum
    checksum += cb[5];
    
    
    //ACTUAL MESSAGE
    for(char i = 0; i < cmdlen; ++i)
    {
        //Send a byte of the actual command
        cb[6+i] = cmd[i];
        //Add to the checksum
        checksum += cb[6+i];
    }
    //cb[6] = 0x60;
    //cb[7] = 0x01;
    //cb[8] = 0x01;
    //cb[9] = 0x10;
    
    
    
    //End That SHEET
    cb[6+cmdlen] = ~(checksum);
    cb[7+cmdlen] = 0x00;
    bus.write(address, cb, cmdlen+8);
    return 1;
}

char NFC::sendCommandCheckAck (char* cmd, char cmdlen, int timeout)
{
    int timer = 0; //Create the timer we'll add to
    if(sendCommand(cmd, cmdlen) == 1) 
    {
        while(checkBusy()) //While IRQ line is busy
        {
            if(timeout != 0) //We only wait for a timeout if it isn't 0
            {
                timer += 10; //Add the 10 miliseconds that we'll wait
                if(timer > timeout)
                {
                    return 0; //Time ran out
                }
            }
            wait_ms(10); //Wait the 10 miliseconds that we'll wait
        }
        return readAck(); //Read and return acknowledge frame if not busy
    }
    return (unsigned char)-1; //return -1 if sendCommand doesn't return 1 (-1)
}

char NFC::SAMConfig(void)
{
    char outBuffer[4] = {0x14, 0x01, 0x14, 0x01};
    char inBuffer[8];
    // PN532_COMMAND_SAMCONFIGURATION // normal mode // timeout 50ms // use IRQ pin!
    
    if (! sendCommandCheckAck(outBuffer, 4))
    {
         return 0;
    }

    // read data packet
    wireReadData(inBuffer, 8);
  
    return  (inBuffer[6] == 0x15);
}

char NFC::readPassiveTargetID(char cardbaudrate, char * uid, char * uidLength, char timeout)
{
    char outBuffer[] = {0x4A, 0x01, cardbaudrate};
    char inBuffer[20];
    // PN532_COMMAND_INLISTPASSIVETARGET // max 1 cards at once (we can set this to 2 later) //cardbaud rate
    
    if (! sendCommandCheckAck(outBuffer, 3, timeout))
    {
        return 0x0;  // no cards read
    }
    
    // Wait for a card to enter the field
    //char status = 0x00;
    
    char timer = 0;
    while (checkBusy())
    {
        if (timeout != 0)
        {
            timer+=10;
            if (timer > timeout)
            {
                return 0x0;
            }
        }
        wait_ms(10);
    }
    
    wireReadData(inBuffer, 20);
    
    if (inBuffer[7] != 1)
    {
        return 0;
    }
    
    int sens_res = inBuffer[9];
    sens_res <<= 8;
    sens_res |= inBuffer[10];
    
    /* Card appears to be Mifare Classic */
    *uidLength = inBuffer[12];

    for (char i=0; i < inBuffer[12]; i++) 
    {
        uid[i] = inBuffer[13+i];
    }
    
    return 1;
}

char NFC::mifareclassic_AuthenticateBlock (char * uid, char uidLen, int blockNumber, char keyNumber, char * keyData)
{
  //char len;
  char i;
  
  char _uid[7];  // ISO14443A uid
  char _uidLen;  // uid len
  char _key[6];  // Mifare Classic key
  
  // Hang on to the key and uid data
  memcpy (_key, keyData, 6); 
  memcpy (_uid, uid, uidLen); 
  _uidLen = uidLen;  
  
  // Prepare the authentication command //
  pn532_packetbuffer[0] = 0x40;   /* Data Exchange Header */
  pn532_packetbuffer[1] = 1;                              /* Max card numbers */
  pn532_packetbuffer[2] = (keyNumber) ? 0x61 : 0x60;
  pn532_packetbuffer[3] = blockNumber;                    /* Block Number (1K = 0..63, 4K = 0..255 */
  memcpy (pn532_packetbuffer+4, _key, 6);
  for (i = 0; i < _uidLen; i++)
  {
    pn532_packetbuffer[10+i] = _uid[i];                /* 4 byte card ID */
  }

  if (! sendCommandCheckAck(pn532_packetbuffer, 10+_uidLen))
    return 0;


  progStatus(0x02, 500);


  // Read the response packet
  wireReadData(pn532_packetbuffer, 12);
  
  // Check if the response is valid and we are authenticated???
  // for an auth success it should be bytes 5-7: 0xD5 0x41 0x00
  // Mifare auth error is technically byte 7: 0x14 but anything other and 0x00 is not good
  if (pn532_packetbuffer[7] != 0x00)
  {
    return 0;
  }  
  
  return 1;
}

char NFC::mifareclassic_WriteDataBlock (char blockNumber, char * data)
{
  /* Prepare the first command */
  pn532_packetbuffer[0] = 0x40;
  pn532_packetbuffer[1] = 1;                      /* Card number */
  pn532_packetbuffer[2] = 0xA0;       /* Mifare Write command = 0xA0 */
  pn532_packetbuffer[3] = blockNumber;            /* Block Number (0..63 for 1K, 0..255 for 4K) */
  memcpy (pn532_packetbuffer+4, data, 16);          /* Data Payload */

  /* Send the command */
  if (! sendCommandCheckAck(pn532_packetbuffer, 20))
  {
    return 0;
  }  
  wait_ms(10);
  
  /* Read the response packet */
  wireReadData(pn532_packetbuffer, 26);

  return 1;  
}

char NFC::mifareclassic_ReadDataBlock (char blockNumber, char * data)
{
  /* Prepare the command */
  pn532_packetbuffer[0] = 0x40;
  pn532_packetbuffer[1] = 1;                      /* Card number */
  pn532_packetbuffer[2] = 0x30;        /* Mifare Read command = 0x30 */
  pn532_packetbuffer[3] = blockNumber;            /* Block Number (0..63 for 1K, 0..255 for 4K) */

  /* Send the command */
  if (! sendCommandCheckAck(pn532_packetbuffer, 4))
  {
    return 0;
  }
  
  wait_ms(10);
  
  /* Read the response packet */
  wireReadData(pn532_packetbuffer, 26);
  /* If byte 8 isn't 0x00 we probably have an error */
  if (pn532_packetbuffer[7] != 0x00)
  {
    return 0;
  }
    
  /* Copy the 16 data bytes to the output buffer        */
  /* Block content starts at byte 9 of a valid response */
  memcpy (data, pn532_packetbuffer+8, 16);
  return 1;  
}