/**
 * Project:      CDM7160
 * Company:      yosensi.io
 * CodeWriter:   Michal Kozlowski
 * File:         CDM7160.cpp
 * Date:         09-09-2019
 * Description:  CDM7160 Driver Operation File
 */

#include "mbed.h"
#include "CDM7160.h"


CDM7160::CDM7160(PinName sda, PinName scl, uint8_t addr) : pI2C(new I2C(sda, scl)), 
                                                       _I2C(*pI2C)
{
    cdm7160.addr = addr << 1;
    cdm7160.co2 = 0;
}

CDM7160::CDM7160(I2C &oI2C, uint8_t addr) : pI2C(NULL),
                                        _I2C(oI2C) 
{
    cdm7160.addr = (addr << 1);
    cdm7160.co2 = 0;
}

CDM7160::~CDM7160()
{
    if (NULL != pI2C) delete  pI2C;
}

/**
 * ==================== PUBLIC FUNCTIONS ====================
 */

int8_t CDM7160::init(){

    for(uint8_t i=0 ; i<3 ; i++){
        if(control() == CDM7160_SUCCESS)
            return CDM7160_I2C_ADDR;
        wait_ms(20);
    }
    return CDM7160_ERROR;     
}

int8_t CDM7160::initIntegrated( uint16_t hpa, uint16_t height, 
                                eControlMode mode, 
                                eFunctionSetting atm_alt_correct,
                                eFunctionSetting pwm_conce,
                                eFunctionSetting pwm_pin){
    for(uint8_t i=0 ; i<3 ; i++){
        if(control(mode) == CDM7160_SUCCESS){
            if(setSettings(atm_alt_correct, pwm_conce, pwm_pin) == CDM7160_SUCCESS){
                if(setAirPressureAndAltitude(hpa, height) == CDM7160_SUCCESS){
                    return CDM7160_I2C_ADDR;
                }

            }
        }
        wait_ms(20);
    }

    return CDM7160_ERROR;   
}

int8_t CDM7160::softReset(){
    char data[1] = {0x01};
    if(write(RST, data, 1) == CDM7160_SUCCESS){
        wait(1);
        return CDM7160_SUCCESS;
    }
    return CDM7160_ERROR;
}

int8_t CDM7160::control(eControlMode mode){
    char data[1];
    data[0] = mode;
    if(write(CTL, data, 1) == CDM7160_SUCCESS){
        wait_ms(1);
        if(read(data,1) == CDM7160_SUCCESS){
            cdm7160.control = data[0];
            return CDM7160_SUCCESS;
        }
    }
    return CDM7160_ERROR;
}

int CDM7160::getCO2(){
    char data[2];
    uint8_t buff[2];

    if(write(DAL) == CDM7160_SUCCESS){
        if(read(data,2) == CDM7160_SUCCESS){
            for(int i=0; i<2; i++){buff[i] = data[i];}
                cdm7160.co2 = (int16_t)data[1]<<8 | data[0];
                return cdm7160.co2;
        }
    }
    return CDM7160_ERROR;
}

int8_t CDM7160::getStatus(){
    char data[1];

    if(write(ST1) == CDM7160_SUCCESS){
        if(read(data,1) == CDM7160_SUCCESS){
            cdm7160.status = data[0];
            return CDM7160_SUCCESS;
        }
    }

    return CDM7160_ERROR;
}

int8_t CDM7160::getAlarmLimits(){
    char data[2];
    if(write(ALHI) == CDM7160_SUCCESS){
        if(read(data, 2) == CDM7160_SUCCESS){
            cdm7160.highAlertLimit = data[0]*10;
            cdm7160.lowAlertLimit = data[1]*10;
            return CDM7160_SUCCESS;
        }
    }
    return CDM7160_ERROR;
}

int8_t CDM7160::setAlarmLimits(uint16_t high_limit, uint16_t low_limit){
    char data[1];
    data[0] = high_limit/10;
    if(write(ALHI, data, 1) == CDM7160_SUCCESS){
        wait_ms(1);
        data[0] = low_limit/10;
        if(write(ALLO, data, 1) == CDM7160_SUCCESS){
            return CDM7160_SUCCESS;
        }
    }
    return CDM7160_ERROR;
}

int8_t CDM7160::getAirPressureAndAltitude(){
    char data[2];
    if(write(HPA) == CDM7160_SUCCESS){
        if(read(data,1) == CDM7160_SUCCESS){
            cdm7160.atmosferic_pressure = data[0] + 800;
            if(write(HIT) == CDM7160_SUCCESS){
                if(read(data,1) == CDM7160_SUCCESS){
                    cdm7160.altitude = data[0]*10;
                    return CDM7160_SUCCESS;
                }
            }
        }
    }
    return CDM7160_ERROR;
}

int8_t CDM7160::setAirPressureAndAltitude(uint16_t hpa, uint16_t height){
    uint8_t atm, alt;
    atm = hpa - 800;
    alt = height/10;
    char data[1] = {atm};
    if(write(HPA, data, 1) == CDM7160_SUCCESS){
        data[0] = alt;
        if(write(HIT, data, 1) == CDM7160_SUCCESS){
            return CDM7160_SUCCESS;
        }
    }
    return CDM7160_ERROR;
}

int8_t CDM7160::getSettings(){
    char data[1];
    if(write(FUNC) == CDM7160_SUCCESS){
        if(read(data, 1) == CDM7160_SUCCESS){
            cdm7160.settings = data[0];
            return CDM7160_SUCCESS;
        }
    }
    return CDM7160_ERROR;
}

int8_t CDM7160::setSettings(eFunctionSetting atm_alt_correct,
                       eFunctionSetting pwm_conce,
                       eFunctionSetting pwm_pin){
    
    char data[1];
    data[0] = pwm_pin | pwm_conce | atm_alt_correct;

    if(write(FUNC, data, 1) == CDM7160_SUCCESS){
        return CDM7160_SUCCESS;
    }
    return CDM7160_ERROR;
}

int8_t CDM7160::selfDiagnosis(){
    char data[1];
    if(write(ERR) == CDM7160_SUCCESS){
        if(read(data,1) == CDM7160_SUCCESS){
            cdm7160.selfDiag = data[0];
            return CDM7160_SUCCESS;
        }
    }
    return CDM7160_ERROR;
}
 
/**
 * ==================== PRIVATE FUNCTIONS ====================
 */

int8_t CDM7160::read(char data[], uint8_t len) {
    return _I2C.read((cdm7160.addr + DIR_BIT_READ), data, len);
}

int8_t CDM7160::write(uint8_t reg) {
    char regToWrite[1] = {reg};
    return _I2C.write((cdm7160.addr + DIR_BIT_WRITE), regToWrite, 1);
}

int8_t CDM7160::write(uint8_t reg, char data[], uint8_t len) {
    char dataToWrite[len + 1];
    dataToWrite[0] = reg;
    for(uint8_t i=0; i<len; i++)
        dataToWrite[i+1] = data[i];
    return _I2C.write((cdm7160.addr + DIR_BIT_WRITE), dataToWrite, (len + 1));
}
