#pragma once

#include "cantcoap.h"

enum CoapCode;

/** Coap Response class
 * This class contains the response message. It let's you change response information depending on what you need to respond.
 */
class Response : protected CoapPDU
{
    public:
    
    /**  Memory-managed constructor. Buffer for PDU is dynamically sized and allocated by the object.
     * When using this constructor, the CoapPDU class will allocate space for the PDU.
     * Contrast this with the parameterized constructors, which allow the use of an external buffer.
     *
     * Note, the PDU container and space can be reused by issuing a CoapPDU::reset(). If the new PDU exceeds the 
     * space of the previously allocated memory, then further memory will be dynamically allocated.
     *
     * Deleting the object will free the Object container and all dynamically allocated memory.
     *
     * \note It would have been nice to use something like UDP_CORK or MSG_MORE, to allow separate buffers 
     * for token, options, and payload but these FLAGS aren't implemented for UDP in LwIP so stuck with one buffer for now.
     *
     * CoAP version defaults to 1.
     *
     * \sa Response::Response(uint8_t *pdu, int pduLength), Response::Response::(uint8_t *buffer, int bufferLength, int pduLength), 
     * Response:Response()~
     *
     */
    Response();
    
    /** Memory-managed constructor. Buffer for PDU is dynamically sized and allocated by the object.
     * When using this constructor, the CoapPDU class will allocate space for the PDU.
     * Contrast this with the parameterized constructors, which allow the use of an external buffer.
     *
     * Note, the PDU container and space can be reused by issuing a CoapPDU::reset(). If the new PDU exceeds the 
     * space of the previously allocated memory, then further memory will be dynamically allocated.
     *
     * Deleting the object will free the Object container and all dynamically allocated memory.
     *
     * \note It would have been nice to use something like UDP_CORK or MSG_MORE, to allow separate buffers 
     * for token, options, and payload but these FLAGS aren't implemented for UDP in LwIP so stuck with one buffer for now.
     *
     * CoAP version defaults to 1.
     *
     * \sa Response::Response(uint8_t *pdu, int pduLength), Response::Response::(uint8_t *buffer, int bufferLength, int pduLength), 
     * Response:Response()~
     *
     */
    Response(uint8_t *pdu, int pduLength);

    /** Construct a PDU using an external buffer. No copy of the buffer is made.
     * This differs from CoapPDU::CoapPDU(uint8_t *pdu, int pduLength) in that the buffer may be larger
     * than the actual CoAP PDU contained int the buffer. This is typically used when a large buffer is reused
     * multiple times. Note that \b pduLength can be 0.
     * 
     * If an actual CoAP PDU is passed in the buffer, \b pduLength should match its length. CoapPDU::validate() must
     * be called to initiate the object before member functions can be used.
     *
     * A PDU constructed in this manner must be validated with CoapPDU::validate() before the member variables will be accessible.
     *
     * \warning The validation call parses the PDU structure to set some internal parameters. If you do
     * not validate the PDU, then the behaviour of member access functions will be undefined.
     * 
     * The buffer can be reused by issuing a CoapPDU::reset() but the class will not change the size of the buffer. If the
     * newly constructed PDU exceeds the size of the buffer, the function called (for example CoapPDU::addOption) will fail.
     *
     * Deleting this object will only delete the Object container and will not delete the PDU buffer.
     *
     * \param buffer A buffer which either contains a CoAP PDU or is intended to be used to construct one.
     * \param bufferLength The length of the buffer
     * \param pduLength If the buffer contains a CoAP PDU, this specifies the length of the PDU within the buffer.
     *
     * \sa Response::Response(), Response::Response(uint8_t *pdu, int pduLength)
     */
    Response(uint8_t *buffer, int bufferLength, int pduLength);
    
    /** Set the response code
     * @code
     * void get_hello(Request* req, Response* res)
     * {
     *      // get the resource and return it
     *      res->setContent("Hello world\r\n", 13);
     *      res->setCode(CONTENT);
     * }
     * @endcode
     * @code
     * void post_hello(Request* req, Response* res)
     * {
     *      // do something with the post content
     *      res->setCode(CHANGED);
     * }
     * @endcode
     * @code
     * void put_hello(Request* req, Response* res)
     * {
     *      // do add content to your system
     *      res->setCode(CREATED);
     * }
     * @endcode
     *  @param code Coap code. Default value: NOT_IMPLEMENTED (see CoapCode enum for possible values)
     */
    void setCode(int code);
    
    /** Set/change the response type
     *  @param type Coap message type. The value is automatically set depending on the request (Confirmable or not confirmable)
     *  You can change this value if needed (see CoapType enum for possible values)
     */
    void setType(CoapPDU::Type type);
    
    /** Set the payload content to send with the response
     * @code
     * void get_hello(Request* req, Response* res)
     * {
     *      res->setContent("Hello world\r\n", 13);
     * }
     * @endcode
     * @note The response code is automatically set to CONTENT when using setContent()
     *  @param content The content to be send as payload
     *  @param lenght The length of the content
     */
    void setContent(char* content, int length);
    
    protected:
    
};