#pragma once


#include "coap.h"
#include "cantcoap.h"
#include "response.h"
#include "request.h"

#include <vector>

enum CoapCode;

/** Coap Server class
 *  Used to process and respond to coap requests
 *
 * @code
 * #include "mbed.h"
 * #include "coap.h"
 * 
 * void get_demo(Request* req, Response* res)
 * {
 *     // do your stuff here
 * }
 *             
 * int main() {
 *     Server server;
 *     
 *     server.add("/demo", &get_led, GET);
 * 
 *     while(1)
 *     {
 *         server.waitForRequest();  
 *     };
 * }
 * @endcode
 */
class Server{
    
    public:
    /** Create a new Server instance */
    Server();
    
    /** Add a coap resource
     * 
     * @param uri The resource URI to respond on
     * @param function The corresponding function to call when a request for this resource is received
     * @param method The method to respond to (GET, POST, PUT or DELETE)
     *
     */
    void add(char* uri, void (*function)(Request*, Response*), CoapCode method);
    
    /** Wait for a request and process that request 
     * @note This is usually placed inside a loop
     */
    void waitForRequest();
    
    /** Enable broadcast functionality
     * @param broadcast Boolean value to enable or disable broadcasting functionality
     */
    void enableBroadcast(int broadcast = true);
    
    char* getIpAddress();
    
    
    private:
    EthernetInterface eth;
    UDPSocket server;
    Endpoint client;
    
    struct Resource{
        char* uri;
        void (*function)(Request*, Response*);
        int method;   
    };
    
    std::vector<Resource> resources;
    
    int findResource(char* uri, CoapPDU::Code method);
    
    static const int UDP_BUFFER_SIZE = 256;
    static const int TOKEN_BUFFER_SIZE = 8;
    static const int URI_BUFFER_SIZE = 64;
    
};