#pragma once

#include "cantcoap.h"


/** Coap Request class
 * This class contains the coap request. It let's all the information from the request to process a response.
 */
class Request : protected CoapPDU
{
    public:
    /**  Memory-managed constructor. Buffer for PDU is dynamically sized and allocated by the object.
     * When using this constructor, the CoapPDU class will allocate space for the PDU.
     * Contrast this with the parameterized constructors, which allow the use of an external buffer.
     *
     * Note, the PDU container and space can be reused by issuing a CoapPDU::reset(). If the new PDU exceeds the 
     * space of the previously allocated memory, then further memory will be dynamically allocated.
     *
     * Deleting the object will free the Object container and all dynamically allocated memory.
     *
     * \note It would have been nice to use something like UDP_CORK or MSG_MORE, to allow separate buffers 
     * for token, options, and payload but these FLAGS aren't implemented for UDP in LwIP so stuck with one buffer for now.
     *
     * CoAP version defaults to 1.
     *
     * \sa Request::Request(uint8_t *pdu, int pduLength), Request::Request::(uint8_t *buffer, int bufferLength, int pduLength), 
     * Request:Request()~
     *
     */
    Request();
    
    /** Memory-managed constructor. Buffer for PDU is dynamically sized and allocated by the object.
     * When using this constructor, the CoapPDU class will allocate space for the PDU.
     * Contrast this with the parameterized constructors, which allow the use of an external buffer.
     *
     * Note, the PDU container and space can be reused by issuing a CoapPDU::reset(). If the new PDU exceeds the 
     * space of the previously allocated memory, then further memory will be dynamically allocated.
     *
     * Deleting the object will free the Object container and all dynamically allocated memory.
     *
     * \note It would have been nice to use something like UDP_CORK or MSG_MORE, to allow separate buffers 
     * for token, options, and payload but these FLAGS aren't implemented for UDP in LwIP so stuck with one buffer for now.
     *
     * CoAP version defaults to 1.
     *
     * \sa Request::Request(uint8_t *pdu, int pduLength), Request::Request::(uint8_t *buffer, int bufferLength, int pduLength), 
     * Request:Request()~
     *
     */
    Request(uint8_t *pdu, int pduLength);

    /** Construct a PDU using an external buffer. No copy of the buffer is made.
     * This differs from CoapPDU::CoapPDU(uint8_t *pdu, int pduLength) in that the buffer may be larger
     * than the actual CoAP PDU contained int the buffer. This is typically used when a large buffer is reused
     * multiple times. Note that \b pduLength can be 0.
     * 
     * If an actual CoAP PDU is passed in the buffer, \b pduLength should match its length. CoapPDU::validate() must
     * be called to initiate the object before member functions can be used.
     *
     * A PDU constructed in this manner must be validated with CoapPDU::validate() before the member variables will be accessible.
     *
     * \warning The validation call parses the PDU structure to set some internal parameters. If you do
     * not validate the PDU, then the behaviour of member access functions will be undefined.
     * 
     * The buffer can be reused by issuing a CoapPDU::reset() but the class will not change the size of the buffer. If the
     * newly constructed PDU exceeds the size of the buffer, the function called (for example CoapPDU::addOption) will fail.
     *
     * Deleting this object will only delete the Object container and will not delete the PDU buffer.
     *
     * \param buffer A buffer which either contains a CoAP PDU or is intended to be used to construct one.
     * \param bufferLength The length of the buffer
     * \param pduLength If the buffer contains a CoAP PDU, this specifies the length of the PDU within the buffer.
     *
     * \sa Request::Request(), Request::Request(uint8_t *pdu, int pduLength)
     */
    Request(uint8_t *buffer, int bufferLength, int pduLength);
    
    using CoapPDU::Type;
    using CoapPDU::Code;
    
    using CoapPDU::getType;
    using CoapPDU::getCode;
    
    /** Get the payload content send with the request
     *  @return Pointer to the content
     */
    char* getContent();
    
    /** Get the lenght of the content
     *  @code
     *  void get_hello(Request* req, Response* res)
     *  {
     *       if(req->hasContent()){
     *          printf("Content: %s\r\n", req->getContent());
     *       }
     *  }
     *  @endcode
     *  @return integer containing the length of the content
     */
    int getContentLength();
    
    /** Check if the request has content
     *  @return boolean value depending on if the request contains any content
     */
    int hasContent();
    
    /*
    int getMessageId();
    int  getToken();*/

    //using  getContentFormat();
    
    //getOptions();
    
    
};