#include "mbed.h"
#include "xbusparser.h"
#include "mtdata2.h"
#include "xbusmessage.h"

static Serial pc(PA_2, PA_3);
static Serial mt(PB_9, PB_8);
static XbusParser* xbusParser;
static uint8_t rxBuffer[256];

static void* allocateBuffer(size_t bufSize)
{
	return bufSize < sizeof(rxBuffer) ? rxBuffer : NULL;
}

static void mtLowLevelHandler(void)
{
	while (mt.readable())
	{
		XbusParser_parseByte(xbusParser, mt.getc());
	}
}

static void sendCommand(XsMessageId cmdId)
{
	uint8_t buf[8];
	XbusMessage m = {cmdId};
	size_t rawLength = XbusMessage_format(buf, &m);
	for (size_t i = 0; i < rawLength; ++i)
	{
		mt.putc(buf[i]);
	}
}

static void handlePcCommand(char cmd)
{
	switch (cmd)
	{
		case 'c':
			sendCommand(XMID_GotoConfig);
			break;

		case 'm':
			sendCommand(XMID_GotoMeasurement);
			break;
	}
}

static void pcHandler(void)
{
	while (pc.readable())
	{
		handlePcCommand(pc.getc());
	}
}

static void mtDataHandler(struct XbusMessage const* message)
{
	if (message->mid == XMID_MtData2)
	{
		pc.printf("MTData2:");
		uint16_t counter;
		if (MtData2_getItem(&counter, XDI_PacketCounter, message))
		{
			pc.printf(" Packet counter: %5d", counter);
		}
		float ori[4];
		if (MtData2_getItem(ori, XDI_Quaternion, message))
		{
			pc.printf(" Orientation: (% .3f, % .3f, % .3f, % .3f)", ori[0], ori[1],
					ori[2], ori[3]);
		}
		uint32_t status;
		if (MtData2_getItem(&status, XDI_StatusWord, message))
		{
			pc.printf(" Status:%X", status);
		}
		pc.printf("\n");
	}
	else
	{
		pc.printf("Received Xbus message. MID=%X, length=%d\n", message->mid, message->length);
	}
}

static void configureSerialPorts(void)
{
	pc.baud(921600);
	pc.format(8, Serial::None, 2);
	pc.attach(pcHandler, Serial::RxIrq);

	mt.baud(921600);
	mt.format(8, Serial::None, 2);
	mt.attach(mtLowLevelHandler, Serial::RxIrq);
}

int main(void)
{
	XbusParserCallback xbusCallback = {};
	xbusCallback.allocateBuffer = allocateBuffer;
	xbusCallback.handleMessage = mtDataHandler;

	xbusParser = XbusParser_create(&xbusCallback);
	configureSerialPorts();

	for (;;)
	{
		sleep();
	}
}
