/*!
 * \file
 * \copyright
 * Copyright (C) Xsens Technologies B.V., 2015.  All rights reserved.
 *
 * This source code is intended for use only by Xsens Technologies BV and
 * those that have explicit written permission to use it from
 * Xsens Technologies BV.
 *
 * THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
 * PARTICULAR PURPOSE.
 */

#ifndef __XBUSPARSER_H
#define __XBUSPARSER_H

#include <stddef.h>
#include <stdint.h>
#include "xbusmessage.h"

#ifdef __cplusplus
extern "C" {
#endif

struct XbusParser;

/*!
 * \brief Callback function structure for use with the XbusParser.
 */
struct XbusParserCallback
{
	/*!
	 * \brief Allocate a buffer for message reception.
	 * \param bufSize The size of the buffer to allocate.
	 * \returns Pointer to buffer to use for message reception, or NULL if
	 * a buffer cannot be allocated.
	 *
	 * \note It is the resposibility of the user to deallocate the message
	 * data buffers pointed to by XbusMessage structures passed to the
	 * handleMessage() callback function.
	 */
	void* (*allocateBuffer)(size_t bufSize);

	/*!
	 * \brief Deallocate a buffer that was previously allocated by a call to
	 * allocateBuffer.
	 */
	void (*deallocateBuffer)(void const* buffer);

	/*!
	 * \brief Handle a received message.
	 *
	 * \note If the passed XbusMessage structure has a non-null data pointer
	 * then it is the responsibility of the user to free this once handling
	 * of the message is complete.
	 */
	void (*handleMessage)(struct XbusMessage const* message);
};

size_t XbusParser_mem(void);
struct XbusParser* XbusParser_create(struct XbusParserCallback const* callback);
void XbusParser_destroy(struct XbusParser* parser);
struct XbusParser* XbusParser_init(void* parserMem, struct XbusParserCallback const* callback);

void XbusParser_parseByte(struct XbusParser* parser, uint8_t byte);
void XbusParser_parseBuffer(struct XbusParser* parser, uint8_t const* buf, size_t bufSize);

#ifdef __cplusplus
}
#endif // extern "C"

#endif // __XBUSPARSER_H
