/*!
 * \file
 * \copyright
 * Copyright (C) Xsens Technologies B.V., 2015.  All rights reserved.
 *
 * This source code is intended for use only by Xsens Technologies BV and
 * those that have explicit written permission to use it from
 * Xsens Technologies BV.
 *
 * THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
 * PARTICULAR PURPOSE.
 */

#ifndef __XBUSMESSAGE_H
#define __XBUSMESSAGE_H

#include <stddef.h>
#include <stdint.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C" {
#endif

/*! \brief Xbus message IDs. */
enum XsMessageId
{
	XMID_Wakeup             = 0x3E,
	XMID_WakeupAck          = 0x3F,
	XMID_ReqDid             = 0x00,
	XMID_DeviceId           = 0x01,
	XMID_GotoConfig         = 0x30,
	XMID_GotoConfigAck      = 0x31,
	XMID_GotoMeasurement    = 0x10,
	XMID_GotoMeasurementAck = 0x11,
	XMID_MtData2            = 0x36,
	XMID_ReqOutputConfig    = 0xC0,
	XMID_SetOutputConfig    = 0xC0,
	XMID_OutputConfig       = 0xC1,
	XMID_Reset              = 0x40,
	XMID_ResetAck           = 0x41,
	XMID_Error              = 0x42,
};

/*! \brief Xbus data message type IDs. */
enum XsDataIdentifier
{
	XDI_PacketCounter  = 0x1020,
	XDI_SampleTimeFine = 0x1060,
	XDI_Quaternion     = 0x2010,
	XDI_DeltaV         = 0x4010,
	XDI_Acceleration   = 0x4020,
	XDI_RateOfTurn     = 0x8020,
	XDI_DeltaQ         = 0x8030,
	XDI_MagneticField  = 0xC020,
	XDI_StatusWord     = 0xE020,
};

/*!
 * \brief An Xbus message structure with optional payload.
 */
struct XbusMessage
{
	/*! \brief The message ID of the message. */
	enum XsMessageId mid;
	/*!
	 * \brief The length of the payload.
	 *
	 * \note The meaning of the length is message dependent. For example,
	 * for XMID_OutputConfig messages it is the number of OutputConfiguration
	 * elements in the configuration array.
	 */
	uint16_t length;
	/*! \brief Pointer to the payload data. */
	void* data;
};

/*!
 * \brief Output configuration structure.
 */
struct OutputConfiguration
{
	/*! \brief Data type of the output. */
	enum XsDataIdentifier dtype;
	/*!
	 * \brief The output frequency in Hz, or 65535 if the value should be
	 * included in every data message.
	 */
	uint16_t freq;
};

size_t XbusMessage_format(uint8_t* raw, struct XbusMessage const* message);
bool XbusMessage_getDataItem(void* item, enum XsDataIdentifier id, struct XbusMessage const* message);
char const* XbusMessage_dataDescription(enum XsDataIdentifier id);

#ifdef __cplusplus
}
#endif // extern "C"

#endif // __XBUSMESSAGE_H
