#include "mbed.h"
#include "rtos.h"
#include "xbusparser.h"
#include "xbusmessage.h"

#define MEMORY_POOL_SIZE (4)
#define RESPONSE_QUEUE_SIZE (1)
#define MAX_XBUS_DATA_SIZE (128)

static Serial pc(PA_2, PA_3);
static Serial mt(PB_9, PB_8);
static XbusParser* xbusParser;

MemoryPool<XbusMessage, MEMORY_POOL_SIZE> g_messagePool;
MemoryPool<uint8_t[MAX_XBUS_DATA_SIZE], MEMORY_POOL_SIZE> g_messageDataPool;
Queue<XbusMessage, RESPONSE_QUEUE_SIZE> g_responseQueue;

static void* allocateMessageData(size_t bufSize)
{
	return bufSize < MAX_XBUS_DATA_SIZE ? g_messageDataPool.alloc() : NULL;
}

static void deallocateMessageData(void const* buffer)
{
	g_messageDataPool.free((uint8_t(*)[MAX_XBUS_DATA_SIZE])buffer);
}

static void mtLowLevelHandler(void)
{
	while (mt.readable())
	{
		XbusParser_parseByte(xbusParser, mt.getc());
	}
}

XbusMessage const* doTransaction(XbusMessage* m)
{
	uint8_t buf[64];
	size_t rawLength = XbusMessage_format(buf, m);
	for (size_t i = 0; i < rawLength; ++i)
	{
		mt.putc(buf[i]);
	}

	osEvent ev = g_responseQueue.get(500);
	return ev.status == osEventMessage ? (XbusMessage*)ev.value.p : NULL;
}

static void freeMessage(XbusMessage const* m)
{
	if (m->data)
		deallocateMessageData(m->data);
	g_messagePool.free((XbusMessage*)m);
}

static void dumpResponse(XbusMessage const* response)
{
	switch (response->mid)
	{
		case XMID_GotoConfigAck:
			pc.printf("Device went to config mode\n");
			break;

		case XMID_DeviceId:
			pc.printf("Device ID: %08X\n", *(uint32_t*)response->data);
			break;

		case XMID_OutputConfig:
			{
				pc.printf("Output configuration\n");
				OutputConfiguration* conf = (OutputConfiguration*)response->data;
				for (int i = 0; i < response->length; ++i)
				{
					pc.printf("\t%s: %d Hz\n", XbusMessage_dataDescription(conf->dtype), conf->freq);
					++conf;
				}
			}
			break;

		case XMID_Error:
			pc.printf("Device error!");
			break;

		default:
			pc.printf("Received response MID=%X, length=%d\n", response->mid, response->length);
			break;
	}
}

static void sendCommand(XsMessageId cmdId)
{
	XbusMessage m = {cmdId};
	XbusMessage const* response = doTransaction(&m);

	if (response)
	{
		dumpResponse(response);
		freeMessage(response);
	}
	else
	{
		pc.printf("Timeout waiting for response.\n");
	}
}

static void handlePcCommand(char cmd)
{
	switch (cmd)
	{
		case 'c':
			sendCommand(XMID_GotoConfig);
			break;

		case 'm':
			sendCommand(XMID_GotoMeasurement);
			break;

		case 'd':
			sendCommand(XMID_ReqDid);
			break;

		case 'o':
			sendCommand(XMID_ReqOutputConfig);
			break;
	}
}

static void handleDataMessage(struct XbusMessage const* message)
{
	pc.printf("MTData2:");
	uint16_t counter;
	if (XbusMessage_getDataItem(&counter, XDI_PacketCounter, message))
	{
		pc.printf(" Packet counter: %5d", counter);
	}
	float ori[4];
	if (XbusMessage_getDataItem(ori, XDI_Quaternion, message))
	{
		pc.printf(" Orientation: (% .3f, % .3f, % .3f, % .3f)", ori[0], ori[1],
				ori[2], ori[3]);
	}
	uint32_t status;
	if (XbusMessage_getDataItem(&status, XDI_StatusWord, message))
	{
		pc.printf(" Status:%X", status);
	}
	pc.printf("\n");
	deallocateMessageData(message->data);
}

static void mtMessageHandler(struct XbusMessage const* message)
{
	if (message->mid == XMID_MtData2)
	{
		handleDataMessage(message);
	}
	else
	{
		XbusMessage* m = g_messagePool.alloc();
		memcpy(m, message, sizeof(XbusMessage));
		g_responseQueue.put(m);
	}
}

static void configureSerialPorts(void)
{
	pc.baud(921600);
	pc.format(8, Serial::None, 2);

	mt.baud(921600);
	mt.format(8, Serial::None, 2);
	mt.attach(mtLowLevelHandler, Serial::RxIrq);
}

static uint32_t readDeviceId(void)
{
	XbusMessage reqDid = {XMID_ReqDid};
	XbusMessage const* didRsp = doTransaction(&reqDid);
	uint32_t deviceId = 0;
	if (didRsp)
	{
		if (didRsp->mid == XMID_DeviceId)
		{
			deviceId = *(uint32_t*)didRsp->data;
		}
		freeMessage(didRsp);
	}
	return deviceId;
}

static bool configureMotionTracker(void)
{
	uint32_t deviceId = readDeviceId();
	uint8_t deviceType = (deviceId >> 24) & 0x0F;

	if (deviceId)
	{
		pc.printf("Found MTi-%d\n", deviceType);

		OutputConfiguration conf[5] = {
			{XDI_PacketCounter, 65535},
			{XDI_SampleTimeFine, 65535}
		};
		if (deviceType == 1)
		{
			conf[2].dtype = XDI_Acceleration;
			conf[2].freq = 100;
			conf[3].dtype = XDI_RateOfTurn;
			conf[3].freq = 100;
			conf[4].dtype = XDI_MagneticField;
			conf[4].freq = 100;
		}
		else
		{
			conf[2].dtype = XDI_Quaternion;
			conf[2].freq = 100;
			conf[3].dtype = XDI_StatusWord;
			conf[3].freq = 65535;
		}

		XbusMessage outputConfMsg = {XMID_SetOutputConfig, 5, &conf};
		XbusMessage const* outputConfRsp = doTransaction(&outputConfMsg);
		if (outputConfRsp)
		{
			if (outputConfRsp->mid == XMID_OutputConfig)
			{
				pc.printf("Output configuration set to:\n");
				OutputConfiguration* conf = (OutputConfiguration*)outputConfRsp->data;
				for (int i = 0; i < outputConfRsp->length; ++i)
				{
					pc.printf("\t%s: %d Hz\n", XbusMessage_dataDescription(conf->dtype), conf->freq);
					++conf;
				}
			}
			else
			{
				dumpResponse(outputConfRsp);
			}
			freeMessage(outputConfRsp);
		}
		else
		{
			pc.printf("Failed to set output configuration.\n");
		}

		return true;
	}
	else
	{
		return false;
	}
}

int main(void)
{
	XbusParserCallback xbusCallback = {};
	xbusCallback.allocateBuffer = allocateMessageData;
	xbusCallback.deallocateBuffer = deallocateMessageData;
	xbusCallback.handleMessage = mtMessageHandler;

	xbusParser = XbusParser_create(&xbusCallback);
	configureSerialPorts();
	if (configureMotionTracker())
	{
		for (;;)
		{
			while (pc.readable())
			{
				handlePcCommand(pc.getc());
			}
		}
	}
	else
	{
		pc.printf("Failed to configure motion tracker.\n");
		return -1;
	}
}
