#include "mbed.h"
#include "rtos.h"
#include "xbusparser.h"
#include "xbusmessage.h"

#define MEMORY_POOL_SIZE (4)
#define MAX_XBUS_DATA_SIZE (128)

static Serial pc(PA_2, PA_3);
static Serial mt(PB_9, PB_8);
static XbusParser* xbusParser;

MemoryPool<XbusMessage, MEMORY_POOL_SIZE> g_messagePool;
MemoryPool<uint8_t[MAX_XBUS_DATA_SIZE], MEMORY_POOL_SIZE> g_messageDataPool;

static void* allocateMessageData(size_t bufSize)
{
	return bufSize < MAX_XBUS_DATA_SIZE ? g_messageDataPool.alloc() : NULL;
}

static void deallocateMessageData(void const* buffer)
{
	g_messageDataPool.free((uint8_t(*)[MAX_XBUS_DATA_SIZE])buffer);
}

static void mtLowLevelHandler(void)
{
	while (mt.readable())
	{
		XbusParser_parseByte(xbusParser, mt.getc());
	}
}

static void sendCommand(XsMessageId cmdId)
{
	uint8_t buf[8];
	XbusMessage m = {cmdId};
	size_t rawLength = XbusMessage_format(buf, &m);
	for (size_t i = 0; i < rawLength; ++i)
	{
		mt.putc(buf[i]);
	}
}

static void handlePcCommand(char cmd)
{
	switch (cmd)
	{
		case 'c':
			sendCommand(XMID_GotoConfig);
			break;

		case 'm':
			sendCommand(XMID_GotoMeasurement);
			break;

		case 'd':
			sendCommand(XMID_ReqDid);
			break;

		case 'o':
			sendCommand(XMID_ReqOutputConfig);
			break;
	}
}

static void pcHandler(void)
{
	while (pc.readable())
	{
		handlePcCommand(pc.getc());
	}
}

static void handleDataMessage(struct XbusMessage const* message)
{
	pc.printf("MTData2:");
	uint16_t counter;
	if (XbusMessage_getDataItem(&counter, XDI_PacketCounter, message))
	{
		pc.printf(" Packet counter: %5d", counter);
	}
	float ori[4];
	if (XbusMessage_getDataItem(ori, XDI_Quaternion, message))
	{
		pc.printf(" Orientation: (% .3f, % .3f, % .3f, % .3f)", ori[0], ori[1],
				ori[2], ori[3]);
	}
	uint32_t status;
	if (XbusMessage_getDataItem(&status, XDI_StatusWord, message))
	{
		pc.printf(" Status:%X", status);
	}
	pc.printf("\n");
}

static void mtMessageHandler(struct XbusMessage const* message)
{
	if (message->mid == XMID_MtData2)
	{
		handleDataMessage(message);
	}
	else if (message->mid == XMID_DeviceId)
	{
		pc.printf("Device ID: %8X\n", *(uint32_t*)message->data);
	}
	else if (message->mid == XMID_OutputConfig)
	{
		pc.printf("Output configuration:\n");
		struct OutputConfiguration* conf = (struct OutputConfiguration*)message->data;
		for (int i = 0; i < message->length; ++i)
		{
			pc.printf("\t%s: %d Hz\n", XbusMessage_dataDescription(conf->dtype), conf->freq);
			++conf;
		}
	}
	else
	{
		pc.printf("Received Xbus message. MID=%X, length=%d\n", message->mid, message->length);
	}
	if (message->data)
	{
		deallocateMessageData(message->data);
	}
}

static void configureSerialPorts(void)
{
	pc.baud(921600);
	pc.format(8, Serial::None, 2);
	pc.attach(pcHandler, Serial::RxIrq);

	mt.baud(921600);
	mt.format(8, Serial::None, 2);
	mt.attach(mtLowLevelHandler, Serial::RxIrq);
}

int main(void)
{
	XbusParserCallback xbusCallback = {};
	xbusCallback.allocateBuffer = allocateMessageData;
	xbusCallback.deallocateBuffer = deallocateMessageData;
	xbusCallback.handleMessage = mtMessageHandler;

	xbusParser = XbusParser_create(&xbusCallback);
	configureSerialPorts();

	for (;;)
	{
		sleep();
	}
}
